/*
 * This module performs the main sequence of handling a newly established
 * SSH session with a remote client.
 *
 * Sequence:
 *    Authenticate remote user.  (AUTH_USER, AUTH_RSA, AUTH_PASSWORD)
 *
 *    Setup execution options 	 (REQUEST_PTY, etc).
 *
 *    Execute command.
 *
 *
 * Author:	David Jones
 * Date:	18-MAY-1998
 * Revised:	25-MAY-1998	Update for completion ports.
 * Revised:	3-JUN-1998	Check for valid PTY stream prior to close.
 * Revised:	27-JUN-1998	Get DECnet task spec from parmeters.h
 * Revised:	1-JUL-1998	Check for valid message scans.
 * Revised:	22-JUL-1998	Use proper arguments for initiator call.
 * Revised:	5-SEP-1998	Prototype /ADMIN support.
 * Revised:	9-SEP-1998	Disallow PTYs if /ADMIN present.
 * Revised:	13-SEP-1998	Port forwarding support.
 * Revised:	26-SEP-1998	Add terminal type name to cportpty_open() call.
 * Revised:	21-OCT-1998	Support MAX_PACKET message and pty_mode.
 * Revised:	27-OCT-1998	Support client port forwarding.
 * Revised:	28-OCT-1998	Support /noprocess.
 * Revised:	1-NOV-1998	Supoprt port request forwarding.
 * Revised:	26-NOV-1998	Enhance X11 support.
 * Revised:	16-MAR-1999	Fix bug in tt_out rundown handling, stop
 *				reporting errors on tt_out.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "parameters.h"
#include "cport_sshsess.h"		/* SSH session managment */
#include "cport_sshlogin.h"
#include "cport_pipe.h"
#include "cport_admin.h"
#include "cport_noprocess.h"
#include "tutil.h"
#include "tmemory.h"
#include "event_report.h"
#include "port_forwarder.h"

#define PEND_TO_CLIENT 1
#define PEND_FROM_CLIENT 2
#define PEND_TO_PROC 4
#define PEND_FROM_PROC 8

struct execution_state {
    int mode;			/* 0-setup, 1- CMD, 2-SHELL */
    int auth_type;		/* 2-RSA 3-password */
    struct user_account_info uai;  /* Data from SYSUAF record + pwd */
    void *ext_auth_info;

    void *tt_ctx;		/* pseudo-terminal context */
    cport_isb tt_in;
    cport_isb tt_out;

    void *port_fwd;		/* pipe to port forwarder (X11) thread */
    cport_isb pf_in;		/* messages received from forwarder thread */
    cport_isb pf_out;		/* messages sent to thread */
    char  *pf_buffer;		/* Buffer for pf_in */
    int *pf_mask;		/* flags messages to be handled by port fwd */
    int pf_bufsize;
    int pf_mask_size;
    int pf_x11_server;		/* server number */

    char *command;
    char fixed_command[64];	/* buffer for small commands */
    char *tty_type;		/* e.g. vt100 */
    int tty_geometry[4];	/* rows, cols, x, y */
    int tty_modes_len;
    char *tty_modes;
};

static int get_exec_parameters ( sshsess_session, 
	struct execution_state *, char * );
static int do_interactive ( sshsess_session, struct execution_state * );
int ssh_disconnect_session ( sshsess_session, 
	struct execution_state *, char *);
cport_stream_handler cportpty_driver;		/* psuedo-terminal */
cport_stream_handler cportcmd_driver;		/* DECnet (EXECCMD) command */
/***************************************************************************/
/* Use data in session control block to make a display string for the
 * remote connection.  Format is:  'host: nnn.nnn.nnn.nnn port: nnnn'
 */
static void generate_accpornam ( sshsess_session s, char *pornam, int max_size )
{
    int i;
    i = 0;
    tu_strcpy ( pornam, "Host: " ); i = 6;
    tu_strint ( s->remote_host[0], &pornam[i] ); i += tu_strlen(&pornam[i]);
    pornam[i++] = '.';
    tu_strint ( s->remote_host[1], &pornam[i] ); i += tu_strlen(&pornam[i]);
    pornam[i++] = '.';
    tu_strint ( s->remote_host[2], &pornam[i] ); i += tu_strlen(&pornam[i]);
    pornam[i++] = '.';
    tu_strint ( s->remote_host[3], &pornam[i] ); i += tu_strlen(&pornam[i]);

    tu_strcpy ( &pornam[i], " Port: " ); i += 7;
    tu_strint ( s->remote_port, &pornam[i] );

}
/****************************************************************************/
int ssh_do_session ( sshsess_session session, char errmsg[256] )
{
    int status, failure_fmt, length;
    struct execution_state exec;
    char *x11_info;
    /*
     * Initialize the execution context.
     */
    exec.mode = 0;			/* prep. phase. */
    exec.ext_auth_info = exec.uai.password;
    exec.uai.password[0] ='\0';
    exec.command = exec.fixed_command;
    exec.fixed_command[0] = '\0';
    exec.tty_type = (char *) 0;
    exec.tty_modes_len = 0;
    exec.tt_in = exec.tt_out = (cport_isb) 0;
    exec.port_fwd = (void *) 0;
    exec.pf_in = exec.pf_out = (cport_isb) 0;
    exec.pf_mask = (int *) 0;
    exec.pf_mask_size = 0;
    exec.pf_x11_server = 0;
    session->exec_env = (void *) &exec;
    /*
     * Establish credentials of user.
     */
    status = sshlgn_authenticate ( session, &exec.uai,
		&exec.ext_auth_info, errmsg );
    if ( (status&1) == 0 ) {
	evr_event (EVR_LOGFAIL,"Authentication failure at %t (%s)", 0, errmsg);
	return status;
    }
    session->exec_state = 1;		/* disconnects allowed */
    /*
     * Get command to execute and set parameters for execution.
     */
    status = get_exec_parameters ( session, &exec, errmsg );
    if ( (status&1) == 0 ) {
	evr_event (EVR_LOGFAIL,"Operation setup negotiation failure at %t", 0);
	ssh_disconnect_session ( session, &exec, errmsg );
	return status;
    }
    /*
     * If port fowarding active, send EXEC_CMD message to signal start
     * of interactive phase.  Forwarder responds with mask of forwarder-related
     * mesage types.  X11 server number is message type in iosb[3] (send type).
     */
    if ( exec.port_fwd && exec.pf_bufsize > 0 ) {
	int msize;
	cport_pipe_set_send_type ( exec.pf_out, SSH_CMSG_EXEC_CMD );
	status = cport_do_io ( exec.pf_out, CPORT_WRITE, "CMD", 3, &length );
	status = cport_do_io ( exec.pf_in, CPORT_READ, 
		exec.pf_buffer, exec.pf_bufsize, &msize );
	exec.pf_mask_size = 0;

        if ( (status&1) == 0 || msize <= 0 ) {
	    exec.pf_mask_size = 0;
	} else {
	    exec.pf_mask_size = (msize+sizeof(int)-1)/(sizeof(int));
	    msize = exec.pf_mask_size * sizeof(int);
	    exec.pf_mask = tm_malloc ( msize );
	    if ( exec.pf_mask ) memcpy ( exec.pf_mask, exec.pf_buffer, msize );
	    else exec.pf_mask_size = 0;

	    exec.pf_x11_server = exec.pf_in->default_iosb[3];
	}
    }
    if ( exec.port_fwd && (exec.pf_x11_server > 0) ) {
	/*
	 * Make info string used to pass on x11 server number to client
	 * process:  transport:server-number.
	 */
	x11_info = tm_malloc ( 32 );
	if ( x11_info ) {
	    int offset;
	    if ( param.x11_decnet_node->value[0] ) {
		tu_strcpy ( x11_info, "DECNET:" );  offset = 7;
	    } else {
	        tu_strcpy ( x11_info, "TCPIP:" ); offset = 6;
	    }
	    tu_strint ( exec.pf_x11_server, &x11_info[offset] );
	}
    } else x11_info = (char *) 0;
    evr_event ( EVR_LOGIN, 
	"User %s login, authtype: %d, mode %d, x11: %s, at %t", 
		session->username,session->auth_type,  exec.mode,
		x11_info ? x11_info : "-", 0 );
    /*
     * Create user process with connected I/O streams.
     */
    if ( exec.mode == 1 ) {
	/*
	 * exec_cmd, Use cmd driver.
	 */
	int execcmd_open();
	status = execcmd_open ( &exec.tt_ctx,  
		param.cmd_task, &exec.uai, x11_info, errmsg );

	if ( (status&1) == 0 ) {
	    /*
	     * Disconnect.
	     */
	    ssh_disconnect_session ( session, &exec,
		"Command process creation failed" );
	    return status;
	}
	/*
	 * Assign streams to opened DECnet/mailbox channel.
	 */
	exec.tt_in = cport_assign_stream ( session->cport, &cportcmd_driver,
		exec.tt_ctx, 1 );
	exec.tt_out = cport_assign_stream ( session->cport, &cportcmd_driver,
		exec.tt_ctx, 0 );
    } else if ( exec.mode == 2 ) {
	/*
	 * shell, create a PTY.
	 */
	int execpty_open();
	char accpornam[64];

	generate_accpornam ( session, accpornam, sizeof(accpornam) );

	status = execpty_open ( &exec.tt_ctx, accpornam, exec.tty_geometry,
		exec.tty_type, exec.tty_modes_len, exec.tty_modes, &exec.uai, 
		x11_info, errmsg );
	if ( (status&1) == 0 ) {
	    /*
	     * Disconnect.
	     */
	    evr_event ( EVR_LOGFAIL, "PTY creation failed: %d (%s)", 
		status, errmsg );
	    ssh_disconnect_session ( session, &exec,
		"Shell process creation failed" );
	    return status;
	}
	/*
	 * Assign streams.
	 */
	exec.tt_in = cport_assign_stream ( session->cport, &cportpty_driver,
		exec.tt_ctx, 1 );
	exec.tt_out = cport_assign_stream ( session->cport, &cportpty_driver,
		exec.tt_ctx, 0 );

    } else if ( exec.mode == 3 ) {
	/*
	 * Administator interface (command mode only), use pipe between
	 * this thread and ssh monitor thread.
	 */
	status = execadmin_open ( &exec.tt_ctx, session, &exec.uai, errmsg );

	if ( (status&1) == 0 ) {
	    /*
	     * Disconnect.
	     */
	    ssh_disconnect_session ( session, &exec, errmsg );
	    return status;
	}
	/*
	 * Assign streams to opened administrator context.
	 */
	exec.tt_in = cport_assign_stream ( session->cport, &cportadmin_driver,
		exec.tt_ctx, 1 );
	exec.tt_out = cport_assign_stream ( session->cport, &cportadmin_driver,
		exec.tt_ctx, 0 );

    } else if ( exec.mode == 4 ) {
	/*
	 * /noprocess interface (null interface).
	 */
	status = execnoproc_open ( &exec.tt_ctx, session, &exec.uai, errmsg );

	if ( (status&1) == 0 ) {
	    /*
	     * Disconnect.
	     */
	    ssh_disconnect_session ( session, &exec, errmsg );
	    return status;
	}
	/*
	 * Assign streams to opened administrator context.
	 */
	exec.tt_in = cport_assign_stream ( session->cport, &cportnoproc_driver,
		exec.tt_ctx, 1 );
	exec.tt_out = cport_assign_stream ( session->cport, &cportnoproc_driver,
		exec.tt_ctx, 0 );

    } else {
	/*
	 * Mode == 0, log the failure and reject connection.
	 */
	evr_event ( EVR_LOGFAIL, "Mode negotiate failed (%d) after login, %t", 
		exec.mode, 0 );
        ssh_disconnect_session ( session, &exec,
		"Mode negotiation failure" );
	return 20;
    }
    /*
     * Enter interactive mode, relaying I/O between remote client and process.
     */
    status = do_interactive ( session, &exec );
    /*
     * rundown execution process.
     */
    if ( exec.tt_in ) cport_deassign ( exec.tt_in );
    if ( exec.tt_out ) cport_deassign ( exec.tt_out );
    if ( exec.port_fwd ) {
        /* if ( exec.pf_out ) cport_deassign ( exec.pf_out );
	if ( exec.pf_in ) cport_deassign ( exec.pf_in ); */
    }
    return status;
}
/*************************************************************************/
/* Cleanup shutdown async. I/O's currently in progress.
 */
static int cancel_pending ( 
	sshsess_session session, struct execution_state *exec )
{
    int status;
    /*
     * kill pending I/O's
     */
    cport_cancel (exec->tt_in); 
    status = cport_cancel ( session->in_msg );
    /*
     * wait for completion of killed I/Os.
     */
    return status;
}
/*************************************************************************/
/*  Take steps necessary to place session into in 'disconnected' state (2).
 * Attempt to send SSH_MSG_DISCONNECT to client (with reason) and receive 
 * response.
 *
 * The local message context is rundown.
 */
int ssh_disconnect_session ( sshsess_session session, 
	struct execution_state *exec, char *reason ) {
    sshmsg_local msg;
    int status, xcount;

    if ( session->exec_state != 1 ) return 1;

    if ( exec->tt_in ) cport_cancel ( exec->tt_in );
    cport_cancel ( session->in_msg );
    /*
     * wait for I/O to complete.
     */
    sshmsg_rundown_locals ( session->locus );

    sshmsg_init_locals ( session->locus, 1, &msg );
    sshmsg_format_message ( &msg, SSH_MSG_DISCONNECT, tu_strlen(reason),
	reason );
    /* sshpad_set_time_limit ( session->pad, 60 ); */
    status = cport_do_io ( session->out_msg, CPORT_WRITE, &msg, 1, &xcount );
    sshmsg_rundown_locals ( session->locus );
    session->exec_state = 2;
    if ( (status&1) == 0 ) return status;
    return status;
}

/*************************************************************************/
/*
 * Create pipe streams and start port forwarder thread.
 */
static int init_forwarder ( 
	sshsess_session session, struct execution_state *exec )
{
    int status;
    char accpornam[64];

    status = cport_pipe_open (session->cport, &exec->port_fwd);
    if ( (status&1) == 1 ) {
	exec->pf_out = cport_assign_stream ( session->cport, 
		&cportpipe_driver, exec->port_fwd, 0 );
	exec->pf_in = cport_assign_stream ( session->cport,
	    	&cportpipe_driver, exec->port_fwd, 1 );
	generate_accpornam ( session, accpornam, sizeof(accpornam) );

	status = pf_create_forwarder ( session->protocol_flags, 
			    exec->port_fwd, 
			    session->ext_username, accpornam,
			    &exec->pf_buffer, &exec->pf_bufsize );
    }
    return status;
}
/*************************************************************************/
/*
 */
static int get_exec_parameters ( sshsess_session session,
	struct execution_state *exec, char *errmsg )
{
    int status, failure_fmt, length, real_length, xcount, i, is_admin;
    int max_packet, is_noproc;
    sshmsg_local input, output, success, failure, *result;
    char *cmd, *ttype, *modes, *user;
    /*
     * The sshmsg_local data type is a special structure used by
     * the message layer to read and write SSH messages.  The data
     * buffers pointed to by these structures are dynamically allocated
     * and managed by sshmsg_* routines.  Pre-define success and failure.
     */
    sshmsg_init_locals ( session->locus, 4, 
		&input, &output, &success, &failure );
    sshmsg_format_message ( &success, SSH_SMSG_SUCCESS );
    failure_fmt = 0;
    sshmsg_format_message ( &failure, SSH_SMSG_FAILURE );
    /*
     * Check for qualifiers on username and set flag for those detected.
     */
    user = session->ext_username;
    for ( is_noproc = is_admin = i = 0; user[i]; i++ ) if ( user[i] == '/' ) {
	char qualifier[16];
	tu_strnzcpy ( qualifier, &user[i], 15 );
	tu_strupcase ( qualifier, qualifier );
	if ( user[i+1] && (0 == tu_strncmp ( qualifier,
			"/ADMINISTRATE", tu_strlen(&user[i]))) ) {
	    is_admin = 1;
	    break;
	} else if ( user[i+1] && (0 == tu_strncmp ( qualifier,
			"/NOPROCESS", tu_strlen(&user[i]))) ) {
	    is_noproc = 1;
	    break;
	}
    }
    /*
     * Options negotiation phase.  Note that the message layer automatically
     * traps ignore and debug messages by default, so no need to check
     * for them.
     */
    for ( status=1, exec->mode = 0; (status&1) && (exec->mode == 0); ) {
	input.type = 0;
	status = cport_do_io(session->in_msg, CPORT_READ, &input, 1, &xcount);
	if ( status & 1 == 0 ) {
	    tu_strnzcpy ( errmsg, sshmsg_last_error_text(session), 255 );
	    break;
	}
	/* evr_event ( EVR_DEBUG, "Option request type: %d", input.type ); */
	switch ( input.type ) {
	    case SSH_CMSG_REQUEST_PTY:
		/*
		 * User is requesting a pseudo-terminal be connected
		 * to the command.  Copy message parameters to exec structure.
		 */
		if ( 7 != sshmsg_scan_message ( &input, &length, &ttype,
			&exec->tty_geometry[0], &exec->tty_geometry[1],
			&exec->tty_geometry[2], &exec->tty_geometry[3],
			&modes ) ) {
		    status = 20;	/* invalid packet */
		    break;
		}
		exec->tty_type = tm_malloc ( length+1 );
		tu_strnzcpy ( exec->tty_type, ttype, length );
		exec->tty_modes_len = input.length - length - 5*sizeof(int);
		if ( exec->tty_modes_len > 0 ) {
		    exec->tty_modes = tm_malloc ( exec->tty_modes_len+1 );
		    memcpy ( exec->tty_modes, modes, exec->tty_modes_len );
		}
		/*
		 * Check extended username for /ADMIN qualifier and disallow
		 * pty if found.  Also disallow if pty's are disabled.
		 */
		result = is_admin ? &failure : &success;
		if ( param.pty_mode == 0 ) result = &failure;
		status = cport_do_io (session->out_msg, CPORT_WRITE,
			 result, 1, &xcount );
		break;
	    case SSH_CMSG_WINDOW_SIZE:
		/*
		 * Geometry of remote user's screen.
		 */
		status = cport_do_io (session->out_msg, CPORT_WRITE,
			 &failure, 1, &xcount );
		break;


	    case SSH_CMSG_EXEC_SHELL:
	       /*
	        * Exec and cmd messages terminate this phase and
	        * begin the interactive phase.
		*/
		exec->mode = 2;
		if ( is_admin ) {
		    exec->mode = 3;
		    tu_strnzcpy ( exec->command, "SHELL\n", 8 );
		} else if ( is_noproc ) {
		    exec->mode = 4;
		    tu_strnzcpy ( exec->command, "SHELL\n", 8 );
		}
		break;

	    case SSH_CMSG_EXEC_CMD:
		/*
		 * Exec CMD, save command to be executed, include linefeed.
		 */
		exec->mode = 1;
		if ( 2 != sshmsg_scan_message ( &input, &length, &cmd ) ) {
		    status = 20;		/* bad scan */
		    break;
		}
		if ( length > sizeof(exec->fixed_command)-2 ) {
		    exec->command = tm_malloc ( length+2 );
		    if ( !exec->command ) { status = 20; break; };
		}
		tu_strnzcpy ( exec->command, cmd, length );
		real_length = tu_strlen ( exec->command );
		exec->command[real_length++] = '\n';
		exec->command[real_length] = '\0';
		status = 1;
		/*
		 * Check extended username for /ADMIN qualifier and change mode
		 * to 3 (management interface if found) or to 4 for /NOPROCESS
		 */
		if ( is_admin ) exec->mode = 3;
		else if ( is_noproc ) exec->mode = 4;
		break;

	    case SSH_CMSG_X11_REQUEST_FORWARDING:
	    case SSH_CMSG_PORT_FORWARD_REQUEST:
	    case SSH_CMSG_X11_FWD_WITH_AUTH_SPOOFING:
		/*
		 * Setup port forwarding, check for enabled.
		 */
		if ( input.type == SSH_CMSG_PORT_FORWARD_REQUEST &&
			(param.port_forward&2) == 0 ) {
		    status = 20;	/* not enabled */
		} else if ( !exec->port_fwd ) {
		    /*
		     * create pipe and start server.
		     */
		    status = init_forwarder ( session, exec );

		} else status = 1;
		if ( (status&1) == 1 ) {
		    /*
		     * Send packet to forwarder, reply is status.
		     */
		    cport_pipe_set_send_type ( exec->pf_out, input.type );
		    status = cport_do_io ( exec->pf_out, CPORT_WRITE,
			input.data, input.length, &xcount );
		    if ( (status&1) ) status = cport_do_io ( exec->pf_in,
			CPORT_READ, exec->pf_buffer, exec->pf_bufsize, 
			&xcount );
		    if ( (status&1) == 1 ) status = exec->pf_buffer[0];
		}
		/*
		 * Send reply based upon success of operation.
		 */
		status = cport_do_io (session->out_msg, CPORT_WRITE,
			 (status&1) ? &success : &failure, 1, &xcount );
		break;

	    case SSH_CMSG_MAX_PACKET_SIZE:
		/*
		 * Set max packet size.
		 */
		if ( 1 != sshmsg_scan_message ( &input, &max_packet ) ) {
		    status = 20;		/* assertion failure */
		} else if ( max_packet < 4096 ) {
		    status = cport_do_io ( session->out_msg, CPORT_WRITE,
			&failure, 1, &xcount );
		} else {
		    /*
		     * Packet size OK, set in pad layer.
		     */
		    sshpad_set_max_packet ( session->pad, max_packet );
		    status = cport_do_io ( session->out_msg, CPORT_WRITE,
			&success, 1, &xcount );
		}
		break;
	    default:
		/*
		 * Unexpected input, respond with failure message.
		 */
		status = cport_do_io ( session->out_msg, CPORT_WRITE,
			&failure, 1, &xcount );
		break;
	}
    }
    sshmsg_rundown_locals ( session->locus );
    return status;
}
/*************************************************************************/
/* Define queue structure for managing messages.
 */
struct ibuf {
    struct ibuf *next;
    sshmsg_local msg;
};
struct ibuf_queue {
    struct ibuf *head, *tail;
};
static void init_queue ( struct ibuf_queue *queue )
{
    queue->head = queue->tail = (struct ibuf *) 0;
}

static void enqueue_buffer ( struct ibuf_queue *queue, struct ibuf *buf )
{
    if ( queue->head == (struct ibuf *) 0 ) queue->head = buf;
    else queue->tail->next = buf;
    buf->next = (struct ibuf *) 0;
    queue->tail = buf;
}

static struct ibuf *dequeue_buffer ( struct ibuf_queue *queue )
{
    struct ibuf *head;
    head = queue->head;
    if ( head ) {
	queue->head = head->next;
        head->next = (struct ibuf *) 0;
    }
    return head;
}
/*************************************************************************/
/* Handle full-duplex interactive relay of data between client and
 * execution process.
 */
static int do_interactive ( sshsess_session session, 
	struct execution_state *exec )
{
    int status, i, length, client_ok, exec_ok, msize;
    struct ibuf_queue to_client, to_proc, to_x11, free;
    struct ibuf *client_in, *proc_in, *client_out, *proc_out, *cur, buf[6];
    struct ibuf *x11_in, *x11_out;
    struct ibuf blocked;	/* dummy placeholder */
    cport_isb isb;
    cport_port port;
    unsigned short int *iosb;
    char out_buf[4096];
    /*
     * The sshmsg_local data type is a special structure used by
     * the message layer to read and write SSH messages.  The data
     * buffers pointed to by these structures are dynamically allocated
     * and managed by sshmsg_* routines.  Input is process's sys$input
     * stream (we write to it) and output is process's sys$output stream
     * (we read it).
     */
    sshmsg_init_locals ( session->locus, 6, &buf[0].msg, &buf[1].msg,
	&buf[2].msg, &buf[3].msg, &buf[4].msg, &buf[5].msg );
    init_queue ( &to_client );
    init_queue ( &to_proc );
    init_queue ( &to_x11 );
    init_queue ( &free );
    for ( i = 0; i < 4; i++ ) enqueue_buffer ( &free, &buf[i] );
    if ( exec->port_fwd ) {
	for ( i = 4; i < 6; i++ ) enqueue_buffer ( &free, &buf[i] );
    }
    client_in = proc_in = client_out = proc_out = (struct ibuf *) 0;
    x11_in = x11_out = (struct ibuf *) 0;
    /*
     * Initial I/Os are write of command string to execution process
     * or read of additional data from client.
     */
    exec_ok = 1;
    client_ok = 1;
    if ( exec->command[0] ) {
	cur = dequeue_buffer ( &free );
	status = sshmsg_format_message ( &cur->msg, SSH_CMSG_STDIN_DATA,
		tu_strlen ( exec->command ), exec->command );
	enqueue_buffer ( &to_proc, cur );
    }
    /*
     * Continue processing until I/O error or exit_confirmation message.
     */
    port = session->in_msg->port;
    exec->tt_out->user_val = 1;
    exec->tt_in->user_val = 2;
    session->out_msg->user_val = 3;
    session->in_msg->user_val = 4;
    if ( exec->port_fwd ) {
	/*
	 * Init buffers for port forwarder pipe, first message may
	 * be mask.
	 */
	exec->pf_in->user_val = (exec->pf_mask_size > 0) ? 5 : 7;
	exec->pf_out->user_val = 6;
    }
    while ( exec_ok && client_ok ) {
	/*
	 * Start reads if idle.
	 */
	if ( !proc_in ) {
	    proc_in = dequeue_buffer ( &free );
	    if ( proc_in ) {
		status = cport_start_io ( exec->tt_out, CPORT_READ, out_buf,
			sizeof(out_buf) );
	    }
	}
	if ( !client_in ) {
	    client_in = dequeue_buffer ( &free );
	    if ( client_in ) {
		status = cport_start_io ( session->in_msg,
			CPORT_READ, &client_in->msg, 1 );
	    }
	}
	if ( exec->port_fwd && !x11_in ) {
	   /*
	    * Reserve a message buffer so we know we have someplace to
	    * put the data when the read into exec->pf_buffer completes.
	    */
	   x11_in = dequeue_buffer ( &free );
	    if ( x11_in ) {
		status = cport_start_io ( exec->pf_in,
			CPORT_READ, exec->pf_buffer, exec->pf_bufsize );
	    }
	}
	/*
	 * Start writes if idle and something queued.
	 */
	if ( !proc_out && to_proc.head ) {
	    /*
	     * Skip length count in formatted message.
	     */
	    proc_out = dequeue_buffer ( &to_proc );
	    status = cport_start_io ( exec->tt_in, CPORT_WRITE, 
		&proc_out->msg.data[4],	proc_out->msg.length-4 );
	}
	if ( !client_out && to_client.head ) {
	    client_out = dequeue_buffer ( &to_client );
	    status = cport_start_io ( session->out_msg,
		CPORT_WRITE, &client_out->msg, 1 );
	}
	if ( !x11_out && to_x11.head ) {
	    /*
	     * The message type must be sent via the IOSB.
	     */
	    x11_out = dequeue_buffer ( &to_x11 );
	    cport_pipe_set_send_type ( exec->pf_out, x11_out->msg.type );
	    if ( x11_out->msg.length > 0x07fff ) {	/* large message */
	        evr_event ( EVR_EXCEPTION, "Large X11 message received");
	    }
	    status = cport_start_io ( exec->pf_out,
		CPORT_WRITE, x11_out->msg.data, x11_out->msg.length );
	}
	/*
	 * Wait for next I/O  to complete.
	 */
	isb = cport_next_completion ( port );
	if ( !isb ) {
	    evr_event ( EVR_EXCEPTION, "BUGCHECK, null next event returned");
	    break;
	}
	iosb = (unsigned short int *) isb->iosb;
#ifdef DEBUG
	printf("|---completed isb addr: %x %x, value: %d iosb: %d %d\n", isb, 
		iosb, isb->user_val,  iosb[0], iosb[1] );
#endif
	if ( ((*iosb)&1) == 0 ) {
	    evr_event ( EVR_EXCEPTION,
	    	"error on I/O on isb %d: %d (exec_ok: %d)", 
		isb->user_val, *iosb, exec_ok );
	    client_ok = 0;
	}
	/*
	 * Take action based upon which pending action completed.
	 */
	switch ( isb->user_val ) {
	    int type;

	    case 1:
		/*
		 * Data read from process's sys$output channel.
		 */
		cur = proc_in;
		proc_in = (struct ibuf *) 0;

		if ( iosb[1] == 0 ) {
		    /*
		     * Zero length read implies end of file, get final
		     * status and start rundown.
		     */
		    int exitstatus;
		    exec_ok = 2;		/* issued exit status */
		    proc_in = &blocked;		/* stop reading tt_out */
		    cport_start_io ( exec->tt_out, CPORT_FCODES+1, 
			&exitstatus, sizeof(exitstatus) );
		    status = sshmsg_format_message ( &cur->msg,
			    SSH_SMSG_EXITSTATUS, exitstatus );
		} else {
		    /* Read from process's output channel completed */
		    status = sshmsg_format_message ( &cur->msg,
		        SSH_SMSG_STDOUT_DATA, iosb[1], out_buf );
		}
		if ( client_out != &blocked ) {
		    enqueue_buffer ( &to_client, cur );
		} else enqueue_buffer ( &free, cur );
		break;

	    case 2:
		/*  write to process input channel completed */
		if ( !proc_out ) fprintf(stderr,"BUGCHECK, proc_out is null\n");
		else enqueue_buffer ( &free, proc_out);
		proc_out = (struct ibuf *) 0;
		break;

	    case 3:
		/*
		 * Write of output message to client done, move to free queue.
		 */
		if ( client_out ) enqueue_buffer ( &free, client_out );
		client_out = (struct ibuf *) 0;
		break;

	    case 4:
		/*
		 * read of input message from client done, examine message
		 * type and take action accordingly.
		 */
		cur = client_in;
		client_in = (struct ibuf *) 0;
		type = cur->msg.type;
		if ( type == SSH_MSG_DISCONNECT ) {
		    session->exec_state = 2;
		    continue;
		} if ( type == SSH_CMSG_STDIN_DATA && exec_ok ) {
		    enqueue_buffer ( &to_proc, cur );
		} else if ( type == SSH_CMSG_EOF && exec_ok ) {
		    /*
		     * Write zero-length data to process as EOF marker.
		     */
		    sshmsg_format_message ( &cur->msg, SSH_CMSG_STDIN_DATA,
			0, "" );
		    enqueue_buffer ( &to_proc, cur );
		} else if ( type == SSH_CMSG_EXIT_CONFIRMATION ) {
		    /*
		     * Client confirmed our exitstatus.
		     */
		    client_in = &blocked;
		    client_ok = 0;
		} else if ( type == SSH_CMSG_WINDOW_SIZE ) {
		    /*
		     * Interactive window resize, extract new values and
		     * tell terminal driver.
		     */
		    int geometry[4];
	 	    if ( 4 != sshmsg_scan_message ( &cur->msg, &geometry[0],
				&geometry[1], &geometry[2], &geometry[3] ) ) {
			status = 20;
		    } else status = cport_start_io ( exec->tt_in, 
				CPORT_FCODES+0, geometry, sizeof(geometry) );
		    enqueue_buffer ( &free, cur );

		} else if ( exec->port_fwd && 
			    (type > 0) && (type < exec->pf_mask_size) &&
			    (exec->pf_mask[type] != 0) ) {
		    /*
		     * Hand off message to port forwarder thread.
		     */
		    enqueue_buffer ( &to_x11, cur );
		} else if ( type == SSH_MSG_PORT_OPEN ) {
		    /*
		     * No port forwarder, dynamically create one.
		     */
		    if ( (param.port_forward&1) == 0 ) {  /* not enabled */
		        sshmsg_format_message ( &cur->msg, SSH_SMSG_FAILURE );
			enqueue_buffer ( &to_proc, cur );
		    } else if ( exec->port_fwd ) {
			/*
			 * Port forwarder created but we haven't received
			 * mask yet, forward message as if we got it.
			 */
			enqueue_buffer ( &to_x11, cur );
		    } else if ( (1&init_forwarder ( session, exec )) == 1 ) {
			/*
			 * exec->port_fwd sucessfully inititialized, do
			 * buffer initialization that we skipped earlier.
			 */
			int i;
			evr_event(EVR_DEBUG, 
				"Dynamically created port forwarder" );
			for (i=4; i < 6; i++) enqueue_buffer(&free, &buf[i]);
			exec->pf_in->user_val = 7;	/* catch mask */
			exec->pf_out->user_val = 6;
			/*
			 * Send port_open to forwarder, first message
			 * in response will be mask of messagetypes.
			 */
			enqueue_buffer ( &to_x11, cur );
		    } else {
			/*
			 * Port forwarding failed, send failure message
			 */
		        sshmsg_format_message ( &cur->msg, SSH_SMSG_FAILURE );
			enqueue_buffer ( &to_proc, cur );
		    }
		} else if ( exec_ok && client_ok ) {
		    /*
		     * type unknown/unexpected.
		     */
		    evr_event ( EVR_EXCEPTION,  
			"Unknown message received from client: %d", type );
		    status = ssh_disconnect_session ( session, exec,
			exec_ok ? "protocol error" : "network error" );
		    client_in = &blocked;
		    client_ok = 0;
		}
		break;

	    case 5:
		/*
		 * port forwarder read done, relay to client
		 * All messages are int followed by string.
		 */
		if ( x11_in ) {
		    int s_size;
#ifdef DEBUG
unsigned char *bf; int k;
printf("message to relay to client: %d %d %d %x", 
iosb[0], iosb[1], iosb[3], exec->pf_buffer );
for ( k = 0; k < iosb[1] && k < 80; k++ ) printf ( "%c %x", (k%8) ? ' ': '\n',
exec->pf_buffer[k] );
printf("\n");
#endif
		    s_size = iosb[1] - 4;
		    if ( s_size <= 0 ) {
			status = sshmsg_format_message ( &x11_in->msg,
			    SSH_MSG_CHANNEL_CLOSE, exec->pf_buffer );
		    } else if (iosb[3] == SSH_MSG_CHANNEL_OPEN_CONFIRMATION) {
			status = sshmsg_format_message ( &x11_in->msg,
			    SSH_MSG_CHANNEL_OPEN_CONFIRMATION,
			    exec->pf_buffer );
		    } else if (iosb[3] == SSH_MSG_PORT_OPEN) {
			status = sshmsg_format_message ( &x11_in->msg,
				SSH_MSG_PORT_OPEN, exec->pf_buffer, s_size-4,
				&exec->pf_buffer[8], &exec->pf_buffer[4] );
		    } else {
		        status = sshmsg_format_message ( &x11_in->msg,
		            SSH_MSG_CHANNEL_DATA, exec->pf_buffer, s_size,
			    &exec->pf_buffer[4] );
		    }
		    x11_in->msg.type = iosb[3];
		    enqueue_buffer ( &to_client, x11_in );
		}
		x11_in = (struct ibuf *) 0;
		break;

	    case 6:
		/*
		 * Write of output message port forwarder done.
		 */
		if ( x11_out ) enqueue_buffer ( &free, x11_out );
		x11_out = (struct ibuf *) 0;
		break;

	    case 7:
		/*
		 * Dynamically started port forwarder sends mask as first
		 * message.
		 */
		isb->user_val = 5;	/* return to normal dispatching. */
		msize = iosb[1];
		exec->pf_mask_size = (msize+sizeof(int)-1)/(sizeof(int));
		msize = exec->pf_mask_size * sizeof(int);
		exec->pf_mask = tm_malloc ( msize );
		if ( exec->pf_mask ) memcpy ( exec->pf_mask, 
			exec->pf_buffer, msize );
		else exec->pf_mask_size = 0;
evr_event(EVR_DEBUG, "new port forwarder mask size: %d, server_number:", msize );
		/*
		 * Clear the x11_in buffer to force next read.
		 */
		enqueue_buffer ( &free, x11_in );
		x11_in = (struct ibuf *) 0;
		break;

	    default:
		fprintf(stderr, "BUGCHECK, Unexpected isb completion\n");

	}
    }
    /*
     * Cleanly break client connection and rundown execution process.
     */
#ifdef DEBUG
    printf("main loop exitted with client_ok: %d, exec_ok: %d\n",
	client_ok,exec_ok );
#endif
    status = cancel_pending ( session, exec );
    sshmsg_rundown_locals ( session->locus );
    return status;
}
