/*
 * Test program for establishing SSH client connection and issuing a
 * command.
 *
 * Usage:
 *    sethost_ssh [-l username] [-p port] [-c cipher] host [command]
 *
 * Author:	David Jones
 * Date:	4-JUN-1998
 */
#include <stdio.h>
#include <stdlib.h>
#include <ssdef.h>			/* VMS condition values */
#include <ttdef.h>			/* VMS terminal defintions */
#include <dcdef.h>			/* VMS device classes */
#include <iodef.h>			/* VMS QIO symbols */
#include <descrip.h>			/* VMS string descriptors */
#include <dvidef.h>			/* VMS device information */
#include <jpidef.h>			/* get process info */
#include <string.h>
#include <ctype.h>

#include "completion_port.h"
#include "cport_terminal.h"
#include "cport_ucx.h"
#include "cport_sshsess.h"		/* assumes sslinc: defined  */

static struct cipher_table { char *name; int code; } cipherdef[] =
{ { "none", SSH_CIPHER_NONE }, { "idea", SSH_CIPHER_IDEA },
  { "des", SSH_CIPHER_DES }, { "3des", SSH_CIPHER_3DES },
  { "rc4", SSH_CIPHER_RC4 }, { "blowfish", SSH_CIPHER_BLOWFISH },
  { "", -1 } /* end of list marker */
};

int SYS$QIOW(), LIB$GETDVI(), LIB$GETJPI();
struct terminal_iosb { 			/* IOSB for terminal driver: */
   unsigned short status; 		/*	completion status    */
   unsigned short count; 		/*	offset to terminator */
   unsigned short terminator;		/*	Terminating character*/
   unsigned short term_len;		/*	chars in term. sequence */
};

#define DEFAULT_SSH_PORT 22
#define MIN_BUF_DATA 16
#define PASSWORD_PROMPT_FORMAT "\r\n%s@%s password: "
static $DESCRIPTOR(terminal,"TT:");
/*
 * The follow data structures and routines are used to construct a queue to
 *  buffer output data.
 */
struct data_buf {
    struct data_buf *next;
    int type;
    int dlength;
    char data[1];			/* variable allocation */
};
struct data_buf_queue {
    struct data_buf *head, *tail;
    int busy;			/* If true, can't start */
    int closed;
};

static void enqueue_data ( struct data_buf_queue *q, int type, int size,
	char *data )
{
    /*
     * Allocate queue entry, sized to fit the amount of data, copy
     * the data to the entry, and add the entry to the end of the list.
     */
    struct data_buf *buf;
    if ( q->closed ) return;
    buf = (struct data_buf *) malloc ( sizeof(struct data_buf) + 
	(size<MIN_BUF_DATA?MIN_BUF_DATA:size) );
    buf->next = (struct data_buf *) 0;
    buf->type = type;
    buf->dlength = size;
    memcpy ( buf->data, data, size );
    if ( q->head ) q->tail->next = buf;
    else q->head = buf;
    q->tail = buf;
}
static void enqueue_streamed_data ( struct data_buf_queue *q, int type,
	int size, char *data )
{
    if ( q->head && (q->head != q->tail) && 
	(q->tail->type == type) &&
	((size + q->tail->dlength) <= MIN_BUF_DATA) ) {
	/*
	 * Append data to last packet.
	 */
	if ( q->closed ) return;
	memcpy ( &q->tail->data[q->tail->dlength], data, size );
	q->tail->dlength += size;
    } else {
	enqueue_data ( q, type, size, data );
    }
}
static void dequeue_data ( struct data_buf_queue *q ) {
    /*
     * Delete the next item from the head of the queue, do nothing if
     * queue empty.
     */
    struct data_buf *buf;
    buf = q->head;
    if ( buf ) {
	q->head = buf->next;
	free ( buf );
    }
}

static int remote_login(cport_isb, char *, char *, sshsess_session );
static int remote_command ( cport_isb, char *, sshsess_session );
static int remote_terminal ( cport_isb, char *, sshsess_session );
static int interactive_phase ( cport_isb tt, sshsess_session session );

static int register_port ( int code, int local_port, void *arg )
{
    if ( code == 1 ) {
	printf("Socket created with local port: %d\n", local_port );
    } else {
	printf("Socket connect aborted: %d\n", local_port );
    }
    return 1;
}
/***************************************************************************/
/* 
 */
int sethost_ssh ( 
	cport_isb tt,		/* stream assigned to local terminal */
	char *username,		/* Login name to use */
	char *host,		/* remote host name */
	int port,		/* remote SSH port */
	char *cipher_prefs,	/* cipher selection preferences */
	char *auth_prefs,	/* auth-type preferences */
	char *command )		/* remote command or "" */
{
    void *cnx;
    sshpad pad;			/* SSH Packet Assembler/Disassembler */
    int status, count;
    cport_port cport;
    sshsess_session session;
    sshmsg_local input, output;

    char remote_version[128], errmsg[256];
    static char *attributes[] = { "keepalive" };
    /*
     * Establish SSH connection and exchange protocol version strings.
     */
    cport = tt->port;
    status = cportucx_open_tcp_socket ( 
	host, 				/* Remote host */
	port, 				/* TCP port number */
	1, attributes, 			/* socket options */
	&cnx, 
	register_port, (void *) 0,	/* callback + arg */
	errmsg );			/* results */

    if ( (status&1) == 0 ) {
	fprintf ( stderr, "Error establishing connection to %s\n- %s\n", 
		host, errmsg );
	return status;
    }
    pad = sshpad_create (
	cport,				/* completion port to use */
	cnx,				/* TCP/IP context */
	SSHPAD_MODE_CLIENT,		/* We are client end */
	SSHPAD_DEFAULT_PROTOCOL,	/* SSH version string */
	remote_version,			/* server's response */
	errmsg );			/* diagnostic if error */
    if ( !pad ) {
	fprintf ( stderr, "Error establishing SSH connection to %s\n- %s\n",
		host,errmsg );
	return 0;
    }
    fprintf ( stderr, "Remote SSH version: %s", remote_version );
    /*
     * Establish SSH session key and begin encrypted communication.
     */
    session = sshsess_new_session ( 
	(sshsess_server) 0,		/* client mode, no server ctx */
	cport,				/* completion port */
	pad,				/* packet stream */
	cipher_prefs,			/* prioritized list of ciphers */
	auth_prefs,			/* prioritized list of auths */
	errmsg );
    if ( !session ) {
	fprintf ( stderr, "Error setting up secure communication to %s\n- %s\n",
		host, errmsg );
	return 0;
    }
    fprintf(stderr,"Selected cipher code: %d\n", session->cipher );
    /*
     * Authenticate ourselves with the server (send username and password).
     */
    status = remote_login ( tt, username, host, session );
    if ( (status&1) == 0 ) return status;
    /*
     * Select operating mode based upon presence of command on line.
     * (May want to add 'force PTY' flag to command line).
     */
    if ( *command ) {
	status = remote_command ( tt, command, session );
    } else {
	status = remote_terminal ( tt, command, session );
    }
    return status;
}
/***************************************************************************/
/* Handle non-interactive remote command.
 */
static int remote_command ( cport_isb tt, char *command, 
	sshsess_session session )
{
    int status, count, xfer;
    sshmsg_local input, output;
    /*
     * Connect local variables used for messages to session buffer pool.
     */
    sshmsg_init_locals ( session->locus, 2, &input, &output );
    /*
     * Send SSH_CMSG_EXEC_CMD message to server.
     */
    count = sshmsg_format_message ( 
	&output, 			/* receives result */
	SSH_CMSG_EXEC_CMD,		/* message type to build */
	strlen(command), 		/* type-specific argument */
	command );
    if ( count != 2 ) { 
	/*
	 * format routine did not process all arguments passed.
	 */
	fprintf ( stderr, "Internal error formatting message\n" );
	return 0;
    }

    status = cport_do_io ( session->out_msg, CPORT_WRITE, &output, 1, &xfer );
    if ( (status&1) == 0 ) {
	fprintf(stderr, "Error sending command string: %s\n",
		sshmsg_last_error_text ( session->locus ) );
	sshmsg_rundown_locals ( session->locus );
	return status;
    }
    sshmsg_rundown_locals ( session->locus );

    status = interactive_phase ( tt, session );
    return status;
}
/***************************************************************************/
/* Handle interactive remote command.
 */
static int remote_terminal ( cport_isb tt, char *command, 
	sshsess_session session )
{
    int status, count, modes_len, geometry[4], transferred;
    sshmsg_local input, output;
    char *tty_type, *tty_modes;
    /*
     * Connect local variables used for messages to session buffer pool.
     */
    sshmsg_init_locals ( session->locus, 2, &input, &output );
    /*
     * Request a PTY with the same charactistics as our terminal
     */
    status = cport_terminal_get_mode ( tt, &tty_type, geometry,
		&modes_len, &tty_modes );

    count = sshmsg_format_message ( &output, SSH_CMSG_REQUEST_PTY,
		strlen(tty_type), tty_type, 
		geometry[0], geometry[1], geometry[2], geometry[3], 
		modes_len+1, tty_modes );
    if ( count != 8 ) { fprintf ( stderr, "Format error\n" ); }

    status = cport_do_io ( session->out_msg, CPORT_WRITE,
		&output, 1, &transferred );
    if ( (status&1) == 0 ) {
	fprintf(stderr, "Error requesting: %s\n",
		sshmsg_last_error_text ( session->locus ) );
	sshmsg_rundown_locals ( session->locus );
	return status;
    }
    /*
     * Server will send either success or failure status message.
     */
    input.type = 899;
    status = cport_do_io(session->in_msg, CPORT_READ, &input, 1, &transferred);
    if ( (status&1) == 0 ) {
	fprintf(stderr, "Error reading response from server: %s\n",
		sshmsg_last_error_text ( session->locus ) );
	sshmsg_rundown_locals ( session->locus );
	return status;
    }
    if ( input.type != SSH_SMSG_SUCCESS ) {
	fprintf ( stderr, "Remote server failed to make PTY\n" );
	sshmsg_rundown_locals ( session->locus );
	return status;
    }
    /*
     * Send SSH_CMSG_EXEC_SHELL to server.
     */
    count = sshmsg_format_message ( &output, SSH_CMSG_EXEC_SHELL );
    if ( count != 0 ) { 
	/*
	 * format routine did not process all arguments passed.
	 */
	fprintf ( stderr, "Internal error formatting message\n" );
	return 0;
    }

    status = cport_do_io(session->out_msg, CPORT_WRITE, &output, 1, 
		&transferred);
    if ( (status&1) == 0 ) {
	fprintf(stderr, "Error sending username: %s\n",
		sshmsg_last_error_text ( session->locus ) );
	sshmsg_rundown_locals ( session->locus );
	return status;
    }
    sshmsg_rundown_locals ( session->locus );

    status = cport_terminal_set_mode ( tt, 1, 0 );

    if ( status&1 ) status = interactive_phase ( tt, session );
    return status;
}

/***************************************************************************/
/* Handle 'interactive' phase of SSH session.  Full duplex communication
 * between local terminal (tt) and remote shell (session).
 */
static int interactive_phase ( cport_isb tt, sshsess_session session )
{
    int status, count, is_connected, completion_status;
    sshmsg_local input, output;
    char line[1024];
    struct data_buf_queue to_terminal, to_server;
    cport_isb tt_in, isb;
    cport_port cport;
    /*
     * Connect local variables used for messages to session buffer pool.
     */
    sshmsg_init_locals ( session->locus, 2, &input, &output );
    /*
     * clone the tt isb to permit concurrent pending reads and writes.
     */
    cport = tt->port;
    tt_in = cport_terminal_duplex_stream ( tt );
    /*
     * Set up the queue headers for buffering data to be written
     * to the terminal (command's output) and the remote server (typed input).
     */
    to_terminal.head = (struct data_buf *) 0;
    to_server.head = (struct data_buf *) 0;
    to_terminal.busy = to_server.busy = 0;
    to_terminal.closed = to_server.closed = 0;
    completion_status = 0;
    /*
     * We have potentially 4 I/Os in progress at one time:
     *    bit
     *     0	Write to terminal.
     *     1    Read from terminal.
     *     2    Write to SSH connection.
     *     4    Read from SSH connection.
     *
     * The sshmsg_local variable's names are from the end user's perspective,
     * hence, we read the remote command output into output and write
     * the input to the remote server.
     */
    tt->user_val = 0;
    tt_in->user_val = 1;
    session->out_msg->user_val = 2;
    session->in_msg->user_val = 3;
    status = cport_start_io ( tt_in, CPORT_READ, line, sizeof(line) );

    status = cport_start_io ( session->in_msg, CPORT_READ, &output, 1 );

    for ( ; !to_terminal.closed || to_terminal.head ; ) {
	/*
	 * wait for completion of next I/O.
	 */
	isb = cport_next_completion ( cport );
	if ( !isb ) break;
	if ( (isb->default_iosb[0]&1) == 0 ){
	     fprintf(stderr,"I/O error, stream %d: %d\n", isb->user_val,
		isb->default_iosb[0] );
	      break;  /* I/O error */
	}
	/*
	 * Dispatch based on particular I/O that completed.
	 */
	switch ( isb->user_val ) {
	    int dlen, seg;
	    char *dp;
	    case 0:
	        /* 
		 * Write to terminal completed. Delete data just written.
		 */
	        dequeue_data ( &to_terminal );
		to_terminal.busy = 0;
		break;

	    case 1:
		/*
		 * format message to send to remote, either EOF, STDIN_DATA,
		 * or disconnect (other error).
		 */
		if ( line[0] == 28 ) {
		    fprintf(stderr,"sethost_ssh abort\n");
		    to_terminal.closed = 1;
		    break;	/* abort program */
		}
		enqueue_streamed_data ( &to_server, SSH_CMSG_STDIN_DATA,
			tt_in->default_iosb[1], line );
		status = cport_start_io(tt_in, CPORT_READ, line, sizeof(line));
		if ( (status&1) == 0 ) {
		    /*
		     * Failure to queue read means file at EOF.
		     */
		    enqueue_data ( &to_server, SSH_CMSG_EOF, 0, "" );
		}
		break;

	    case 2:
		/*
		 * Write of message completed, see what we wrote.
		 */
		to_server.busy = 0;
		if ( input.type == SSH_CMSG_EOF ) {
		} else if ( input.type == SSH_CMSG_EXIT_CONFIRMATION ) {
		    to_terminal.closed = 1;
		}
		break;
	    case 3:
		/*
		 * Message arrived from remote ssh server.
		 */
		if ( output.type == SSH_SMSG_STDOUT_DATA ) {
		    int i, start, length;
		    status = sshmsg_scan_message ( &output, &dlen, &dp );
		    for ( start = i = 0; i < dlen; i++ ) {
			/*
			 * Received buffer may be big, write to terminal
			 * line-by-line, converting single LFs to CRLF
			 */
			if ( dp[i] == '\n' || (i-start) > 255 ) {
			    enqueue_streamed_data  ( &to_terminal, 
				0, i-start+1, dp+start );
			    start = i + 1;
			}
		    }
		    if ( start < dlen ) enqueue_streamed_data ( &to_terminal, 0,
			dlen - start, dp+start );
		} else if ( output.type == SSH_SMSG_STDERR_DATA ) {
		    /*
		     * Extract data in message and send to terminal.
		     */
		    status = sshmsg_scan_message ( &output, &dlen, &dp );
		    enqueue_data ( &to_terminal, 0, dlen, dp );
		} else if ( output.type == SSH_SMSG_EXITSTATUS ) {
		    status= sshmsg_scan_message ( &output, &completion_status );
		    /*
		     * Acknowlege by writing SSH_CMSG_EXIT_CONFIRMATION
		     */
		    enqueue_data ( &to_server, SSH_CMSG_EXIT_CONFIRMATION,
			0, "" );
		    to_server.closed = 1;
		} else if ( output.type == SSH_MSG_DISCONNECT  ) {
		    /*
		     * Conneection aborted.
		     */
		    status = sshmsg_scan_message ( &output, &dlen, &dp );
		    fprintf(stderr,"Server disconnected" );
		    dp[dlen] = '\0'; fprintf(stderr, ": '%s'\n", dp );
		    to_server.closed = to_terminal.closed = 1;
		} else if ( output.type == SSH_SMSG_FAILURE  ) {
		    fprintf(stderr, "Command failed\n" );
		    to_server.closed = to_terminal.closed = 1;
		}
		/*
		 * Start reading next message.
		 */
		if ( !to_server.closed ) status = cport_start_io ( 
		     	session->in_msg, CPORT_READ, &output, 1 );
		break;
	    default:
		fprintf(stderr,"Unexpected I/O completion\n");
		    to_server.closed = to_terminal.closed = 1;
		break;
	}
	/*
	 * Check for idle output queues.
	 */
	if ( to_terminal.head && !to_terminal.busy ) {
	    to_terminal.busy = 1;
	    status = cport_start_io(tt, CPORT_WRITE, to_terminal.head->data,
		to_terminal.head->dlength );
	    if ( (status&1)==0 ) fprintf(stderr,
			"asynch TT error dumping buffer: %d\n",status);

	}
	if ( to_server.head  && !to_server.busy ) {
	    /*
	     * Always format as STDIN_DATA and then change.
	     */
	    to_server.busy = 1;
	    sshmsg_format_message ( &input, SSH_CMSG_STDIN_DATA,
		to_server.head->dlength, to_server.head->data );
	    input.type = to_server.head->type;
	    dequeue_data ( &to_server );
	    status = cport_start_io(session->out_msg, CPORT_WRITE, &input, 1);
	}
    }
    return 1;
}
/***************************************************************************/
/*  Handle authentication phase of new ssh session:
 *      1. Send username to SSH server and read response.
 *      2. If failure, prompt for password, send it to server and read response.
 *      3. Write result of authentication of stderr, return 1 or 0 for
 *	   success/failure.
 */
static int remote_login ( cport_isb tt, char *username, char *host,
		sshsess_session session )
{
    int status, count, transferred;
    sshmsg_local input, output;
    /*
     * Connect local variables used for messages to session buffer pool.
     */
    sshmsg_init_locals ( session->locus, 2, &input, &output );
    /*
     * Send SSH_CMG_USER message to server.
     */
    count = sshmsg_format_message ( 
	&output, 			/* receives result */
	SSH_CMSG_USER,			/* message type to build */
	strlen(username), 		/* type-specific argument */
	username );
    if ( count != 2 ) { 
	/*
	 * format routine did not process all arguments passed.
	 */
	fprintf ( stderr, "Internal error formatting message\n" );
	return 0;
    }

    status = cport_do_io ( session->out_msg, CPORT_WRITE,
		&output, 1, &transferred );
    if ( (status&1) == 0 ) {
	fprintf(stderr, "Error sending username: %s\n",
		sshmsg_last_error_text ( session->locus ) );
	sshmsg_rundown_locals ( session->locus );
	return status;
    }
    /*
     * Read response and prompt for password if needed.
     */
    status = cport_do_io ( session->in_msg, CPORT_READ, &input, 1,
		&transferred );
    if ( (status & 1) == 0 ) {
	fprintf(stderr, "Error in authentication phase: %s\n",
		sshmsg_last_error_text ( session->locus ) );
	sshmsg_rundown_locals ( session->locus );
	return status;
    }
    if ( input.type == SSH_SMSG_SUCCESS ) {
      fprintf ( stderr, "No password requested by server.\n");
    } else if ( input.type == SSH_SMSG_FAILURE ) {
	/*
	 * prompt user for password, noecho.
	 */
	struct terminal_iosb iosb;
	char *prompt;
	int prompt_len, pwd_len;
	char password[256];

	prompt_len = strlen(username) + strlen(host) +
		strlen(PASSWORD_PROMPT_FORMAT) + 8;
	prompt = malloc ( prompt_len + 1 );
	if ( !prompt ) return 0;
	sprintf ( prompt, PASSWORD_PROMPT_FORMAT, username, host );

	status = SYS$QIOW ( 0, tt->channel, 	/* event flag and channel */
	    IO$_READPROMPT|IO$M_NOECHO,	/* function code */
	    &iosb, 0, 0,		/* io status, AST arguments */
	    password, sizeof(password),	/* P1/P2: input buffer */
	    0, 0,			/* P3/P4: timeout, terminator block */
	    prompt, strlen(prompt) );	/* P5/P6: prompt string */

	if ( (status&1) == 0 ) {
	    fprintf ( stderr, "Internal error in terminal read: %d\n", status );
	    return status;
	}
  	fprintf ( stderr, "\n" );
	status = iosb.status;
	if ( (status&1) == 0 ) return status;
	/*
	 * Send password message.
	 */
	count = sshmsg_format_message ( &output, SSH_CMSG_AUTH_PASSWORD,
		iosb.count, password );
	if ( count != 2 ) { 
	    /*
	     * format routine did not process all arguments passed.
	     */
	    fprintf ( stderr, "Internal error formatting message\n" );
	    return 0;
        }
	status = cport_do_io ( session->out_msg, CPORT_WRITE, &output, 1,
		&transferred );
	if ( (status&1) == 0 ) {
	    fprintf(stderr, "Error sending password: %s\n",
			sshmsg_last_error_text ( session->locus ) );
	    sshmsg_rundown_locals ( session->locus );
	    return status;
	}
	/*
	 * Read response and continue.
	 */
	status = cport_do_io ( session->in_msg, CPORT_READ, &input, 1,
		&transferred );
	if ( (status & 1) == 0 ) {
	    fprintf(stderr, "Error in authentication phase: %s\n",
		sshmsg_last_error_text ( session->locus ) );
	    sshmsg_rundown_locals ( session->locus );
	    return status;
	}
    }
    /*
     * Test final response.
     */
    status = 0;
    if ( input.type == SSH_SMSG_SUCCESS ) {
	fprintf(stderr, "Authentication successful... \n" );
	status = 1;
    } else if ( input.type != SSH_SMSG_FAILURE ) {
	fprintf ( stderr, "Unexpected response from SSH server\n" );
    } else {
	fprintf(stderr, "Authentication failed.\n" );
    }
    sshmsg_rundown_locals ( session->locus );
    return status;

}

/***************************************************************************/
/* Main program, setup arguments and call sethost_ssh
 */
int main ( int argc, char **argv )
{
    char *username, *infile, *outfile, *host, *cipher, **command, *cmd_line;
    int port, command_count, i, status, length;
    cport_port cport;
    cport_isb tt;
    char errmsg[256];
    static char cipher_prefs[10] = { SSH_CIPHER_IDEA, SSH_CIPHER_3DES, SSH_CIPHER_RC4, SSH_CIPHER_DES, 0 };
    static char auth_prefs[] = { SSH_AUTH_PASSWORD, 0 };
    void *ttctx;
    /*
     * Process command line.
     */
    host = NULL;
    cipher = "des3";
    username = infile = outfile = NULL;
    command_count = 0;
    port = DEFAULT_SSH_PORT;

    for ( i = 1; i < argc && !host; i++ ) {
	if ( argv[i][0] == '-' ) {
	    switch ( argv[i][1] ) {
		case 'l':
		    i++;
		    if ( i < argc ) username = argv[i];
		    else fprintf(stderr,"Missing argument after -l\n");
		    break;

		case 'p':
		    i++;
		    if ( i < argc ) port = atoi ( argv[i] );
		    else fprintf(stderr,"Missing argument after -p\n");
		    break;

		case 'c':
		    i++;
		    if ( i < argc ) {
			/*
			 * rebuild cipher prefs list.
			 */
			int j, k;
			char *tok;
			k = 0;
			for ( tok = strtok ( argv[i], ","); tok && (k<9); 
				tok=strtok((char *) 0, tok) ) {
			    for ( j = 0; cipherdef[j].code >= 0; j++ ) {
				if ( strcmp(tok,cipherdef[j].name)==0 ) {
				    cipher_prefs[k++] = cipherdef[j].code;
				    break;
				}
			    }
			}
			cipher_prefs[k] = 0;
		    }
		    else fprintf(stderr,"Missing argument after -p\n");
		    break;

		default:
		    fprintf(stderr,"Unknown option: %s\n", argv[i] );
		    i = argc;
		    break;
	    }
	} else {
	    host = argv[i];
	    command_count = argc - i - 1;
	    if ( command_count > 0 ) command = &argv[i+1];
	}
    }
    if ( !host ) {
	fprintf ( stderr,
	    "Usage:\n  sethost_ssh [-l username] [-p port] [-c cipher] host [command ...]\n\n" );
	return 20;
    }
    /*
     * Initialize crypto.
     */
    sshrsa_init ( "" );
    /*
     * Make connection to user's terminal and establish it is a terminal.
     */
    cport = cport_create_port ( 0 );
    status = cport_terminal_open ( &ttctx, "TT:" );
    if ( (status&1) == 0 ) {
	status = cport_last_assign_status ( cport, errmsg );
	fprintf ( stderr, 
		"Error assigning channel to TT: %d '%s'\n", status, errmsg );
	return status;
    }
    tt = cport_assign_stream ( cport, &cportterminal_driver, ttctx, 0 );
    if ( tt ) {
	int code, class;
        code = DVI$_DEVCLASS;
        status = LIB$GETDVI ( &code, &tt->channel, 0, &class, 0, 0 );
	if ( (status&1) == 0 ) return status;
	if ( class != DC$_TERM ) {
	    fprintf ( stderr, "TT must be a terminal\n" );
	    return 1;
	}
    }
    /*
     * Get current username if missing.
     */
    if ( !username ) {
	int code = JPI$_USERNAME;
	static $DESCRIPTOR(current_user, "            ");

	current_user.dsc$a_pointer = malloc ( current_user.dsc$w_length +1 );
	status = LIB$GETJPI ( &code, 0, 0, 0, &current_user );
	username = current_user.dsc$a_pointer;
	for ( i =0; i < current_user.dsc$w_length; i++ ) {
	    username[i] = tolower(username[i]);
	    if ( username[i] == ' ' ) { username[i] = '\0'; break; }
	}
	fprintf(stderr, "Current user: '%s'\n", username );
    }
    /*
     * Build command_line from argv pieces, first total lengths of
     * of pieces, then allocate string and concatenate separated by spaces.
     */
    for ( length = i = 0; i < command_count; i++ ) {
	length = length + strlen ( command[i] ) + 1;
    }
    cmd_line = malloc ( length + 1 );
    for ( length = i = 0; i < command_count; i++ ) {
	strcpy ( &cmd_line[length], command[i] );
	length += strlen(command[i]);
	if ( (i+1) < command_count ) cmd_line[length++] = ' ' ;
    }
    cmd_line[length] = '\0';
    /*
     * Do the remote session.
     */
    status = sethost_ssh (
	tt, username, host, port, cipher_prefs, auth_prefs, cmd_line );

    return status;
}
