/*
 * Command line arguments:
 *    argv[1]	File name of parameters file.  If null, only defaults used.
 *
 * Logicals:
 *    ssh_host_key		PEM file containing RSA private key.
 *    ssh_server_key		PEM file containing RSA private key.
 *    ssh_login_configuration	File containing login configuration info.
 *    ssh_server_ciphers	Comma-separated list of ciphers to allow, e.g:
 *				    "none,idea,des,3des,rc4,blowfish"
 *    ssh_event_formatter	Program to log reported events.
 *    ssh_rsa_engine_command	If defined, command to spawn to start
 *				independant RSA computation engine.
 *    ssh_keygen_command	If define, command to spawn to start key
 *				generator background process.
 *
 * Author:	David Jones
 * Date:	17-MAY-1998
 * Revised:	20-MAY-1998	Additional cipher support: RC4, IDEA, Blowfish
 * Revised:	25-MAY-1998	Rework for completion port model.
 * Revsised:	13-JUN-1998	Add sshrsa_init with separate RSA engine option.
 * Revised:	19-JUN-1998	Change to use sshcipher module.
 * Revised:	28-JUN-1998	Switch from envir. var to parameter file.
 * Revised:	11-JUL-1998	isolate process spawn.
 * Revised:     25-JUL-1998	Re-order helper process startup.
 * Revised:	 7-AUG-1998	Change port parameter to list type.
 * Revsised:	 6-SEP-1998	Add admin interface.
 */
#include "pthread_1c_np.h"
#ifndef PTHREAD_USE_D4
	    typedef void * (*startroutine_t)(void *);
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <jpidef.h>		/* VMS getjpi codes */
#include <lckdef.h>
#include <descrip.h>
#include <ssdef.h>
int LIB$GETJPI(), SYS$DASSGN();
#define A(v) (lval=v,&lval)

#define KEYGEN_MAILBOX_NAME "SSH_KEYGEN_MAILBOX"

#define SSH_SERVER_VERSION "SSH-1.5-OSU_1.4alpha3\n"
#define PORT_COUNT_LIMIT 32

#include "parameters.h"		/* Global parameter settings */
#include "completion_port.h"
#include "cport_sshsess.h"	/* SSH session managment */
#include "cport_sshlogin.h"	/* login authentication */
#include "cport_ucx.h"		/* TCP/IP access via completion port */
#include "cport_admin.h"	/* Administrator interface */
#include "sshcipher.h"		/* Encryption support */
#include "event_report.h"	/* Event logging routines */
#include "helper.h"		/* Helper process creation */
#include "initiator_client.h"	/* User process initiator (no passwords) */
#include "pty_map.h"
#include "tmemory.h"		/* per-thread heaps */
#include "tutil.h"
#include "monitor.h"
/*
 * Forward and external references.
 */
static int spawn_key_generator ( char *command, char *skey_file,
	cport_port port, cport_isb *isb, helper_ctx ctx );
int ssh_monitor ( char *, sshsess_server server_ctx, cport_isb *listener, 
	cport_isb key_gen, helper_ctx helper, int stacksize, int client_limit, 
	char errmsg[256] );
/***********************************************************************/
/* Main function.
 */
int main ( int argc, char **argv )
{
    int sslmt_initialize();		/* SSLeay multithreading init */
    int auth_types;			/* Mask of authent. types supported */
    int ciphers;			/* mask of ciphers supported */
    int status, port, i, port_count;
    struct parameter_list_elem *next_port;
    struct helper_context generator_ctx;
    int stacksize, client_limit;	/* Thread parameters */
    int astlm, diolm, biolm, bytlm, code, lval, ltmp;
    sshsess_server server_ctx;
    cport_isb listener[PORT_COUNT_LIMIT], key_generator;
    cport_port cport;
    void *listen_ctx[PORT_COUNT_LIMIT];
    struct parameter_list_elem *citem;
    char *tok, *server_key_generator;
    char errmsg[256], version[80];
    static char *port_attributes[] = { "reuseaddr", "keepalive" };
    /*
     * Read parameter file.
     */
    status = init_parameters ( argc > 1 ? argv[1] : "" );
    /*
     * Initialize SSLeay to use DECthreads syncronization.
     */
    sslmt_initialize();
    cport = cport_create_port ( 0 );
    /*
     * Set up event logger.
     */
    status = evr_initialize ( );
    if ( (status&1) == 0 ) return status;
    strcpy ( version, SSH_SERVER_VERSION );
    version[strlen(version)-1] = '\0';		/* trim trailing newline */
    evr_event ( EVR_CONFIG, "%s server initializing, %t", version, 0 );
    pty_map_init();
    /*
     * Initialize authentication module.  It returns mask of auth_types
     * it can support.
     */
    status = sshlgn_initialize ( &auth_types, errmsg );
    if ( (status&1) == 0 ) {
	evr_event ( EVR_CONFIG,
		"Error initializing login module: %s", errmsg );
	return status;
    }
    /*
     * Create server context, which describes a single configuration
     * of server+host keys, ciphers, and authentication methods.
     */
    ciphers = 0;
    for ( citem = param.ciphers; citem; citem = citem->next ) {
	 int number;
	 if ( strcmp(citem->value, "all") == 0 ) {
	    /* allow all but none */
	    ciphers |= 0x0ffffff - (1<<sshciph_cipher_number("none"));
	} else {
	    number = sshciph_cipher_number ( citem->value );
	    if ( number >= 0 ) ciphers |= (1<<number);
	    else fprintf(stderr,"Unknown cipher name '%s' ignored.\n",citem->value );
	}
    }
    /*
     * Create initiator process if configured.
     */
    if ( *param.initiator ) {
	errmsg[0] = '\0';
	status = init_initiator ( param.initiator, errmsg );
	evr_event ( EVR_CONFIG, "Process initiator startup status: %d '%s'",
		status, errmsg );
    }
    /*
     * Create RSA computation helper process if configured.
     */
    status = sshrsa_init ( param.rsa_engine );
    if ( *param.rsa_engine ) evr_event ( EVR_CONFIG, 
	"rsa_engine startup status: %d", status );
    else evr_event ( EVR_CONFIG, "using internal RSA computation" );

    server_ctx = sshsess_server_context ( 
		param.skey_file, 		/* server key pem file */
		param.hkey_file,		/* host key pem file */
		ciphers,			/* Allowed ciphers */
		auth_types,			/* Allowed auth_types */
		errmsg );
    if ( !server_ctx ) {
	evr_event ( EVR_CONFIG,
		"Error creating server context: %s", errmsg );
	return 1;
    }
    /*
     * Create server key generator process if configured, this process
     * generates new server keys in the backgound.
     */
    key_generator = (cport_isb) 0;
    if ( *param.keygen ) {
	status = spawn_key_generator ( param.keygen, param.skey_file,
		cport, &key_generator, &generator_ctx );
    	evr_event ( EVR_CONFIG, 
	   "Key generator startup status: %d pid: %x", status,
		generator_ctx.pid );
    }
    /*
     * Create listener objects for accepting TCP connections and assign
     * a completion port stream to it.
     */
    i = 0;
    for ( next_port = param.ports; next_port; next_port = next_port->next) {
	if ( i >= (PORT_COUNT_LIMIT-1) ) {
	    evr_event ( EVR_CONFIG, 
		"Too many ports specified in parameters file" );
	    break;
	}
        port = atoi(next_port->value);
        status = cportucx_create_tcp_listener ( 
		port,				/* TCP port number */
		5,				/* backlog */
		2, port_attributes,		/* socket options */
		&listen_ctx[i],			/* returned handle */
		errmsg );			/* diagnostic message */
        if ( (status&1) == 0 ) {
	    evr_event ( EVR_CONFIG, 
		"Error listening on port %d, %s", port, errmsg );
	    return status;
	}
        listener[i] = cport_assign_stream ( cport, &cportucx_driver, 
		listen_ctx[i], 0 );
	i++;
    }
    listener[i] = (cport_isb) 0;	/* terminate list */
    port_count = i;
    /*
     * Compute client limit based upon bytelimit and asts.
     */
    LIB$GETJPI ( A(JPI$_BYTCNT), 0, 0, &bytlm );
    LIB$GETJPI ( A(JPI$_BIOLM), 0, 0, &biolm );
    LIB$GETJPI ( A(JPI$_DIOLM), 0, 0, &diolm );
    LIB$GETJPI ( A(JPI$_ASTCNT), 0, 0, &astlm );
    client_limit = bytlm / 6000;
    ltmp = astlm / 6; if ( ltmp < client_limit ) client_limit = ltmp;
    ltmp = biolm /5; if ( ltmp < client_limit ) client_limit = ltmp;
    ltmp = diolm / 4; if ( ltmp < client_limit ) client_limit = ltmp;
    client_limit = client_limit - port_count + 1;
    if ( client_limit < param.max_client ) param.max_client = client_limit;

    evr_event ( EVR_CONFIG, 
	"server initialized, port: %s%s, connection limit: %d", 
		param.ports->value, param.ports->next ? ",..." : "", 
		param.max_client );
    /*
     * Call function to service requests.
     */
    stacksize = 60000;
    status = ssh_monitor ( 
	SSH_SERVER_VERSION,		/* Server version to advertise */
	server_ctx, 			/* server context (keys) */
	listener, 			/* TCP channel */
	key_generator, &generator_ctx,	/* server key genrator info */
	stacksize,			/* Client thread stack size */
	param.max_client,		/* max number of connections */
	errmsg );
    if ( (status&1) == 0 ) {
	evr_event ( EVR_EXCEPTION, 
		"Error returned from ssh_server: '%s' at %t", errmsg, 0 );
    }

    return status;
}
/************************************************************************/
/* Create mailbox and spawn command.
 */
static int spawn_key_generator ( 
	char *command, 			/* DCL command to run generator image*/
	char *skey_file,		/* server key filename */
	cport_port port,		/* completion port for isb assign */
	cport_isb *isb, helper_ctx generator_ctx )
{
    int flags, ctl_chan, status, transferred;
    static long lksb[2];
    static char lock_name_buf[64];
    static $DESCRIPTOR(lock_name,lock_name_buf);
    long code, pid;
    /*
     * Create process.
     */
    *isb = (cport_isb) 0;
    status = helper_create(KEYGEN_MAILBOX_NAME, 256, command, generator_ctx);
    if ( (status&1) ) {
	/*
	 * Mailbox created, assign stream to mailbox for port-based IO.
	 */
	*isb = cport_assign_channel ( port, KEYGEN_MAILBOX_NAME, 0 );
	if ( *isb ) {
	    /*
	     * Write server keyfile name to process, this ensures it
	     * has initialized prior to our calculating client limits.
	     */
	    if ( status&1 ) status = cport_do_io ( *isb, CPORT_WRITE,
		skey_file, strlen(skey_file), &transferred );
	    evr_event(EVR_CONFIG,"Sent key file name to key generator: %d",
		status );
	} else {
	    fprintf(stderr,"Stream assign to control mbx failed\n");
	    status = 44;
	}
    }
    return status;
}
