/*
 * Encapsulate cipher handlers used by SSH packet layer.
 *
 * Author:  David Jones
 * Date:    19-JUN-1998
 * Revised: 27-JUN-1998		add sshcipher.h include
 * Revised: 29-JUN-1998		Fix RC4 cipher.
 * Revices: 12-APR-1999		Support OpenSSL library.
 * Revised: 26-MAY-1999		Fix for OpenSSL regression problems.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*
 * Logical name sslinc must point to SSLeay include directory.
 */
#ifdef VAX
#define des_init_random_number_generator des_ini_rnd_num_generator
#endif
#include "sslinc:crypto.h"		/* defines openssl_version if present*/
#include "sslinc:des.h"
#include "sslinc:idea.h"
#include "sslinc:rc4.h"
#include "sslinc:blowfish.h"
/*
 * Check for OpenSSL crypto library and set appropriate cast for argument
 * passing.
 */
#define DES_ARG_PTR des_cblock *
#ifdef OPENSSL_VERSION_NUMBER
#if (OPENSSL_VERSION_NUMBER >= 0x0920) && (OPENSSL_VERSION_NUMBER < 0x0903000L)
#undef DES_ARG_PTR
#define DES_ARG_PTR unsigned char *
#endif
#endif
/*
/*
 * Declare private overlay.
 */
union cipher_state {
     struct {
	struct idea_context {
	    int ivec_pos;		/* position in ivec for cfb */
	    int enc_flag;		/* Encrypting or decrypting */
	    IDEA_KEY_SCHEDULE ks;	/* expanded key schedule */
	    des_cblock ivec;		/* Initialization vector */
	} in, out;			/* context for each direction */
     } idea;

      struct{ struct des_context {
	char sequence[4];		/* encrypt/decrypt sequence */
	struct {
	    des_key_schedule ks;	/* expanded form of key */
	    des_cblock ivec;		/* initialization vector */
	} step[3];
       } in, out;			/* separate contexts for each. dir. */
      } des;

     struct {
	struct rc4_context {
	    RC4_KEY key;		/* key state */
	} in, out;			/* context for each direction */
     } rc4;

     struct {
	struct blowfish_context {
	    int enc_flag;
	    BF_KEY ks;			/* expanded key state */
	    des_cblock ivec;		/* initialization vector */
	} in, out;			/* context for each direction */
     } blowfish;

};
#define SSHCIPHER_STATE union cipher_state

#include "cport_sshsess.h"
#include "sshcipher.h"			/* verify templates */
#include "tmemory.h"
#include "tutil.h"

#define CIPHER_NAMES 7
static char *cipher_name[CIPHER_NAMES] = { 
	"none", "idea", "des", "3des", "tss", "rc4", "blowfish" };

int sshciph_cipher_number ( char *name )
{
    int i;
    for ( i = 0; i < CIPHER_NAMES; i++ ) {
	if ( tu_strncmp ( cipher_name[i], name, 9 ) == 0 ) return i;
    }
    return -1;		/* name not found */
}
char *sshciph_cipher_name  ( int number )
{
    if ( number < 0 || number >= CIPHER_NAMES ) return "";
    return cipher_name[number];
}

#ifdef DEBUG
static void dump_buffer ( char *heading, unsigned char *buffer, int bufsize )
{
    int i, j;
    printf ( "%s (%d bytes):", heading, bufsize );
    j = 15;
    for ( i = 0; i < bufsize; i++ ) {
	j++;
	if ( j > 15 ) {
	    printf("\n   %04x:", i ); j = 0; }
	printf(" %02x", 255&buffer[i] );
    }
    printf("\n");
}
int my_set_key ( DES_ARG_PTR key, des_key_schedule ks )
{
    unsigned char *p; int i, ret;

    p = (unsigned char *) key;
    printf ( "DES key is: "); for ( i=0;i<8;i++ ) printf(" %02x", p[i] );
    ret = des_set_key ( key, ks );
    printf("\nschedule:" );
    for ( i=0; i < sizeof(des_key_schedule); i++) printf (  " %02x", p[i] );
    printf ( "\nsched addr: %x\n", ks );
    printf ( "now DES key is: "); for ( i=0;i<8;i++ ) printf(" %02x", p[i] );
    printf("\n");
    return ret;
#define des_set_key my_set_key
}
#else
#define dump_buffer(a,b,c) 1
#endif

/************************************************************************/
/* Callback function to encrypt/decrypt data using IDEA cipher.
 */
static int idea_handler ( void *ctx, char *buffer, int size )
{
    struct idea_context *dctx;

    dctx = (struct idea_context *) ctx;
    idea_cfb64_encrypt ( 
	(unsigned char *) buffer, 	/* input */
	(unsigned char *) buffer,	/* output */
	size,				/* buffer length */
	&dctx->ks,			/* key shedule */
	dctx->ivec,			/* cipher block chain */
	&dctx->ivec_pos,
	dctx->enc_flag );		/* encrypt or decrypt indicator */
    return 1;
}
/************************************************************************/
/* Callback function to encrypt/decrypt data using blowfish cipher.
 * The blowfish routines in SSLeay load the cipher blocks big-endian, while
 * the blowfish code in SSH loads the 32-bit words little-endian.
 */
static void invert_byte_order ( int len, char *data )
{
    int i;
    unsigned char b0, b1, b2, b3, b4;
    for ( i = 0; i < len; i += 4 ) {
	b0 = data[i]; b1 = data[i+1];
	data[i] = data[i+3]; data[i+1] = data[i+2];
	data[i+2] = b1; data[i+3] = b0;
    }
}
static int blowfish_handler ( void *ctx, char *buffer, int size )
{
    struct blowfish_context *dctx;

    dctx = (struct blowfish_context *) ctx;
    dump_buffer ( "blowfish pre", (unsigned char *) buffer, size );
    invert_byte_order ( size, buffer );
    BF_cbc_encrypt ( 
	(unsigned char *) buffer, 	/* input */
	(unsigned char *) buffer,	/* output */
	size,				/* buffer length */
	&dctx->ks,			/* key shedule */
	dctx->ivec,			/* cipher block chain */
	dctx->enc_flag );		/* encrypt or decrypt indicator */
    invert_byte_order ( size, buffer );
    dump_buffer ( "blowfish post", (unsigned char *) buffer, size );
    return 1;
}
/************************************************************************/
/* Callback function to encrypt/decrypt data using RC4 stream cipher.
 */
static int rc4_handler ( void *ctx, char *buffer, int size )
{
    struct rc4_context *dctx;

    dctx = (struct rc4_context *) ctx;
    RC4 ( &dctx->key, size, 
	(unsigned char *) buffer, 		/* input */
	(unsigned char *) buffer );		/* output */
    return 1;
}
/************************************************************************/
/*
 * Callback function to encrypt data for DES and Triple-DES ciphers.
 */
static char des_enc_sequence[4] = { DES_ENCRYPT, 9, 0, 0 };
static char des_dec_sequence[4] = { DES_DECRYPT, 9, 0, 0 };
static char des3_enc_sequence[4] = {DES_ENCRYPT, DES_DECRYPT, DES_ENCRYPT, 9};
static char des3_dec_sequence[4] = {DES_DECRYPT, DES_ENCRYPT, DES_DECRYPT, 9};

static int des_handler ( void *ctx, char *buffer, int size )
{
    struct des_context *dctx;
    int i;
    dctx = (struct des_context *) ctx;
    for ( i = 0; dctx->sequence[i] < 2; i++ ) {
#ifdef DEBUG
	printf ("des_handler step %d, %x %x dir: %d\n", i, *((int *)buffer), 
	*((int *)&buffer[4]), dctx->sequence[i] );
#endif
        des_ncbc_encrypt ( (DES_ARG_PTR) buffer, (DES_ARG_PTR) buffer, 
		size, dctx->step[i].ks, (DES_ARG_PTR) &dctx->step[i].ivec, 
			dctx->sequence[i] );
#ifdef DEBUG
	printf ("          step %da, %x %x\n", i, *((int *)buffer), 
		*((int *)&buffer[4])  );
#endif
    }
    return 1;
}
/****************************************************************************/
int sshciph_state_size()
{
    return sizeof(union cipher_state);
}
/****************************************************************************/
/* Initialize cipher state and rig packet assem/disassem to encrypt.
 * Return value is 0 on err, 1 on success.
 */
int sshciph_setup ( struct sshsess_session_st *ctx, char errmsg[256] )
{
    int status, i;
    /*
     * set PAD to use selected cipher and initialize cipher context.
     * (use memset to set initialization vectors to zero).
     */
    switch ( ctx->cipher ) {
	case SSH_CIPHER_NONE:		/* no cipher */
	    break;

	case SSH_CIPHER_IDEA:			/* idea, cfb mode */
	    memset ( &ctx->cstate.idea, 0, sizeof(ctx->cstate.idea) );
	    idea_set_encrypt_key ( (unsigned char *) ctx->session_key,
		&ctx->cstate.idea.in.ks );
	    ctx->cstate.idea.in.enc_flag = IDEA_DECRYPT;
	    idea_set_encrypt_key ( (unsigned char *) ctx->session_key,
		&ctx->cstate.idea.out.ks );
	    ctx->cstate.idea.out.enc_flag = IDEA_ENCRYPT;

	    ctx->inbound = ctx->outbound = idea_handler;
	    ctx->in_state = &ctx->cstate.idea.in;
	    ctx->out_state = &ctx->cstate.idea.out;
	    break;

	case  SSH_CIPHER_DES:	/* des cipher */

	    memcpy ( ctx->cstate.des.in.sequence, des_dec_sequence, 4 );
	    memcpy ( ctx->cstate.des.out.sequence, des_enc_sequence, 4 );
	    status = des_set_key ( (DES_ARG_PTR) ctx->session_key,
			ctx->cstate.des.in.step[0].ks );
	    status = des_set_key ( (DES_ARG_PTR) ctx->session_key,
			ctx->cstate.des.out.step[0].ks );
	    memset ( ctx->cstate.des.in.step[0].ivec, 0, 
			sizeof(ctx->cstate.des.in.step[0].ivec) );
	    memset ( ctx->cstate.des.out.step[0].ivec, 0, 
			sizeof(ctx->cstate.des.out.step[0].ivec) );

	    ctx->inbound = ctx->outbound = des_handler;
	    ctx->in_state = &ctx->cstate.des.in,
	    ctx->out_state = &ctx->cstate.des.out;
	    break;

	case SSH_CIPHER_3DES:			/*  triple-des in CBC mode */
	    memset ( &ctx->cstate.des.in, 0, sizeof(ctx->cstate.des.in) );
	    memset ( &ctx->cstate.des.out, 0, sizeof(ctx->cstate.des.out) );
	    memcpy ( ctx->cstate.des.in.sequence, des3_dec_sequence, 4 );
	    memcpy ( ctx->cstate.des.out.sequence, des3_enc_sequence, 4 );
	    for ( i = 0; i < 3; i++ ) {
	    status = des_set_key ( (DES_ARG_PTR) &ctx->session_key[(2-i)*8],
			ctx->cstate.des.in.step[i].ks );
	    status = des_set_key ( (DES_ARG_PTR) &ctx->session_key[(i)*8],
			ctx->cstate.des.out.step[i].ks );
	    }
	    ctx->inbound = ctx->outbound = des_handler;
	    ctx->in_state = &ctx->cstate.des.in,
	    ctx->out_state = &ctx->cstate.des.out;
	    break;

	case SSH_CIPHER_TSS:			/* TSS experimental */
	    strcpy ( errmsg, "Unsupported cipher incorrectly negotiated");
	    return 0;
		
	    break;

	case SSH_CIPHER_RC4:			/* RC4 */
	    if ( !ctx->server_ctx ) {
		/*  client side. */
		RC4_set_key(&ctx->cstate.rc4.in.key, 16, ctx->session_key);
		RC4_set_key(&ctx->cstate.rc4.out.key, 16, ctx->session_key+16);
	    } else {
		RC4_set_key (&ctx->cstate.rc4.in.key, 16, ctx->session_key+16);
		RC4_set_key(&ctx->cstate.rc4.out.key, 16, ctx->session_key);
	    }
	    ctx->inbound = ctx->outbound = rc4_handler;
	    ctx->in_state = &ctx->cstate.rc4.in,
	    ctx->out_state = &ctx->cstate.rc4.out;
	    break;

	case SSH_CIPHER_BLOWFISH:
	    memset ( &ctx->cstate.blowfish, 0, sizeof(ctx->cstate.blowfish) );
	    BF_set_key ( &ctx->cstate.blowfish.in.ks, 32,
			(unsigned char *) ctx->session_key );
	    ctx->cstate.blowfish.in.enc_flag = BF_DECRYPT;
	    BF_set_key ( &ctx->cstate.blowfish.out.ks, 32,
		(unsigned char *) ctx->session_key );
	    ctx->cstate.blowfish.out.enc_flag = BF_ENCRYPT;

	    ctx->inbound = ctx->outbound = blowfish_handler;
	    ctx->in_state = &ctx->cstate.blowfish.in,
	    ctx->out_state = &ctx->cstate.blowfish.out;
	    break;

	default:
	    strcpy ( errmsg, "Invalid cipher negotiated" );
	    return 0;
    }
    return 1;
}
/*
	encrypted data format for key exhange (sent by client)
	cookie[8] = random data (from server_key message)
	session_id[0..15] = MD5(host_key//server_key//cookie)
	ctx->session_key[0..31] = random data.

	PKCS1(data) = 0//2//rand-fill//0//data

        transfer = RSA(server_key,RSA(PKCS1(host_key,session_key^session_id)))
 */
