/*
 * Support routines for RSA encryption used in SSH.  Since RSA computation is
 * CPU intensive, provide option to export operation to separate process.
 *
 * Author:	David Jones
 * Date:	19-JUN-1998
 * Revised:	11-JUL-1998
 * Revised:	14-JUL-1998		add lookup_public_key function.
 * Revised:	25-JUL-1998		Verify file ownership of auth keys.
 * Revised:	11-NOV-1998		Include login time in random init.
 * Revised:	11-APR-1998		do condition compile for OpenSSL
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stat.h>
#include <descrip.h>			/* VMS string descriptors */
#include <iodef.h>			/* VMS QIO function codes */
#include <lckdef.h>			/* VMS lock manager codes */
#include "tutil.h"
#include "pthread_1c_np.h"
/*
 * Following headers are from Eric Young's SSLeay, define logical sslinc
 * to point to directory containing the header files.
 */
#ifdef VAX
#define des_init_random_number_generator des_ini_rnd_num_generator
#define X509_REVOKED_get_ext_by_critical X509_REVOKED_get_ext_by_critcl
#endif
#include "sslinc:des.h"
#include "sslinc:pem.h"
#include "sslinc:rsa.h"
#include "sslinc:rand.h"
/*
 * Check for OpenSSL crypto library and set flag if alternate interface used.
 */
#ifdef OPENSSL_VERSION_NUMBER
#if OPENSSL_VERSION_NUMBER >= 0x0920
#define OPENSSL_CRYPTO 1
#endif
#endif

#include "sshrsa.h"
#include "helper.h"

#undef RSA_PKCS1_PADDING
#define RSA_PKCS1_PADDING 0
/*
 * Access to remote server is serialized via server.lock mutex, therefore
 * message buffers may be global.
 */
static struct helper_context server;

#include "sshrsa_rpc.h"			/* mailbox format */

static struct mbx_message  server_request, server_reply;
#ifdef DEBUG
static void dump_buffer ( char *heading, char *buffer, int bufsize )
{
    int i, j;
    printf ( "%s (%d bytes):", heading, bufsize );
    j = 15;
    for ( i = 0; i < bufsize; i++ ) {
	j++;
	if ( j > 15 ) {
	    printf("\n   %04x:", i ); j = 0; }
	printf(" %02x", 255&buffer[i] );
    }
    printf("\n");
}
#else
#define dump_buffer(a,b,c) 1
#endif
/*
 * Request function makes request and reads reply.  Caller must lock
 * mutex, fill in server_request, call request_operation, read result.
 */
static int request_rsa_operation ( int data_len )
{
    int status, reply_len;

    status = helper_write ( &server, &server_request,
	sizeof(server_request)-sizeof(server_request.data)+data_len );
    if ( status&1 == 0 ) return status;

    status = helper_read ( &server, &server_reply, sizeof(server_reply),
	&reply_len );
    dump_buffer ("raw reply recieved", (char *) &server_reply, reply_len );
		
    return status;
}
/*
 * Utility function to seed random number generator from process state.
 */
#include <jpidef.h>
#include <ssdef.h>
int sshrsa_init(char *rsa_server_command)
{
    long pid, iosb[2];
    int SYS$GETJPIW(), status, i, mbx_chan, SYS$GETTIM(), SYS$ENQW();
    static char lock_name_buf[64];
    static $DESCRIPTOR(lock_name,lock_name_buf);
    static long lksb[2];
    struct { short length, code; long *buffer; int *retlen; } item[10];
    struct { long pid, bufio, cputim, dio, pageflts, wssize, lgn[2]; } info;
#define SET_ITEM(A,B,C) item[A].length=sizeof(long); item[A].code=(B);\
	item[A].buffer = &info.C; item[A].retlen = (int *) 0;
    /*
     * Set up for JPI scan.
     */
    SET_ITEM(0,JPI$_PID,pid)
    SET_ITEM(1,JPI$_BUFIO,bufio)
    SET_ITEM(2,JPI$_DIRIO,dio)
    SET_ITEM(3,JPI$_CPUTIM,cputim)
    SET_ITEM(4,JPI$_PAGEFLTS,pageflts)
    SET_ITEM(5,JPI$_WSSIZE,wssize)
    SET_ITEM(6, JPI$_LOGINTIM, lgn[0])
    item[6].length = sizeof(info.lgn);
    item[7].length = item[7].code = 0;	/* terminate list */
    /*
     * Scan process list and load random number generator with results.
     */
    pid = -1;			/* search context */
    for ( i = 0; ; i++ ) {
	status = SYS$GETJPIW ( 8, &pid,  0, item, iosb, 0, 0 );
	if ( status == SS$_NOMOREPROC ) break;
	if ( status == SS$_NOPRIV ) continue;
	if ( status == SS$_SUSPENDED ) continue;
	if ( status == SS$_NORMAL ) {
	    RAND_seed ( (unsigned char *) &info, sizeof(info) );
	} else {
	    printf("unexpected error in JPI call: %d\n", status );
	    break;
	}
    }
    SYS$GETTIM ( iosb );
    RAND_seed ( (unsigned char *) iosb, sizeof(iosb) );
    /*
     * Set up for RPC calls.
     */
    server.pid = 0;
    if ( *rsa_server_command ) {
	int status;
	/*
	 * Create serarate process to handle RSA computations.
	 */
	status = helper_create ( MBX_NAME, 4096, 
		rsa_server_command, &server );
	return status;
    }
    return 1;
}


void verify_key ( RSA *key )
{
    unsigned char plain[1024], encrypted[1024], decrypted[1024];
    int i, size, result, result2;
    RSA *tmp;

    size = RSA_size(key) - RSA_PKCS1_PADDING;
    for (i=0; i < size; i++ ) plain[i] = (i&255);

    tmp = RSA_new();
    tmp->n = key->n;		/* modulus */
    tmp->e = key->e;		/* public exponent */

    result = RSA_public_encrypt ( size, plain, encrypted, tmp,
	 RSA_PKCS1_PADDING  );
    printf("Encrypted result: %d\n", result );
    tmp->n = tmp->e = NULL;
    RSA_free ( tmp );

    result2 = RSA_private_decrypt ( result, encrypted, decrypted, key,
		RSA_PKCS1_PADDING );
    printf("encrypted bytes: %d, decrypted: %d\n", result, result2);
    for ( i = 0; i < result; i++ ) printf ( "plain[%d] = %2x -> %2x ->%2x\n",
		i, plain[i], encrypted[i], decrypted[i] );

    return;
}

/****************************************************************************/
/* public functions.  Create a new key value.
 */
sshrsa sshrsa_create()
{
    /*
     * Allocate block to hold RSA structure.
     */
    RSA *new;

    if ( server.pid ) {
	/*
	 * Request new key from remote server.
	 */
	int status;
	helper_lock ( &server );
	server_request.code = 1;		/* allocate key */
	status = request_rsa_operation ( 0 );
	new = server_reply.key_id;
	helper_unlock ( &server );
    } else {
	/*
	 * No helper process, do it locally.
	 */
        new = RSA_new();
    }
    return (sshrsa) new;
}
sshrsa sshrsa_generate_key(int bits, int e, void (*callback)(int,int) )
{
    /*
     * Allocate block to hold RSA structure.
     */
    RSA *new;

#ifdef OPENSSL_CRYPTO
    typedef void (*openssl_cb)(int,int,char *);
    /*
     * Note: we assume calling convention allow calling a routine with
     * extra arguments.
     */
    new = RSA_generate_key(bits, e, (openssl_cb) callback, "" );
#else
    new = RSA_generate_key(bits, e, callback);
#endif
    if ( new ) {

	/* printf ( "Modulus size: %d bytes\n", RSA_size ( new ) );
	verify_key ( new ); */

    }
    return (sshrsa) new;
}
void sshrsa_destroy(sshrsa rsa)
{
    if ( server.pid ) {
	int status;
	helper_lock ( &server );
	server_request.code = 2;		/* deallocate key */
	server_request.key_id = rsa;
	status = request_rsa_operation ( 0 );
	helper_unlock ( &server );
    } else {
	RSA_free ( (RSA *) rsa );
    }
}
/*
 * Convert integer packed as bytes in big-endian order into SSLeay BIGNUM.
 */
static BIGNUM *pkcs1_to_bignum ( int bits, char *data )
{
    return BN_bin2bn ( (unsigned char *) data, (bits+7)/8, NULL );
}
/*
 * Convert public key info in key (exponent, modulus) to binary byte format.
 */
int sshrsa_extract_public_key ( sshrsa rsa, int *e_bits, char **e_data,
	int *m_bits, char **m_data )
{
    int bytes;
    RSA *key;
    if ( server.pid ) {
	int status;
	helper_lock ( &server );
	server_request.code = 4;		/* extract key */
	server_request.key_id = rsa;
	status = request_rsa_operation ( 0 );

	*e_bits = server_reply.len1 * 8;
	*e_data = malloc ( server_reply.len1 );
	memcpy ( *e_data, server_reply.data, server_reply.len1 );

	*m_bits = server_reply.len2 * 8;
	*m_data = malloc ( server_reply.len2 );
	memcpy(*m_data, server_reply.data+server_reply.len1,server_reply.len2);
	helper_unlock ( &server );

    } else {
    key = (RSA *) rsa;		/* look into opaque type */

    bytes = BN_num_bytes ( key->e );
    *e_bits = bytes*8;
    *e_data = malloc ( bytes );
    if ( !*e_data ) return 0;
    bytes = BN_bn2bin ( key->e, (unsigned char *) *e_data );
    if ( bytes*8 != *e_bits ) return 0;		/* consistency check */

    bytes = BN_num_bytes ( key->n );
    *m_bits = bytes*8;
    *m_data = malloc ( bytes );
    if ( !*m_data ) return 0;
    bytes = BN_bn2bin ( key->n, (unsigned char *) *m_data );
    if ( bytes*8 != *m_bits ) return 0;		/* consistency check */
    }
    dump_buffer ( "public key e bytes", *e_data, *e_bits/8 );
    dump_buffer ( "public key m bytes", *m_data, *m_bits/8 );
    return 1;
}

int sshrsa_set_private_key ( int ebits, char *exp, int mbits, 
	char *modulus, sshrsa rsa )
{
    /*
     *  load private exponent (d) and public modulus (n) into RSA structure,
     * converting byte-encoded integers (big-endian) into BIGNUM structues.
     */
    RSA *ctx;

    ctx = (RSA *) rsa;
    ctx->d = pkcs1_to_bignum ( ebits, exp );
    ctx->n = pkcs1_to_bignum ( mbits, modulus );
    return 1;
}
int sshrsa_set_public_key ( int ebits, char *exp, int mbits, 
	char *modulus, sshrsa rsa )
{
    /*
     *  load public exponent (e) and public modulus (n) into RSA structure,
     * converting byte-encoded integers (big-endian) into BIGNUM structues.
     */
    RSA *ctx;

    ctx = (RSA *) rsa;
    ctx->e = pkcs1_to_bignum ( ebits, exp );
    ctx->n = pkcs1_to_bignum ( mbits, modulus );
    return 1;
}

sshrsa sshrsa_read_keyfile ( char *keyfile_name, int *bits, char errmsg[256] )
{
    FILE *kf;
    RSA *key, *result;

    if ( server.pid ) {
	int status;
	helper_lock ( &server );
	server_request.code = 3;		/* read keyfile key */
	strcpy ( server_request.data, keyfile_name );
	status = request_rsa_operation ( strlen(keyfile_name) );
	result = server_reply.key_id;
	*bits = server_request.len1 * 8;
	helper_unlock ( &server );
    } else {
    errmsg[0] = '\0';
    *bits = 0;
    kf = fopen ( keyfile_name, "r" );
    if ( !kf ) {
	strcpy ( errmsg, "Error openning requested key file" );
	return (sshrsa) 0;
    }
    key = RSA_new();
    if ( !key ) {
	tu_strcpy ( errmsg, "Error creating RSA key structure" );
	fclose ( kf );
	return (sshrsa) 0;
    }
    result = PEM_read_RSAPrivateKey ( kf, &key, NULL );
    fclose ( kf );
    if ( !result ) {
	tu_strcpy ( errmsg, "Error reading RSA key from file" );
	return (sshrsa) 0;
    }
    *bits = RSA_size(result) * 8;
    }
    return (sshrsa) result;
}
int sshrsa_write_keyfile ( char *keyfile_name, sshrsa inp, char errmsg[256] )
{
    FILE *kf;
    RSA *key, *result;
    int status;

    if ( server.pid ) {
	int status;
	helper_lock ( &server );
	server_request.code = 7;		/* write keyfile key */
	strcpy ( server_request.data, keyfile_name );
	status = request_rsa_operation ( strlen(keyfile_name)+1 );
	helper_unlock ( &server );
    } else {
    errmsg[0] = '\0';
    kf = fopen ( keyfile_name, "r+" );
    if ( !kf ) {
	strcpy ( errmsg, "Error openning requested key file" );
	return 0;
    }
    key = (RSA *) inp;
    status = PEM_write_RSAPrivateKey ( kf, key, NULL, NULL, 0, NULL );
    fclose ( kf );
    if ( !status ) {
	tu_strcpy ( errmsg, "Error writing RSA key to file" );
	return 0;
    } else status = 1;
    }
    return status;
}
/*
 * Interpret character string as a very very large decimal number and
 * return as character array that encodes the raw bits.
 */
static int decode_number ( char *number, int numsize, int *bits, char **data )
{
    BIGNUM *a, *b, *c, *billion;
    int digit, i, j, pos, size;
    unsigned long value;
#ifdef OPENSSL_CRYPTO
    BN_CTX *ctx;
#else
#endif
    /*
     * Initialize Bignum variables.
     */
#ifdef OPENSSL_CRYPTO
    ctx = BN_CTX_new();
    if ( !ctx ) return 0;
    BN_CTX_init ( ctx );
#endif
    a = BN_new();			/* initially zero */
    b = BN_new();
    c = BN_new();
    billion = BN_new();
    BN_set_word ( billion, 1000000000 );
    /*
     * Get length of string and break into 9 char chunks.
     */
    size = 0;
    for ( i = 0; i < numsize; i += size ) {
	/*
	 * Convert next 'size' characters to an integer.
	 */
	if ( size == 0 ) {
	    size = numsize%9;
	    if ( size == 0 ) size = 9;
	} else size = 9;
	value = 0;
	for ( j = 0; j < size; j++ ) {
	    value = (value*10) + (number[i+j]-'0');
	}
	/*
	 * Shift and add.
	 */
#ifdef OPENSSL_CRYPTO
	BN_mul ( b, a, billion, ctx );		/* b = a * 10^9 */
#else
	BN_mul ( b, a, billion );		/* b = a * 10^9 */
#endif
	BN_set_word ( c, value );
	BN_add ( a, b, c );			/* a = b + c */
    }
    /*
     * Convert to raw array.
     */
    BN_free ( billion );
    BN_free ( c );
    BN_free ( b );
    if ( *bits > 0 ) a = bn_expand ( a, *bits );
    size = BN_num_bytes ( a );
    *data = malloc ( size );
    if ( *data ) i = BN_bn2bin ( a, (unsigned char *) *data );
    else i = 0;
    BN_free ( a );
#ifdef OPENSSL_CRYPTO
    BN_CTX_free(ctx);
#endif
    *bits = i*8;
    return 1;
}
/*
 * Match user file, looks up public key in specfied user key file and
 * sets key to match.  Format of lines in  authfile:
 *    bits exponent modulus description
 *
 * If uic is non-zero, authfile owner must match.
 * On success, errmsg holds first 255 bytes of matching description.
 */
int sshrsa_lookup_public_key ( char *authfile, long uic, int match_bits,
	char *match_modulus, sshrsa rsa, char errmsg[256] )
{
    FILE *af;
    int lookup_status, status, test_bits;
    char *token[4], *test_modulus;
    char line[4096];

    dump_buffer ( "User key to match", match_modulus, (match_bits+7)/8 );
    errmsg[0] = '\0';
    if ( server.pid ) {
	/*
	 * Make remote request to server.
	 */
	int status;
	helper_lock ( &server );
	server_request.code = 8;		/* Search file. */
	server_request.key_id = (void *) rsa;
	memcpy ( server_request.data, &uic, sizeof(uic) );
	strcpy ( &server_request.data[sizeof(uic)], authfile );
	server_request.len1 = sizeof(uic) + strlen(authfile) + 1;
	server_request.len2 = (match_bits+7)/8;
	if ( server_request.len2+server_request.len1 > 
	    sizeof(server_request.data) ) server_request.len2 =
	    sizeof(server_request.data) - server_request.len1;
	memcpy ( &server_request.data[server_request.len1],
		match_modulus, server_request.len2 );
	server_reply.len1 = 0;
	
	status = request_rsa_operation ( server_request.len1 +
		server_request.len2 );
	lookup_status = server_reply.len1;
	if ( lookup_status&1 ) tu_strnzcpy ( errmsg, server_reply.data, 255 );
	helper_unlock ( &server );
    } else {
    /*
     * Open authorization file and verify owner.
     */
    af = fopen ( authfile, "r" );
    if ( !af ) {
	tu_strcpy ( errmsg, "Error openning user authorized keys file" );
	return 0;
    }
    if ( uic ) {
	/*
	 * Get fileheader info and compare owner.
	 */
	struct stat statblk;
	LOCK_C_RTL
	status= fstat ( fileno(af), &statblk );
	UNLOCK_C_RTL
	if ( status < 0 ) {
	   tu_strcpy ( errmsg, "Error verifying authorized keys file owner" );
	   return 0;
	}
	if ( uic != statblk.st_uid ) {
	   tu_strcpy ( errmsg, "Authorized keys file has wrong ower UIC" );
	   return 0;
	}
    }
    /*
     * Scan file lines.
     */
    lookup_status = 0;
    while ( fgets ( line, sizeof(line), af ) ) {
	/*
	 * parse line into 4 separate strings.
	 */
	char c;
	int j, i, state;
	state = 0;
	for ( j = i = 0; line[i] && i < sizeof(line); i++ ) {
	    c = line[i];
	    if ( c == '\n' ) {
		line[i] = '\0';
		break;
	    }
	    if ( state == 0 ) {		/* looking for start of token */
		if ( c != ' ' && c != '\t' ) {
		    state = 1;
		    if ( j == 0 && c == '#' ) break;	/* comment */
		    if ( j >= 4 ) break;	/* too many tokens */
		    token[j] = &line[i];
		    if ( c == '"' ) {	/* quoted string */
			token[j]++;
			state = 2;
		    }
		    j++;
		}
	    } else if ( state == 1 ) {  /* in token */
		if ( c == ' ' || c == '\t' ) {
		    line[i] = '\0';
		    state = 0;
		}
	    } else if ( state == 2 ) {
		if ( c == '"' ) {
		    state = 0;
		    line[i] = '\0';
		}
	    }
	}
	/*
	 * Ignore blanks and comments.
	 */
	if ( j == 0 )  continue;
	if ( j < 3 ) {
	    tu_strcpy ( errmsg, "Invalid line in authorized keys file" );
	    status = 0;
	    break;
	}
	if ( j < 4 ) token[3] = "";		/* default description. */
	/*
	 * Convert modulus string to binary and compare.
	 */
	test_bits = match_bits;
	status = decode_number ( token[2], tu_strlen(token[2]),
		&test_bits, &test_modulus );
	if ( (status&1) == 0 ) continue;
	dump_buffer ( "User key to test", match_modulus, (test_bits+7)/8 );
        j = (match_bits+7)/8;
	if ( test_bits == match_bits ) {
	    for ( i = 0; i < j; i++ ) 
	        if ( test_modulus[i] != match_modulus[i] ) break;
	} else i = 0;
	free ( test_modulus );
	if ( (i >= j)  && (i > 0) ) {
	    /*
	     * Matched public key.  Generate RSA entity using exponent
	     * in file record and exponent to match.
	     */
	    char *matched_exponent;
	    test_bits = 32;
	    status = decode_number ( token[1], tu_strlen(token[1]),
		&test_bits, &matched_exponent );
	    if ( status&1 == 1 ) {
	        tu_strnzcpy ( errmsg, token[3], 255 );
		status = sshrsa_set_public_key ( test_bits, 
			matched_exponent, match_bits, match_modulus, rsa );
		free ( matched_exponent );
	        lookup_status = 1;
	    }
	    break;
	}
    }
    fclose ( af );
    }
    return lookup_status;
}
/***************************************************************************/
/*
 * Routines below handle the RSA part of the special encryption format used
 * for session keys.
 */
int sshrsa_encrypt_number ( int bits, char *number, sshrsa opaque_key,
	char *buffer, int bufsize, int *outsize )
{
    int len, size, i, status;
    char *tmp, tmp_buffer[512];
    RSA *key;
    /*
     * Trim buffer of leading zeros, get final length.
     */
    dump_buffer ( "RSA- plaintext", number, (bits+7)/8 );
    for ( len = (bits+7)/8; len > 0 && (*number == '\0'); --len ) number++;

    if ( server.pid ) {
	int status;
	helper_lock ( &server );
	server_request.code = 5;		/* encrypt key */
	server_request.key_id = opaque_key;
	memcpy ( server_request.data, number, len );
	server_request.len1 = len;
	server_request.len2 = bufsize;
	status = request_rsa_operation ( len );
	*outsize = server_reply.len1;
	if ( server_reply.len1 < bufsize ) bufsize = server_reply.len1;
	memcpy ( buffer, server_reply.data, bufsize );
	helper_unlock ( &server );
	return 1;
    } else {
    key = (RSA *) opaque_key;
    size = RSA_size ( key );
    if ( size > bufsize ) return 0;		/* buffer too small */
    if ( len+3 > size ) return 0;		/* number too big */
    /*
     * Initialize header and fill with random data terminated by a 0.
     */
    if ( size <= sizeof(tmp_buffer) ) tmp = tmp_buffer;
    else tmp = malloc ( size );
    tmp[0] = 0;
    tmp[1] = 2;
    RAND_bytes ( (unsigned char*) &tmp[2], (size-len-3) );
    for ( i = 2; i < size-len-1; i++ ) {
	/*
	 * redo any zeros we encounter.
	 */
	if ( tmp[i] == 0 ) tmp[i] ^= tmp[i-1];
    }
    tmp[size-len-1] = '\0';
    /*
     * Concatenate data bytes.
     */
    memcpy ( &tmp[size-len], number, len );
    dump_buffer ( "RSA- encrypting", tmp, size );
    status = RSA_public_encrypt ( size, (unsigned char *) tmp, 
	(unsigned char *) buffer, key, 	RSA_NO_PADDING );
    dump_buffer ( "RSA- result", buffer, size );
    *outsize = status;

    if ( tmp != tmp_buffer ) free ( tmp );
    }
    return (status==size) ? 1 : 0;
}
int sshrsa_decrypt_number ( int insize, char *number, sshrsa opaque_key,
	char *buffer, int bufsize, int *outsize )
{
    int len, size, i, status;
    char *tmp, tmp_buffer[512];
    RSA *key;
    /*
     * size of number assumed to be size of key.
     */
    if ( server.pid ) {
	int status;
	size = 16;
	helper_lock ( &server );
	server_request.code = 6;		/* decrypt buffer */
	server_request.key_id = opaque_key;
	server_request.len1 = insize;
	server_request.len2 = bufsize;
	if ( insize > sizeof(server_request.data) ) 
		insize = sizeof(server_request.data);
	memcpy ( server_request.data, number, insize );
	status = request_rsa_operation ( insize );
	if ( (status&1) == 1 ) {
	   *outsize = server_reply.len1;
	   if ( *outsize > sizeof(server_request.data) ) 
		*outsize = sizeof(server_request.data);
	   memcpy ( buffer, server_reply.data, *outsize );
	}
	helper_unlock ( &server );

    } else {
    key = (RSA *) opaque_key;
    size = RSA_size ( key );
    if ( size != insize ) return 0;		/* wrong size */
    if ( size <= sizeof(tmp_buffer) ) tmp = tmp_buffer;
    else tmp = malloc ( size );
    *outsize = 0;

    dump_buffer ( "RSA- decrypting", number, size );
    i = RSA_private_decrypt ( size, (unsigned char*) number,
	(unsigned char *) tmp, key, RSA_NO_PADDING );
    dump_buffer ( "RSA- result", tmp, i );

    if ( tmp[0] == '\0' && tmp[1] == '\2' ) {
	/*
	 *   buffer starts out OK, trim the garbage.
	 */
	*outsize = bufsize;
	memcpy ( buffer, &tmp[size-bufsize], bufsize );
    dump_buffer ( "RSA- trimmed", buffer, bufsize );
	status = 1;
    } else {
	/* Decryption failed. */
	status = 0;
    }
    if ( tmp != tmp_buffer ) free ( tmp );
    }
    return status;
}
/**************************************************************************/
/* Handle processing of remote messages for RPC-based RSA engines.
 */
int sshrsa_remote_request ( struct mbx_message *request, int request_dlen, 
	struct mbx_message *reply, int *reply_dlen )
{
    FILE *kf;
    RSA *key;
    int bytes, e_size, n_size, status;
    long uic;

    *reply_dlen = 0;
    switch ( request->code ) {
	case 1:			/* allocate key */
	    reply->key_id = RSA_new();
	    break;

	case 2:			/* deallocate key */
	    RSA_free ( request->key_id );
	    break;

	case 3:			/* read keyfile */
	    request->data[request_dlen] = '\0';
	    key = RSA_new();
	    kf = fopen ( request->data, "r" );
	    reply->key_id = PEM_read_RSAPrivateKey ( kf, &key, NULL );
	    fclose ( kf );
	    break;

	case 4:			/* extract public key */
	    key = (RSA *) request->key_id;
	    reply->key_id = request->key_id;
	    e_size = BN_num_bytes ( key->e );
	    n_size = BN_num_bytes ( key->n );
	    if ( (e_size + n_size) < sizeof(reply->data) ) {
		
		    reply->len1 = BN_bn2bin ( key->e, 
			(unsigned char *) reply->data );
		    if ( reply->len1 <= 0 ) break;

		    reply->len2 = BN_bn2bin ( key->n, 
			(unsigned char *) &reply->data[reply->len1] );
		    *reply_dlen = reply->len1 + reply->len2;
	    }
	    break;

	case 5:			/* encrypt via public key */
	    sshrsa_encrypt_number ( request->len1*8, request->data,
		(RSA *) request->key_id, reply->data,
			request->len2, &reply->len1 );
	    *reply_dlen = reply->len1;
	    break;

	case 6:			/* decrypt via private key */
	    sshrsa_decrypt_number ( request->len1, request->data,
		(RSA *) request->key_id, reply->data,
		request->len2, &reply->len1 );
	    *reply_dlen = reply->len1;
	    break;

	case 7:			/* write keyfile */
	    sshrsa_write_keyfile ( request->data, (RSA *) request->key_id, 
			reply->data );
	    *reply_dlen = strlen ( reply->data );
	    break;

	case 8:			/* match public key in user file */
	    memcpy ( &uic, request->data, sizeof(uic) );
	    reply->len1 = sshrsa_lookup_public_key ( 
		&request->data[sizeof(uic)], uic, request->len2*8,
		&request->data[request->len1], request->key_id,	reply->data );
	    *reply_dlen = strlen ( reply->data ) + 1;
	    break;

	default:
	    reply->code = -1;
	    break;
    }
    return 1;
}
