/*
 *    int sslmt_initialize()	Initializes pthread-based lock structures
 *				and sets up callbacks in  SSL library.
 *				Returns 1 on success.
 *
 *  Author: David Jones
 *  Date:   9-JUN-1998		Pulled from bss_mst routine.
 *  Revised: 27-JUl-1998	Use crypto.h instead of cryptlink.h.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*
 */
#include "pthread_1c_np.h"	/* pthreads portability wrapper */

#include "sslinc:crypto.h"
/* #include "buffer.h" */

/*************************************************************************/
/*
 * Declare storage for locks and string tables to convert arguments to text.
 */
struct pt_rw_lock {		/* basic structure for read/write lock */
    pthread_mutex_t mutex;	/* Mutex for accessing structure elements */
    pthread_cond_t update_done;
    int reader_count;
    int update_pending;
};

static struct pt_rw_lock lock_table[CRYPTO_NUM_LOCKS];

static char *lock_modes[4] = { "null", "lock", "unlock", "Lock+unlock" };
static char *access_modes[4] = { "none", "read", "write", "read+write" };
static char* lock_names[CRYPTO_NUM_LOCKS] =	/* For debugging display */
	{
	"<<ERROR>>",	"err",	"err_hash",	"x509",
	"x509_info",	"x509_pkey",	"x509_crl",	"x509_req",
	"dsa",	"rsa",	"evp_pkey",	"certificate_ctx",
	"ssl_ctx",	"ssl_cert",	"ssl_session",	"ssl",
	"Random", "malloc", "bio", "bio_gethostbyname"
	};

/***********************************************************************/
/* Routines for returning thread ID, which is zero until set by a
 * higher layer.  Save in thread specific storage.
 */
static pthread_key_t identity;
static void identity_rundown ( void *ctx )
{
}

static unsigned long pt_thread_id()
{
    unsigned long *result;
    GET_SPECIFIC ( identity, result );
    return result ? *result : 0;
}
static pt_set_identity ( unsigned long *ndx )
{
    /* note that storage pointed to by ndx must remain for life of thread. */
    pthread_setspecific ( identity, ndx );
}
/***********************************************************************/
static void pt_lock ( int mode, int id, char *file, int line )
{
    struct pt_rw_lock *lock;
#ifdef LOCK_DEBUG
    char *fname;
    fname = strstr ( file, "]" );	/* trim directory path */
    if ( !fname ) fname = file; else fname++;
    printf("pt_lock request for %s: %s - %s %s/%d\n", lock_names[id], 
	lock_modes[mode&3], access_modes[(mode/4)&3], fname, line );
#endif
    /*
     * Always need mutex locked before we can examine structure.
     */
    lock = &lock_table[id];
    pthread_mutex_lock ( &lock->mutex );
    /*
     * Interpret the mode argument.
     */
    if ( mode&CRYPTO_LOCK ) {
	/*
	 * In all cases, update_pending flag must be clear before we can
	 * proceed.
	 */
	while ( lock->update_pending )
	    pthread_cond_wait ( &lock->update_done, &lock->mutex );

	if ( mode&CRYPTO_READ ) lock->reader_count++;
	else /* if ( mode&CRYPTO_WRITE ) */ {
	    /*
	     * Assume not-read mode means write (safe way to allow caller to 
	     * get away  with a using mode CRYPTO_LOCK/CRYPT_UNLOCK only.
	     * Set write pending flag to block out new access to lock and wait
	     * for readers to clear.
	     */
	    lock->update_pending = 1;
	    while ( lock->reader_count > 0 )
		pthread_cond_wait ( &lock->update_done, &lock->mutex );
	    lock->reader_count = 1;	/* re-aquire read lock for ourselves */
	}
    } else if ( (mode&CRYPTO_UNLOCK) && (lock->reader_count > 0) ) {
	/*
	 * Signal potential waiters when we are last reader/writer to release
	 * lock.  If we are the writer (mode not CRYPTO_READ) clear write
	 * pending flag.
	 */
	--lock->reader_count;
	if ( (lock->reader_count == 0) && lock->update_pending ) {
	    if ( (mode&CRYPTO_READ) == 0 ) lock->update_pending = 0;
	    pthread_cond_broadcast ( &lock->update_done );
	}
    } else {
	/* oops */
	fprintf ( stderr,
	   "%s '%s', mode=%s access=%s, thread=%d at line %d in file %s\n",
		"BUGCHECK, invalid request for lock", lock_names[id], 
		lock_modes[mode&3], access_modes[(mode/4)&3],
		pt_thread_id(), line, file );
    }
    pthread_mutex_unlock ( &lock->mutex );
}

static int pt_add_lock ( int *counter, int increment, int id, char *file, int line )
{
     /*
      * Operation is nothing more complicated than a mutex around an update.
      */
    int result;
#ifdef LOCK_DEBUG
    char *fname;
    fname = strstr ( file, "]" );
    if ( !fname ) fname = file; else fname++;
    printf("pt_add_lock request for %s: %d %s/%d\n", lock_names[id], increment, fname, line );
#endif
    pthread_mutex_lock ( &lock_table[id].mutex );
    result = *counter + increment;
    *counter = result;
    pthread_mutex_unlock ( &lock_table[id].mutex );
    return result;
}

int sslmt_initialize ()
{
    /*
     * Initialize for multithreaded access.
     */
    int status, i;
    /*
     * Make context for saving thread ID and build lock table.
     */
    status = CREATE_KEY ( &identity, identity_rundown );
    for ( i = 0; i < CRYPTO_NUM_LOCKS; i++ ) {
	INITIALIZE_MUTEX ( &lock_table[i].mutex );
	INITIALIZE_CONDITION ( &lock_table[i].update_done );
	lock_table[i].reader_count = lock_table[i].update_pending = 0;
    }
    /*
     * Replace cryptlib's lock function with our own callbacks.
     */
    CRYPTO_set_locking_callback ( pt_lock );
    CRYPTO_set_add_lock_callback ( pt_add_lock );
    CRYPTO_set_id_callback ( pt_thread_id );

    return 1;
}
