/*
 * Return user authorization information about a user:
 *	uic			User uic
 *	Login dir.		User's login directory (device + dir),
 *				converted to unix filename format.
 *
 * Revised:	10-AUG-1994	Fixed bug in checking for disable account.
 * Revised:	14-FEB-1995	Check OS version and use context arg in GETUAI.
 * Revised:	28-MAR-1995	Fixed bug in GETSYI call (length returned as short)
 * Revised:	29-JUL-1996	Ensure global lock around getuai calls.
 * Revised:	17-NOV-1996	Rework initialization code.
 * Revised:	21-NOV-1996	Rework cmexec code.
 * Revised:	8-MAY-1998	Add password fields.
 * Revised:	17-JUL-1998	Re-organize API, return data in struct.
 * Revised:	25-JUL-1998	Add newmail item.
 * Revised:	7-AUG-1998	Add ssh_update_user_logfails function.
 */
#include <stdio.h>
#include <stdlib.h>
#include <descrip.h>
#include <uaidef.h>
#include <syidef.h>
#include <prvdef.h>
#include <ssdef.h>
#include "pthread_1c_np.h"
#include "tutil.h"
#include "user_info.h"

int SYS$GETUAI(), SYS$SETUAI(), SYS$GETSYIW(), SYS$SETPRV(), SYS$CMEXEC();
#define USERNAME_MAX 12

static pthread_once_t uaf_setup = PTHREAD_ONCE_INIT;
static int uai_version = 0;		/* 0- unknown, 1- old, 2 - new */
static int cmexec_serialization;	/* If true, use $CMEXEC */
static long context;			/* GETUAI context argument */
static long upd_context;		/* SETUAI context argument */
typedef struct { short length, code; char *buffer; int *ret_len; } item_list;
/***************************************************************************/
/* Serialized call to GETUAI, set and reset SYSPRV privilege bit.
 */
static int do_getuai ( long *ctxptr, 
	struct dsc$descriptor_s *username_dx, item_list *item )
{
    int status, privs[2];
    privs[1] = 0; privs[0] = PRV$M_SYSPRV;
    SYS$SETPRV ( 1, privs, 0, 0 );
    status = SYS$GETUAI ( 0, ctxptr, username_dx, item, 0, 0, 0 );
    SYS$SETPRV ( 0, privs, 0, 0 );
    return status;
}
/***************************************************************************/
/* Serialized call to SETUAI, set and reset SYSPRV privilege bit.
 */
static int do_setuai ( long *ctxptr, 
	struct dsc$descriptor_s *username_dx, item_list *item )
{
    int status, privs[2];
    privs[1] = 0; privs[0] = PRV$M_SYSPRV;
    SYS$SETPRV ( 1, privs, 0, 0 );
    status = SYS$SETUAI ( 0, ctxptr, username_dx, item, 0, 0, 0 );
    SYS$SETPRV ( 0, privs, 0, 0 );
    return status;
}
/***************************************************************************/
/* One-time initialzation, check environment and set static flags to indicate
 * capabilities.
 */
static void user_info_init() {
    int status, length;
    long iosb[2], priv_mask[2], prev_privs[2];
    char version[64];
    item_list item[5];
    /*
     * Initialize uai_version flag to indicate whether this version of VMS
     * supports the context argument for the GETUAI call (added after 6.1).
     * Keeping the context when allowed save considerable I/O.
     */
    uai_version = 1; context = -1;		/* default value */
    upd_context = -1;
    item[0].code = SYI$_NODE_SWVERS;
    item[0].length = 8;
    item[0].buffer = version;
    item[0].ret_len = &length;
    item[1].code = item[1].length = 0;
    length = 0;
    status = SYS$GETSYIW ( 0, 0, 0, item, iosb, 0, 0 );
    if ( status&1 ) status = iosb[0];
    if ( status&1 ) {
	version[length] = '\0';
	if ( tu_strncmp ( version, "V6.1", 8 ) >= 0 ) uai_version = 2;
    }
    /*
     * Check privileges and set flag to determine method of invoking
     * $GETUAI: 1-priority, 2-cmexec.
     */
    cmexec_serialization = 0;			/* default to old method */
    priv_mask[1] = 0; priv_mask[0] = PRV$M_CMEXEC;
    status = SYS$SETPRV ( 1, priv_mask, 0, prev_privs );
    if ( (status == SS$_NORMAL) && (uai_version==2) ) cmexec_serialization = 1;

    /* evr_event ( EVR_CONFIG, 
	"GETUAI calls will %s context and %s serialzation.",
	uai_version==2 ? "use persistent" : "NOT use persistent",
	cmexec_serialization ? "CMEXEC" : "setscheduler" ); */
}
/***************************************************************************/
/* Primary function for returning username info.  Return even for error
 * and odd (1) for success.
 */
int ssh_get_user_information ( 
	char *username, int user_len,		/* username to lookup */
	struct user_account_info *info )	/* Returns data */
{
    int i, flags, devlen, dirlen, status, length;
    int delta, priority, scheduler;
    item_list item[12];
    long *ctxptr;
    pthread_t self;
    char device[40], defdir[64], userkey[USERNAME_MAX+1];
    $DESCRIPTOR (username_dx,"");    
    struct sched_param param;
    /*
     * Determine if safe to use context argument.
     */
    pthread_once ( &uaf_setup, user_info_init );
    ctxptr = (uai_version == 2) ? &context : (long *) 0;
    /*
     * Build upcased descriptor for username.
     */
    if ( user_len > USERNAME_MAX ) return 20;	/* invalid name */
    if ( user_len >= sizeof(info->username) ) return 20;
    if ( user_len <= 0 ) return 20;			/* Null username */

    tu_strnzcpy ( info->username, username, user_len );
    tu_strupcase ( userkey, info->username );
    username_dx.dsc$w_length = user_len;
    username_dx.dsc$a_pointer = userkey;
    info->password[0] = '\0';
    /*
     * Make item list for getuai, most data loads directly into info structure.
     */
    item[0].code = UAI$_FLAGS; item[0].length = sizeof(flags);
    item[0].buffer = (char *) &info->flags; item[0].ret_len = (int *) 0;

    item[1].code = UAI$_DEFDEV; item[1].length = sizeof(device);
    item[1].buffer = device; item[1].ret_len = &devlen; 
    devlen = 0; device[0] = '\0';

    item[2].code = UAI$_DEFDIR; item[2].length = sizeof(defdir);
    item[2].buffer = defdir; item[2].ret_len = &dirlen; 
    dirlen = 0; defdir[0] = '\0';

    item[3].code = UAI$_UIC; item[3].length = sizeof(int);
    item[3].buffer = (char *) &info->uic; item[3].ret_len = (int *) 0;

    item[4].code = UAI$_ENCRYPT; item[4].length = 1;
    item[4].buffer = (char *) &info->algorithm; item[4].ret_len = (int *) 0;
    info->algorithm = 0;

    item[5].code = UAI$_SALT; item[5].length = 2;
    item[5].buffer = (char *) &info->salt; item[5].ret_len = (int *) 0;
    info->salt = 0;

    item[6].code = UAI$_PWD; item[6].length = 8;
    item[6].buffer = (char *) info->hash; item[6].ret_len = (int *) 0;
    info->hash[0] = 0;  info->hash[1] = 0;

    item[7].code = UAI$_LOGFAILS; item[7].length = 2;
    item[7].buffer = (char *) &info->fails; item[7].ret_len = (int *) 0;
    info->fails = 0;

    item[8].code = UAI$_LASTLOGIN_I; item[8].length = 8;
    item[8].buffer = (char *) info->last_ilogin; item[8].ret_len = (int *) 0;

    item[9].code = UAI$_LASTLOGIN_N; item[9].length = 8;
    item[9].buffer = (char *) info->last_nlogin; item[9].ret_len = (int *) 0;

    /*item[10].code = UAI$_MAIL; item[10].length = 2;
    item[10].buffer = (int *) &info->newmail; item[10].ret_len = (int *) 0; */

    item[10].code = 0; item[10].length = 0;	/* terminate list */
    /*
     * Fetch information abour user.  Block other activity and set privilege.
     */
    if ( cmexec_serialization ) {
	/*
	 * Do getuai call from exec mode to ensure no other threads get
	 * benefit of privilege elevation.
	 */
	long arg_list[4] = { 3, 0, 0, 0 };
	arg_list[1] = (long) ctxptr;
	arg_list[2] = (long) &username_dx;
	arg_list[3] = (long) item;
	status = SYS$CMEXEC ( do_getuai, arg_list );
    } else {
	/*
	 * Don't have cmexec, next best thing is to raise priority to FIFO
	 * level to block as many alternate threads as possible from executing.
	 */
	self = pthread_self();
#ifdef PTHREAD_USE_D4
	priority = pthread_getprio ( self );
	scheduler = pthread_getscheduler ( self );
	status = pthread_setscheduler ( self, SCHED_FIFO, PRI_FIFO_MAX );
#else
	pthread_getschedparam ( self, &scheduler, &param );  /* save current */
	priority = param.sched_priority; 
	param.sched_priority = PRI_FIFO_MAX;
	status = pthread_setschedparam ( self, SCHED_FIFO, &param );
#endif
	pthread_lock_global_np();
	status = do_getuai ( ctxptr, &username_dx, item );
	pthread_unlock_global_np();
#ifdef PTHREAD_USE_D4
	i = pthread_setscheduler ( self, scheduler, priority );
#else
	param.sched_priority = priority;
	i = pthread_setschedparam ( self, scheduler, &param );
#endif
    }
    /*
     * Return if error or account not enabled.
     */
    if ( (status&1) == 0 ) return status;		/* GETUAI error */
    if ( (info->flags&UAI$M_DISACNT) != 0 ) return 20;	/* Disusered account */
    /*
     * Fixup punctuation for conversion to unix filepath format.
     */
    devlen = device[0];		/* counted ascii */
    if ( device[devlen] == ':' ) devlen = devlen - 1;	/* trim colon */
    dirlen = defdir[0];
    if ( (defdir[dirlen] == ']') || (defdir[dirlen] == '>') ) {
	defdir[1] = '.';	/* Convert leading bracket */
	defdir[dirlen] = '.';	/* Convert trailing bracket */
    }
    for ( i = 1; i <= dirlen; i++ ) if ( defdir[i] == '.' ) defdir[i] = '/';
    /*
     * Load into users output.
     */
    if ( (devlen + dirlen + 2) > sizeof(info->login_dir)) return 20;
    info->login_dir[0] = '/';
    tu_strncpy ( &info->login_dir[1], &device[1], devlen );
    tu_strnzcpy ( &info->login_dir[1+devlen], &defdir[1], dirlen );

    return 1;
}
/******************************************************************************/
/* Increment the fails parameter uai block and update corresponding UAF record.
 */
int ssh_update_user_logfails ( 
	struct user_account_info *info )	/* Returns data */
{
    int i, flags, status, length;
    int delta, priority, scheduler;
    item_list item[2];
    long *ctxptr;
    pthread_t self;
    char userkey[USERNAME_MAX+1];
    $DESCRIPTOR (username_dx,"");    
    struct sched_param param;
    /*
     * Determine if safe to use context argument.
     */
    pthread_once ( &uaf_setup, user_info_init );
    ctxptr = (uai_version == 2) ? &upd_context : (long *) 0;
    /*
     * Build upcased descriptor for username.
     */
    length = tu_strlen ( info->username );
    if ( length >= sizeof(userkey) ) return 20;
    if ( length <= 0 ) return 20;			/* Null username */

    tu_strupcase ( userkey, info->username );
    username_dx.dsc$w_length = length;
    username_dx.dsc$a_pointer = userkey;
    /*
     * Make item list for setuai, most data loads directly into info structure.
     */
    item[0].length = 2;
    item[0].code = UAI$_LOGFAILS;
    item[0].buffer = (char *) &info->fails;
    item[0].ret_len = (int *) 0;
    item[1].code = 0; item[1].length = 0;	/* terminate list */
    /*
     * Block other activity and update UAF via setuai.
     */
    info->fails++;			/* update the fails count */
    if ( cmexec_serialization ) {
	/*
	 * Do getuai call from exec mode to ensure no other threads get
	 * benefit of privilege elevation.
	 */
	long arg_list[4] = { 3, 0, 0, 0 };
	arg_list[1] = (long) ctxptr;
	arg_list[2] = (long) &username_dx;
	arg_list[3] = (long) item;
	status = SYS$CMEXEC ( do_setuai, arg_list );
    } else {
	/*
	 * Don't have cmexec, next best thing is to raise priority to FIFO
	 * level to block as many alternate threads as possible from executing.
	 */
	self = pthread_self();
#ifdef PTHREAD_USE_D4
	priority = pthread_getprio ( self );
	scheduler = pthread_getscheduler ( self );
	status = pthread_setscheduler ( self, SCHED_FIFO, PRI_FIFO_MAX );
#else
	pthread_getschedparam ( self, &scheduler, &param );  /* save current */
	priority = param.sched_priority; 
	param.sched_priority = PRI_FIFO_MAX;
	status = pthread_setschedparam ( self, SCHED_FIFO, &param );
#endif
	pthread_lock_global_np();
	status = do_setuai ( ctxptr, &username_dx, item );
	pthread_unlock_global_np();
#ifdef PTHREAD_USE_D4
	i = pthread_setscheduler ( self, scheduler, priority );
#else
	param.sched_priority = priority;
	i = pthread_setschedparam ( self, scheduler, &param );
#endif
    }
    return status;
}
