/** EMULib Emulation Library *********************************/
/**                                                         **/
/**                        LibUnix.c                        **/
/**                                                         **/
/** This file contains implementation for some commonly     **/
/** used Unix/X11 routines.                                 **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1996-1998                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/
#ifdef UNIX

#ifdef VMS
/* #include "UNIX_TIME.H" */
#include <time.h>
#define ITIMER_REAL     0
#endif

#include "LibUnix.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifdef MITSHM
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#endif

#define NEXT(N) ((N+1)%MAXALLOC)  /* Used when allocating colors */

struct Timer
{
  struct Timer *Next;
  int Freq;
  void (*Handler)(void);
};

static Display *Dsp      = NULL;
static Screen *Scr       = NULL;
static Colormap CMap;
static unsigned long Black;

static unsigned long ColorPen[MAXALLOC];
static int ColorQueue[MAXALLOC];
static int ColorValue[MAXALLOC];
static int ColorCount[MAXALLOC];
static int AllocLo   = 0;
static int AllocHi   = 1;

struct Timer *FirstTimer = NULL;
static int TimerLock     = 0;

static void MasterTimer(int Arg);

/** InitLibUnix() ********************************************/
/** This function is called to obtain the display and other **/
/** values at startup. It returns 0 on failure.             **/
/*************************************************************/
int InitLibUnix(Display *AppDisplay)
{
  int J;

  /* X11 display */
  Dsp=AppDisplay;
  Scr=DefaultScreenOfDisplay(Dsp);
  CMap=DefaultColormapOfScreen(Scr);
  Black=BlackPixelOfScreen(Scr);

  /* Color allocation */
  for(J=0;J<MAXALLOC;J++) ColorCount[J]=0;
  AllocLo=0;AllocHi=1;

  /* Timers */  
  FirstTimer=NULL;
  TimerLock=0;

  return(1);
}

/** MasterTimer() ********************************************/
/** The main timer handler which is called MAXTIMERFREQ     **/
/** times a second. It then calls user-defined timers.      **/
/*************************************************************/
static void MasterTimer(int Arg)
{
  static unsigned int Counter=0;
  register struct Timer *P;

  if(!TimerLock)
    for(P=FirstTimer;P;P=P->Next)
      if(!(Counter%P->Freq)) (P->Handler)();

  Counter++;
  signal(Arg,MasterTimer);
}

/** X11PutImage() ********************************************/
/** Copy image to a window, centering it.                   **/
/*************************************************************/
void X11PutImage(Window Wnd,Image *Img,int DX,int DY,int SX,int SY,int W,int H)
{
  /* Need to initalize library first */
  if(!Dsp) return;
  
  /* If image not initialized, fall out */
  if(!Img->Data) return;
  
  /* If incorrect parameters, fall out */
  if((SX+W>Img->W)||(SY+H>Img->H)) return;

#ifdef MITSHM
  if(Img->Attrs&USE_SHM)
  {
    XShmPutImage
    (
      Dsp,Wnd,DefaultGCOfScreen(Scr),Img->XImg,
      SX,SY,DX,DY,W,H,False
    );
    if(Img->NeedRMID)
    { shmctl(Img->SHMInfo.shmid,IPC_RMID,NULL);Img->NeedRMID=0; }
  }
  else
#endif
    XPutImage
    (
      Dsp,Wnd,DefaultGCOfScreen(Scr),Img->XImg,
      SX,SY,DX,DY,W,H
    );

  XFlush(Dsp);
}

/** X11NewImage() ********************************************/
/** Build a new Image structure of given dimensions.        **/
/*************************************************************/
int X11NewImage(Image *Img,int W,int H,int Attrs)
{
  XVisualInfo VInfo;
  int Depth,J,I;

  /* Set data fields to ther defaults */
  Img->Data=NULL;Img->W=Img->H=Img->D=Img->Attrs=0;

  /* Need to initalize library first */
  if(!Dsp) return(0);

  /* Image depth we are going to use */
#ifdef BPP32
  Depth=32;
#else
#ifdef BPP24
  Depth=24;
#else
#ifdef BPP16
  Depth=16;
#else
#ifdef BPP8
  Depth=8;
#else
  Depth=DefaultDepthOfScreen(Scr);
#endif
#endif
#endif
#endif
printf ("(Depth %dbpp) ",Depth);

  /* Get appropriate Visual */
  I=XScreenNumberOfScreen(Scr);
  for(J=7;J>=0;J--)
    if(XMatchVisualInfo(Dsp,I,Depth,J,&VInfo)) break;
  if(J<0) return(0);

#ifdef MITSHM
  if(Attrs&USE_SHM)
  {
    Img->XImg =
      XShmCreateImage
      (Dsp,VInfo.visual,Depth,ZPixmap,NULL,&Img->SHMInfo,W,H);
    if(!Img->XImg) return(0);

    Img->SHMInfo.shmid =
      shmget
      (IPC_PRIVATE,Img->XImg->bytes_per_line*Img->XImg->height,IPC_CREAT|0777);
    if(Img->SHMInfo.shmid==-1) return(0);

    Img->XImg->data=Img->SHMInfo.shmaddr=shmat(Img->SHMInfo.shmid,0,0);
    if(!Img->XImg->data) return(0);

    Img->SHMInfo.readOnly=False;
    Img->NeedRMID=1;

    if(!XShmAttach(Dsp,&Img->SHMInfo)) return(0);
  }
  else
#endif
  {
    Img->XImg =
      XCreateImage
      (Dsp,VInfo.visual,Depth,ZPixmap,0,NULL,W,H,Depth,0);
    if(!Img->XImg) return(0);

    Img->XImg->data=malloc(Img->XImg->bytes_per_line*Img->XImg->height);
    if(!Img->XImg->data) return(0);
  }

  Img->Data=Img->XImg->data;
  Img->W=W;Img->H=H;
  Img->D=Depth;
  Img->Attrs=Attrs;
  return(1);
}

/** X11FreeImage() *******************************************/
/** Free Image structure allocated by X11NewImage.          **/
/*************************************************************/
void X11FreeImage(Image *Img)
{
  /* Need to initalize library first */
  if(!Dsp) return;

#ifdef MITSHM
  if(Img->Attrs&USE_SHM)
    if(Img->SHMInfo.shmaddr)
    { XShmDetach(Dsp,&Img->SHMInfo);shmdt(Img->SHMInfo.shmaddr); }
#endif MITSHM
  if(Img->XImg) XDestroyImage(Img->XImg);
}

/** X11Window() **********************************************/
/** Open a window of a given size with a given title.       **/
/*************************************************************/
Window X11Window(char *Title,int W,int H)
{
  XSetWindowAttributes Attrs;
  XSizeHints Hints;
  XWMHints WMHints;
  Window Wnd;

  /* Need to initalize library first */
  if(!Dsp) return(0);

  /* Set necessary attributes */
  Attrs.event_mask=FocusChangeMask|KeyPressMask|KeyReleaseMask;
  Attrs.background_pixel=BlackPixelOfScreen(Scr);
  Attrs.backing_store=Always;

  /* Create a window */
  Wnd=XCreateWindow
      (
        Dsp,RootWindowOfScreen(Scr),0,0,W,H,0,
        CopyFromParent,CopyFromParent,CopyFromParent,
        CWBackPixel|CWEventMask|CWBackingStore,&Attrs
      );
  if(!Wnd) return(0);

  /* Set hints */
  Hints.flags=PSize|PMinSize|PMaxSize;
  Hints.min_width=Hints.max_width=Hints.base_width=W;
  Hints.min_height=Hints.max_height=Hints.base_height=H;
  WMHints.input=True;WMHints.flags=InputHint;
  XSetWMHints(Dsp,Wnd,&WMHints);
  XSetWMNormalHints(Dsp,Wnd,&Hints);
  XStoreName(Dsp,Wnd,Title);

  /* Do additional housekeeping and return */
  XMapRaised(Dsp,Wnd);
  XClearWindow(Dsp,Wnd);
  return(Wnd);
}

/** X11GetColor() ********************************************/
/** Allocate a new color. Returns 0 (black color) on        **/
/** failure.                                                **/
/*************************************************************/
unsigned int X11GetColor(unsigned char R,unsigned char G,unsigned char B)
{
  register int I,J;
  unsigned long C;
  XColor Color;

  I=((int)R<<16)|((int)G<<8)|B;
      
  /* If color allocation table full, flush its first third */
  if(AllocHi==AllocLo) X11FlushColors(MAXALLOC*2/3);

  /* First, find if there is a same color allocated */
  for(J=0;J<MAXALLOC;J++)
    if(ColorCount[J]&&(I==ColorValue[J]))
    {
      ColorCount[J]++;        /* Increase color count by 1 */
      ColorQueue[AllocHi]=J;  /* Add color to the queue    */
      AllocHi=NEXT(AllocHi);
      return(ColorPen[J]);    /* Return preallocated pen   */
    }

  /* If failed to match a color, look for a free entry */
  for(J=0;J<MAXALLOC;J++)
    if(!ColorCount[J])
    {
      Color.flags=DoRed|DoGreen|DoBlue;
      Color.red=(int)R<<8;
      Color.green=(int)G<<8;
      Color.blue=(int)B<<8;

      /* Try to allocate a pen */
      if(XAllocColor(Dsp,CMap,&Color))
      {
        ColorPen[J]=Color.pixel;  /* Store pen number          */
        ColorValue[J]=I;          /* Sotre color value         */
        ColorCount[J]=1;          /* Set usage count to 1      */
        ColorQueue[AllocHi]=J;    /* Add color to the queue    */
        AllocHi=NEXT(AllocHi);
        return(Color.pixel);      /* Return new allocated pen  */
      }

      /* There is a free entry in the table, but we are out of pens */
      break;
    }
  
  /* If matching and allocation failed, start flushing queue */
  for(AllocLo=NEXT(AllocLo);AllocLo!=AllocHi;AllocLo=NEXT(AllocLo))
  {
    J=ColorQueue[AllocLo];
    if(!--ColorCount[J])
    {
      /* Free a color pen */
      C=ColorPen[J];
      XFreeColors(Dsp,CMap,&C,1,0);

      Color.flags=DoRed|DoGreen|DoBlue;
      Color.red=(int)R<<8;
      Color.green=(int)G<<8;
      Color.blue=(int)B<<8;

      /* Try to allocate a pen */
      if(XAllocColor(Dsp,CMap,&Color))
      {
        ColorPen[J]=Color.pixel;  /* Store pen number          */
        ColorValue[J]=I;          /* Sotre color value         */
        ColorCount[J]=1;          /* Set usage count to 1      */
        ColorQueue[AllocHi]=J;    /* Add color to the queue    */
        AllocHi=NEXT(AllocHi);
        return(Color.pixel);      /* Return new allocated pen  */
      }
    }
  }

  /* Everything failed. Returning black color. */
  if(AllocLo==AllocHi) AllocHi=NEXT(AllocHi);
  return(Black);
}

/** X11FlushColors() *****************************************/
/** Flush allocated colors, but preserve N last ones.       **/
/*************************************************************/
void X11FlushColors(int N)
{
  register int J;
  unsigned long C;
  
  /* Number of colors to flush */
  N=AllocHi-AllocLo-1-N;
  if(AllocHi<=AllocLo) N+=MAXALLOC;

  /* If nothing to flush, fall out */
  if(N<=0) return;

  /* Decrease usage counters */
  for(AllocLo=NEXT(AllocLo);N;AllocLo=NEXT(AllocLo),N--)
  {
    J=ColorQueue[AllocLo];
    if(!--ColorCount[J])
    { C=ColorPen[J];XFreeColors(Dsp,CMap,&C,1,0); }
  }
}

/** TimerSignal() ********************************************/
/** Establish a signal handler called with given frequency  **/
/** (Hz). Returns 0 if failed.                              **/
/*************************************************************/
int TimerSignal(int Freq,void (*Handler)(int))
{
  struct itimerval TimerValue;

  TimerValue.it_interval.tv_sec  =
  TimerValue.it_value.tv_sec     = 0;
  TimerValue.it_interval.tv_usec =
  TimerValue.it_value.tv_usec    = 1000000L/Freq;
  if(setitimer(ITIMER_REAL,&TimerValue,NULL)) return(0);
  signal(SIGALRM,Handler);
  return(1);
}

/** AddTimer() ***********************************************/
/** Establish a periodically called routine at a given      **/
/** frequency (1..MAXTIMERFREQ Hz). Returns 0 if failed.    **/
/*************************************************************/
int AddTimer(int Freq,void (*Handler)(void))
{
  struct Timer *P;

  /* Freq must be 1..MAXTIMERFREQ */
  if((Freq<1)||(Freq>MAXTIMERFREQ)) return(0);

  /* Lock the timer queue */
  TimerLock=1;

  /* Look if this timer is already installed */
  for(P=FirstTimer;P;P=P->Next)
    if(P->Handler==Handler)
    { P->Freq=Freq;TimerLock=0;return(1); }

  /* Make up a new one if not */
  if(FirstTimer)
  {
    for(P=FirstTimer;P->Next;P=P->Next);
    P->Next=malloc(sizeof(struct Timer));
    P=P->Next;
  }
  else
  {
    /* Allocate the first timer */
    FirstTimer=malloc(sizeof(struct Timer));
    P=FirstTimer;

    /* Start up the main handler */
    if(P)
      if(!TimerSignal(MAXTIMERFREQ,MasterTimer))
      { free(P);P=NULL; }
  }

  /* Set up the timer and exit */
  if(P) { P->Next=0;P->Handler=Handler;P->Freq=Freq; }
  TimerLock=0;
  return(P!=0);
}

/** DelTimer() ***********************************************/
/** Remove routine added with AddTimer().                   **/
/*************************************************************/
void DelTimer(void (*Handler)(void))
{
  struct itimerval TimerValue;
  struct Timer *P,*T;

  TimerLock=1;

  if(FirstTimer)
    if(FirstTimer->Handler==Handler)
    {
      /* Delete first timer and free the space */
      P=FirstTimer;FirstTimer=P->Next;free(P);

      /* If no timers left, stop the main handler */
      if(!FirstTimer)
      {
         TimerValue.it_interval.tv_sec  =
         TimerValue.it_value.tv_sec     =
         TimerValue.it_interval.tv_usec =
         TimerValue.it_value.tv_usec    = 0;
         setitimer(ITIMER_REAL,&TimerValue,NULL);
         signal(SIGALRM,SIG_DFL);
      }
    }
  else
  {
    /* Look for our timer */
    for(P=FirstTimer;P;P=P->Next)
      if(P->Next->Handler==Handler) break;

    /* Delete the timer and free the space */
    if(P) { T=P->Next;P->Next=T->Next;free(T); }
  }

  TimerLock=0;
}

#endif /* UNIX */
