// **********************************************************************
//
// Copyright (c) 1999
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#include <OB/CORBA.h>
#include <OB/Util.h>

#include <stdlib.h>
#include <errno.h>

#ifdef HAVE_FSTREAM
#   include <fstream>
#else
#   include <fstream.h>
#endif

static void
usage(const char* progName)
{
    cerr << "Usage:\n";
    cerr << progName << " [options] [-f FILE | IOR ...]\n"
	"\n"
	"Options:\n"
	"-h, --help             Show this message.\n"
	"-v, --version          Show ORBacus version.\n"
        "-f FILE                Read IORs from FILE.\n"
	;
}

static void
DumpIOR(const char* ior)
{
    try
    {
	OBPrintObjref(cout, ior);
    }
    catch(const CORBA_INV_OBJREF&)
    {
	cout << "IOR is invalid" << endl;
    }

    cout << flush;
}

int
main(int argc, char* argv[], char*[])
{
    try
    {
	//
	// Create ORB
	//
	CORBA_ORB_var orb = CORBA_ORB_init(argc, argv);
	
	//
	// Name of file with IOR
	//
	const char* filename = 0;
	
	//
	// Set program name
	//
	const char* progName = argv[0];
	
	//
	// Get options
	//
	CORBA_Long i;
	for(i = 1 ; i < argc && *argv[i] == '-' ; i++)
	{
	    if(strcmp(argv[i], "--help") == 0 ||
	       strcmp(argv[i], "-h") == 0)
	    {
		usage(progName);
		return 0;
	    }
	    else if(strcmp(argv[i], "--version") == 0 ||
		    strcmp(argv[i], "-v") == 0)
	    {
		cerr << "ORBacus " << OBVersion << endl;
		return 0;
	    }
	    else if(strcmp(argv[i], "-f") == 0)
	    {
		if(i + 1 < argc)
		    filename = argv[++i];
		else
		{
		    cerr << progName << ": missing filename\n" << endl;
		    usage(progName);
		    return 1;
		}
	    }
	    else
	    {
		cerr << progName << ": unknown option `"
		     << argv[i] << "'\n" << endl;
		usage(progName);
		return 1;
	    }
	}
	
	//
	// Get repository
	//
	if(!filename && i == argc)
	{
	    cerr << progName << ": no IOR" << endl;
	    usage(progName);
	    return 1;
	}
	
	if(filename == 0)
	{
	    //
	    // Dump all IORs given as arguments
	    //
	    int count = 0;
	    for(; i < argc ; i++)
	    {
		if(count > 0)
		    cout << '\n';
		cout << "IOR #" << ++count << ':' << endl;
		DumpIOR(argv[i]);
	    }
	}
	else
	{
	    //
	    // Dump all IORs from the specified file
	    //
	    ifstream in;
	    // Must use open(name), not ifstream in(name) (VC++ bug)
	    in.open(filename);
	    if(in.fail())
	    {
		cerr << argv[0] << ": can't open `" << filename << "': "
		     << strerror(errno) << endl;
		return 1;
	    }
	    
	    CORBA_String_var str = CORBA_string_dup("");
	    int count = 0;
	    
	    while(!in.eof())
	    {
		char c;
		in.read(&c, 1);
		if(in.eof() || c == '\n' || c == '\r' || c == '\0')
		{
		    if(strlen(str) > 0)
		    {
			if(count > 0)
			    cout << '\n';
			cout << "IOR #" << ++count << ':' << endl;
			DumpIOR(str);
			str = CORBA_string_dup("");
		    }
		}
		else
		    str += c;
	    }
	}
    }
    catch(CORBA_SystemException& ex)
    {
	OBPrintException(ex);
	return 1;
    }

    return 0;
}
