/******************************************************************************
 *
 * Copyright (c) 1998,99 by Mindbright Technology AB, Stockholm, Sweden.
 *                 www.mindbright.se, info@mindbright.se
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *****************************************************************************
 * $Author: mats $
 * $Date: 1999/08/24 18:24:28 $
 * $Name: rel0-99 $
 *****************************************************************************/
package mindbright.application;

import java.io.*;

import java.applet.Applet;

import java.awt.*;
import java.awt.event.*;
import java.util.NoSuchElementException;
import java.util.Properties;
import java.util.Hashtable;
import java.util.Enumeration;

import mindbright.ssh.*;
import mindbright.security.*;
import mindbright.terminal.*;

public class MindTerm extends Applet implements Runnable {

  static Properties paramTermProps = new Properties();
  static Properties paramSSHProps  = new Properties();

  public static String javaVersion = "<unknown>";
  public static String javaVendor  = "<unknown>";
  public static String osName      = "<unknown>";
  public static String osArch      = "<unknown>";
  public static String osVersion   = "<unknown>";

  Frame                frame;
  TerminalWin          term;
  SSHInteractiveClient client;
  SSHInteractiveClient sshClone;
  SSHStdIO             console;
  Thread               clientThread;

  boolean    mergedTermProps;
  Properties sshProps;
  Properties termProps;

  String[]   cmdLineArgs;

  String  commandLine = null;
  String  sshHomeDir  = null;
  String  propsFile   = null;

  boolean usePopMenu = false;
  boolean haveMenus  = true;
  boolean haveGUI    = true;
  boolean cmdsh      = false;
  boolean quiet      = false;

  boolean autoSaveProps = true;
  boolean autoLoadProps = true;

  int     popButtonNum = 3;

  boolean isClosing  = false;

  // !!!
  boolean     separateFrame = true;
  boolean     weAreAnApplet = false;

  static Hashtable terminals = new Hashtable();

  static synchronized boolean isLastTerminal() {
    return terminals.isEmpty();
  }

  static synchronized void addTerminal(MindTerm mindterm) {
    terminals.put(mindterm, mindterm);
  }

  static synchronized void removeTerminal(MindTerm mindterm) {
    terminals.remove(mindterm);
  }

  public MindTerm() {
    super();
    this.sshProps  = paramSSHProps;
    this.termProps = paramTermProps;
    addTerminal(this);
  }

  public MindTerm(Properties sshProps, Properties termProps) {
    this.sshProps  = sshProps;
    this.termProps = termProps;
    addTerminal(this);
  }

  public static void main(String[] argv) {
    MindTerm controller    = new MindTerm(paramSSHProps, paramTermProps);
    controller.cmdLineArgs = argv;
    try {
      controller.getApplicationParams();
      controller.run();
    } catch (Exception e) {
      System.out.println("Error, please mail below stack-trace to mats@mindbright.se");
      e.printStackTrace();
    }
  }

  public void init() {
    if(SSH.NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("TerminalEmulator");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }
    weAreAnApplet  = true;
    autoSaveProps  = false;
    autoLoadProps  = false;
    getAppletParams();
    (new Thread(this)).start();
  }

  public void run() {
    try {
      // Default when running single command is not to allocate a PTY
      //
      if(commandLine != null && sshProps.getProperty("forcpty") == null) {
	sshProps.put("forcpty", "false");
      }

      if(sshClone != null) {
	client = new SSHInteractiveClient(sshClone);
	sshClone = null;
      } else {
	if(propsFile == null) {
	  client = new SSHInteractiveClient(quiet, cmdsh, autoSaveProps, autoLoadProps,
					    sshProps);
	} else {
	  client = new SSHInteractiveClient(quiet, cmdsh, autoSaveProps, autoLoadProps,
					    propsFile);
	  client.setProperties(sshProps, true);
	}
      }
      console = (SSHStdIO)client.getConsole();

      // If we loaded a specific property-file we merge the termProps from there
      //
      if(client.initTermProps != null) {
	Properties newTermProps = new Properties(client.initTermProps);
	if(!termProps.isEmpty()) {
	  Enumeration enum = termProps.propertyNames();
	  while(enum.hasMoreElements()) {
	    String name = (String)enum.nextElement();
	    newTermProps.put(name, termProps.getProperty(name));
	  }
	  mergedTermProps = true;
	}
	termProps = newTermProps;
      }

      // First we initialize the GUI if we have one (to be able to set
      // properties to terminal
      //
      if(haveGUI) {
	initGUI();
	console.setTerminal(term);
	console.setOwnerContainer(frame);
	console.setOwnerName(SSH.VER_MINDTERM);
	console.updateTitle();
	try {
	  while(!frame.isShowing())
	    Thread.sleep(100);
	} catch(InterruptedException e) {
	    // !!!
	}
      }

      client.printCopyright();
      client.setSSHHomeDir(sshHomeDir);

      if(commandLine != null) {
	client.doSingleCommand(commandLine, false, 0);
      } else {
	try {
	  clientThread = new Thread(client);
	  clientThread.start();
	  clientThread.join();
	} catch(InterruptedException e) {
	  // !!!
	}
      }

    } catch (IllegalArgumentException ae) {
      if(client != null)
	client.alert(ae.getMessage());
      System.out.println(ae.getMessage());
    } catch (FileNotFoundException fe) {
      System.out.println("Settings-file not found: " + fe.getMessage());
    } catch (Exception e) {
      if(client != null)
	client.alert("Error: "  + e.getMessage());
      System.out.println("Error: " + e.getMessage());
      if(SSH.DEBUGMORE) {
	System.out.println("Please send the below stack-trace to mats@mindbright.se");
	e.printStackTrace();
      }
    }

    windowClosing(null);
    if(isLastTerminal())
      doExit();
  }

  public void getAppletParams() {
    String    name;
    String    value;
    String    param;
    int       i;

    try {
      separateFrame = (new Boolean(getParameter("sepframe"))).booleanValue();
    } catch (Exception e) {
      separateFrame = true;
    }
    try {
      SSH.DEBUG = (new Boolean(getParameter("verbose"))).booleanValue();
    } catch (Exception e) {
      SSH.DEBUG = false;
    }

    try {
      SSH.DEBUGMORE = (new Boolean(getParameter("debug"))).booleanValue();
      SSH.DEBUG = SSH.DEBUGMORE;
    } catch (Exception e) {
    }

    try {
      quiet = (new Boolean(getParameter("quiet"))).booleanValue();
    } catch (Exception e) {
      quiet = false;
    }

    try {
      cmdsh = (new Boolean(getParameter("cmdsh"))).booleanValue();
    } catch (Exception e) {
      cmdsh = false;
    }

    param = getParameter("menus");
    if(param != null) {
      if(param.equals("no"))
	haveMenus = false;
      else if(param.startsWith("pop")) {
	getPopupButtonNumber(param);
	usePopMenu = true;
      }
    }

    param = getParameter("autoprops");
    if(param != null) {
      if(param.equals("save")) {
	autoSaveProps = true;
	autoLoadProps = false;
      } else if(param.equals("load")) {
	autoSaveProps = false;
	autoLoadProps = true;
      } else if(param.equals("both")) {
	autoSaveProps = true;
	autoLoadProps = true;
      }
    }

    sshHomeDir  = getParameter("sshhome");
    propsFile   = getParameter("propsfile");
    commandLine = getParameter("commandline");

    getDefaultParams();

    for(i = 0; i < SSHInteractiveClient.defaultPropDesc.length; i++) {
      name  = SSHInteractiveClient.defaultPropDesc[i][client.PROP_NAME];
      value = getParameter(name);
      if(value != null)
	paramSSHProps.put(name, value);
    }
    i = 0;
    while((value = getParameter("local" + i)) != null) {
      paramSSHProps.put("local" + i, value);
      i++;
    }
    i = 0;
    while((value = getParameter("remote" + i)) != null) {
      paramSSHProps.put("remote" + i, value);
      i++;
    }

    for(i = 0; i < TerminalWin.defaultPropDesc.length; i++) {
      name  = term.defaultPropDesc[i][term.PROP_NAME];
      value = getParameter(name);
      if(value != null)
	termProps.put(name, value);
    }

  }

  public void getApplicationParams() throws Exception {
    String    name;
    String    value;
    int       numOfOpts;
    int       i;

    // First we check the MindTerm options (i.e. not the ssh/terminal-properties)
    //
    try {
      for(i = 0; i < cmdLineArgs.length; i++) {
	String arg = cmdLineArgs[i];
	if(!arg.startsWith("--"))
	  break;
	switch(arg.charAt(2)) {
	case 'h':
	  sshHomeDir = cmdLineArgs[++i];
	  break;
	case 'f':
	  propsFile = cmdLineArgs[++i];
	  break;
	case 'c':
	  cmdsh = true;
	  break;
	case 'd':
	  haveGUI = false;
	  break;
	case 'm': {
	  String typ = cmdLineArgs[++i];
	  if(typ.equals("no"))
	    haveMenus = false;
	  else if(typ.startsWith("pop")) {
	    getPopupButtonNumber(typ);
	    usePopMenu = true;
	  } else
	    throw new Exception();
	  break;
	}
	case 'p': {
	  String typ = cmdLineArgs[++i];
	  if(typ.equals("save")) {
	    autoSaveProps = true;
	  } else if(typ.equals("load")) {
	    autoLoadProps = true;
	  } else if(typ.equals("both")) {
	    autoSaveProps = true;
	    autoLoadProps = true;
	  } else if(typ.equals("none")) {
	    autoSaveProps = false;
	    autoLoadProps = false;
	  } else
	    throw new Exception();
	  break;
	}
	case 'v':
	  System.out.println("verbose mode selected...");
	  SSH.DEBUG = true;
	  break;
	case 'q':
	  quiet = true;
	  break;
	case 'V':
	  System.out.println("SSH version " + SSH.VER_SSH + ", protocol version " +
			     SSH.VER_MAJOR + "." + SSH.VER_MINOR);
	  System.out.println(SSH.VER_MINDTERM);
	  System.exit(0);
	  break;
	case 'D':
	  SSH.DEBUG     = true;
	  SSH.DEBUGMORE = true;
	  break;
	case '?':
	  printHelp();
	  System.exit(0);
	default:
	  throw new Exception();
	}
      }
    } catch (Exception e) {
      printHelp();
      throw new Exception();
    }

    getDefaultParams();

    numOfOpts = i;
    for(i = numOfOpts; i < cmdLineArgs.length; i += 2) {
      name = cmdLineArgs[i];
      if((name.charAt(0) != '-') || ((i + 1) == cmdLineArgs.length))
	break;
      name  = name.substring(1);
      value = cmdLineArgs[i + 1];
      if(SSHInteractiveClient.isProperty(name))
	paramSSHProps.put(name, value);
      else if(TerminalWin.isProperty(name))
	paramTermProps.put(name, value);
      else
	System.out.println("Unknown property '" + name + "'");
    }

    if(i < cmdLineArgs.length) {
      commandLine = "";
      for(; i < cmdLineArgs.length; i++) {
	commandLine += cmdLineArgs[i] + " ";
      }
    }
  }

  void printHelp() {
    System.out.println("usage: MindTerm [options] [properties] [command]");
    System.out.println("Options:");
    System.out.println("  --c            Enable local command-shell.");
    System.out.println("  --d            No terminal-window, only dumb command-line and port-forwarding.");
    System.out.println("  --f <file>     Use settings from the given file.");
    System.out.println("  --h dir        Name of the MindTerm home-dir (default: ~/mindterm/).");
    System.out.println("  --m <no | pop | popN>");
    System.out.println("                  Use no menus or popup (on mouse-button N) menu instead of menubar.");
    System.out.println("  --p <save | load | both | none>");
    System.out.println("                 Sets automatic save/load flags for property-files.");
    System.out.println("  --q            Quiet; don't query for server/username if given.");
    System.out.println("  --v            Verbose; display verbose messages.");
    System.out.println("  --D            Debug; display extra debug info.");
    System.out.println("  --V            Version; display version number only.");
    System.out.println("  --?            Help; display this help.");
  }

  void getPopupButtonNumber(String param) {
    if(param.length() == 4) {
      try {
	popButtonNum = Integer.valueOf(param.substring(3)).intValue();
	if(popButtonNum < 1 || popButtonNum > 3)
	  popButtonNum = 3;
      } catch (NumberFormatException e) {
	// !!!
      }
    }
  }

  void getDefaultParams() {
    try {
      if(sshHomeDir == null) {
	String hDir = System.getProperty("user.home");
	if(hDir == null)
	  hDir = System.getProperty("user.dir");
	if(hDir == null)
	  hDir = System.getProperty("java.home");
	sshHomeDir = (hDir + File.separator + "mindterm" + File.separator);
      }
    } catch (Throwable t) {
      // !!!
    }
    if(weAreAnApplet)
      paramSSHProps.put("server", getCodeBase().getHost());
    try {
      if(!quiet)
	paramSSHProps.put("usrname", System.getProperty("user.name", ""));
    } catch (Throwable t) {
      // !!!
    }
    try {
      javaVersion = System.getProperty("java.version");
      javaVendor  = System.getProperty("java.vendor");
      osName      = System.getProperty("os.name");
      osArch      = System.getProperty("os.arch");
      osVersion   = System.getProperty("os.version");
    } catch (Throwable t) {
      // !!!
    }

  }

  public void initGUI() {
    Container container;
    MenuBar   menubar = null;
    if(separateFrame) {
      frame = new Frame();
      frame.addWindowListener(new WindowAdapter() {
	public void windowClosing(WindowEvent e)  { MindTerm.this.windowClosing(e); }
	public void windowDeiconified(WindowEvent e)  { term.requestFocus(); }
      });
      container = frame;
      if(haveMenus && !usePopMenu) {
	menubar = new MenuBar();
	frame.setMenuBar(menubar);
	if(javaVersion != null && (javaVersion.indexOf("1.2") != -1)) {
	  frame.pack();
	  frame.validate();
	}
      }
    } else {
      Component comp = this;
      do {
	comp = comp.getParent();
      } while(!(comp instanceof Frame));
      frame = (Frame)comp;
      container = this;
    }

    term  = new TerminalWin(frame, new TerminalXTerm(), termProps);

    if(mergedTermProps)
      term.propsChanged = true;

    container.add(term.getPanelWithScrollbar());

    if(haveMenus) {
      SSHMenuHandler      menus  = new SSHMenuHandler(this, client, frame, term);
      TerminalMenuHandler tmenus = new TerminalMenuHandler(term);
      tmenus.setTerminalMenuListener(menus);
      client.setMenus(menus);
      term.setMenus(tmenus);
      if(menubar == null) {
	PopupMenu popupmenu = term.getPopupMenu("MindTerm Menu");
	menus.preparePopupMenu(popupmenu);
	menus.setPopupButton(popButtonNum);
      } else {
	menus.prepareMenuBar(menubar);
      }
    }

    frame.pack();
    frame.show();

    term.requestFocus();
  }

  public synchronized void windowClosing(WindowEvent e) {
    if(isClosing)
      return;
    isClosing = true;

    if(!confirmClose()) {
      isClosing = false;
      return;
    }

    if(separateFrame && haveGUI && frame != null) {
      frame.dispose();
    }

    if(clientThread != null && clientThread.isAlive())
      clientThread.stop();

    removeTerminal(this);
  }

  public void doExit() {
      System.out.println("Thank you for using MindTerm...");
      if(!separateFrame && term != null) {
	  int i;
	  try {
	      term.cursorSetPos(term.rows() - 1, 0, false);
	      for(i = 0; i < term.rows(); i++) {
		  term.write("\n\r");
		  Thread.sleep(50);
	      }
	      term.cursorSetPos(0, 0, false);
	      for(i = 0; i < term.rows() - 1; i++) {
		  term.write(".\n\r");
		  Thread.sleep(50);
	      }
	      term.write("Thank you for using MindTerm...");
	      for(i = 0; i < term.rows() - 1; i++) {
		  term.write("\n\r");
		  Thread.sleep(50);
	      }
	      term.cursorSetPos(2, 0, false);
	      term.setAttribute(TerminalWin.ATTR_BOLD, true);
	      term.write("Visit <http://www.mindbright.se/mindterm> for more information.");
	      term.cursorSetPos(term.rows() - 1, term.cols() - 1, false);
	  } catch (Exception ee) {
	      // !!!
	  }
      }
      if(SSH.secureRandom != null && SSH.secureRandom().updater != null && SSH.secureRandom().updater.isAlive())
	SSH.secureRandom().updater.stop();
      if(!weAreAnApplet) {
	System.exit(0);
      }
  }

  boolean confirmedClose = false;
  boolean confirmClose() {
    if(client != null && !confirmedClose) {
      try {
	client.checkSave();
      } catch(IOException ee) {
	client.alert("Error saving settings: " + ee.getMessage());
      }
      if(client.isOpened() &&
	  !client.askConfirmation("Do you really want to disconnect from " + client.getProperty("server") + "?"))
	confirmedClose = false;
      else
	confirmedClose = true;
    }
    return confirmedClose;
  }

  void initParams(MindTerm mindterm) {
    this.sshHomeDir     = mindterm.sshHomeDir;
    this.propsFile      = mindterm.propsFile;
    this.usePopMenu     = mindterm.usePopMenu;
    this.haveMenus      = mindterm.haveMenus;
    this.haveGUI        = mindterm.haveGUI;
    this.cmdsh          = mindterm.cmdsh;
    this.quiet          = mindterm.quiet;
    this.separateFrame  = true;
    this.autoLoadProps  = mindterm.autoLoadProps;
    this.popButtonNum   = mindterm.popButtonNum;
  }

  public void cloneWindow() {
    MindTerm mindterm = new MindTerm(client.getProperties(), term.getProperties());
    mindterm.initParams(this);
    mindterm.sshClone = this.client;
    (new Thread(mindterm)).start();
  }

  public void newWindow() {
    MindTerm mindterm = new MindTerm(paramSSHProps, paramTermProps);
    mindterm.initParams(this);
    (new Thread(mindterm)).start();
  }

  public void close() {
    if(!confirmClose())
      return;
    clientThread.stop();
  }

  public void exit() {
    if(!confirmClose())
      return;
    Enumeration enum = terminals.elements();
    while(enum.hasMoreElements()) {
      MindTerm mt = (MindTerm)enum.nextElement();
      if(mt.clientThread != null) {
	mt.clientThread.stop();
      }
    }
  }

}

