#define programName     "fastDNAml"
#define programVersion  "1.0.8"
#define programDate     "April 30, 1993"
/* #define WIN_MAC */   /* must be uncommented for MacIntosh machines */


#    define Sequential 1
#    define Master 0
#    define Slave 0
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>

#ifdef UNIX
#include <sys/types.h>
#include <unistd.h>
#endif

#include "mlpw.h"  /*  Requires version 1.0  */

extern int Gafter;
extern int Gduring;
extern double ttrat;



/*  Global variables */

longer  seed;      /*  random number seed with 6 bits per element */

xarray
    *usedxtip, *freextip;

double
    rate[maxcategories+1],    /*  rates for categories  */
    *ratvalue,		      /*  rates per site */
    *wgt_rate,		      /*  weighted rate per site */
    *wgt_rate2;		      /*  weighted rate**2 per site */

double
    xi, xv, ttratio,          /*  transition/transversion info */
    freqa, freqc, freqg, freqt,  /*  base frequencies */
    freqr, freqy, invfreqr, invfreqy,
    freqar, freqcy, freqgr, freqty,
    totalwrate,    /*  total of weighted rate values */
    fracchange;    /*  random matching fraquency (in a sense) */

int
    *alias,        /*  site corresponding to pattern */
    *aliasweight,  /*  weight of given pattern */
    *category,     /*  rate category of sequence positions */
    *catnumb,      /*  category number by pattern number */
    *weight;       /*  weight of sequence positions */

int
    categs,        /*  number of rate categories */
    endsite,       /*  number of unique sequence patterns */
    extrainfo,     /*  runtime information switch */
    numsp,         /*  number of species (same as tr->mxtips) */
    nkeep,         /*  number of best trees to keep */
    sites,         /*  number of input sequence positions */
    trout,         /*  write tree to "treefile" */
    weightsum;     /*  sum of weights of positions in analysis */

boolean
    anerror,       /*  error flag */
    bootstrap,     /*  do a set of bootstrap column weights */
    freqsfrom,     /*  use empirical base frequencies */
    interleaved,   /*  input data are in interleaved format */
    jumble,        /*  use random addition order */
    outgropt,      /*  use user-supplied outgroup */
    printdata,     /*  echo data to output stream */
    fastadd,       /*  test addition sites without global smoothing */
    restart,       /*  continue sequential addition to partial tree */
    treeprint;     /*  print tree to output stream */

char
    *y[maxsp+1];   /*  sequence data array */



/*=======================================================================*/
/*                              PROGRAM                                  */
/*=======================================================================*/
/*                    Best tree handling for dnaml                       */
/*=======================================================================*/

/*  Tip value comparisons
 *
 *  Use void pointers to hide type from other routines.  Only tipValPtr and
 *  cmpTipVal need to be changed to alter the nature of the values compared
 *  (e.g., names instead of node numbers).
 *
 *    cmpTipVal(tipValPtr(nodeptr p), tipValPtr(nodeptr q)) == -1, 0 or 1.
 *
 *  This provides direct comparison of tip values (for example, for
 *  definition of tr->start).
 */

static void newview(nodeptr p);

static void hookup (nodeptr p, nodeptr q, double z)
   { /* hookup */
    p->back = q;
    q->back = p;
    p->z = q->z = z;
  } /* hookup */

static void initrav (nodeptr p)
  { /* initrav */
    if (! p->tip) {
      initrav(p->next->back);
      initrav(p->next->next->back);
      newview(p);
      }
  } /* initrav */


static void  *tipValPtr (nodeptr p) { return  (void *) & p->number; }


static int  cmpTipVal (void* v1, void* v2)
  { /* cmpTipVal */
    int  i1, i2;

    i1 = *((int *) v1);
    i2 = *((int *) v2);
    return  (i1 < i2) ? -1 : ((i1 == i2) ? 0 : 1);
  } /* cmpTipVal */


/*  These are the only routines that need to UNDERSTAND topologies */


static topol  *setupTopol (int maxtips, int nsites)
  { /* setupTopol */
    topol   *tpl;

      if (! (tpl = (topol *) malloc((unsigned) sizeof(topol))) || 
        ! (tpl->links = (connptr) malloc((unsigned) ((2 * maxtips - 3) *
                                                     sizeof(connect)))) || 
        (nsites && ! (tpl->log_f
                = (double *) malloc((unsigned) (nsites * sizeof(double)))))) {
      (void) printf("ERROR: Unable to get topology memory");
      tpl = (topol *) NULL;
      }
     

    else {
      if (nsites == 0)  tpl->log_f = (double *) NULL;
      tpl->likelihood  = unlikely;
      tpl->start       = (node *) NULL;
      tpl->nextlink    = 0;
      tpl->ntips       = 0;
      tpl->nextnode    = 0;
      tpl->opt_level   = 0;     /* degree of branch swapping explored */
      tpl->scrNum      = 0;     /* position in sorted list of scores */
      tpl->tplNum      = 0;     /* position in sorted list of trees */
      tpl->log_f_valid = 0;     /* log_f value sites */
      tpl->smoothed    = FALSE; /* branch optimization converged? */
    }

    return  tpl;
  } /* setupTopol */


static void  freeTopol (topol* tpl)
  { /* freeTopol */
    free((char *) tpl->links);
    if (tpl->log_f)  free((char *) tpl->log_f);
    free((char *) tpl);
  } /* freeTopol */


static int  saveSubtree (nodeptr p, topol* tpl)
    /*  Save a subtree in a standard order so that earlier branches
     *  from a node contain lower value tips than do second branches from
     *  the node.  This code works with arbitrary furcations in the tree.
     */
  { /* saveSubtree */
    connptr  r, r0;
    nodeptr  q, s;
    int      t, t0, t1;

    r0 = tpl->links;
    r = r0 + (tpl->nextlink)++;
    r->p = p;
    r->q = q = p->back;
    r->z = p->z;
    r->descend = 0;                     /* No children (yet) */

    if (q->tip) {
      r->valptr = tipValPtr(q);         /* Assign value */
      }

    else {                              /* Internal node, look at children */
      s = q->next;                      /* First child */
      do {
        t = saveSubtree(s, tpl);        /* Generate child's subtree */

        t0 = 0;                         /* Merge child into list */
        t1 = r->descend;
        while (t1 && (cmpTipVal(r0[t1].valptr, r0[t].valptr) < 0)) {
          t0 = t1;
          t1 = r0[t1].sibling;
          }
        if (t0) r0[t0].sibling = t;  else  r->descend = t;
        r0[t].sibling = t1;

        s = s->next;                    /* Next child */
        } while (s != q);

      r->valptr = r0[r->descend].valptr;   /* Inherit first child's value */
      }                                 /* End of internal node processing */

    return  r - r0;
  } /* saveSubtree */


static nodeptr  minSubtreeTip (nodeptr p0)
  { /* minTreeTip */
    nodeptr  minTip, p, testTip;

    if (p0->tip) return p0;

    p = p0->next;
    minTip = minSubtreeTip(p->back);
    while ((p = p->next) != p0) {
      testTip = minSubtreeTip(p->back);
      if (cmpTipVal(tipValPtr(testTip), tipValPtr(minTip)) < 0)
        minTip = testTip;
      }
    return minTip;
  } /* minTreeTip */


static nodeptr  minTreeTip (nodeptr p)
  { /* minTreeTip */
    nodeptr  minp, minpb;

    minp  = minSubtreeTip(p);
    minpb = minSubtreeTip(p->back);
    return cmpTipVal(tipValPtr(minp), tipValPtr(minpb)) < 0 ? minp : minpb;
  } /* minTreeTip */


static void saveTree (tree* tr, topol* tpl)
    /*  Save a tree topology in a standard order so that first branches
     *  from a node contain lower value tips than do second branches from
     *  the node.  The root tip should have the lowest value of all.
     */
  { /* saveTree */
    connptr  r;
    double  *tr_log_f, *tpl_log_f;
    int  i;

    tpl->nextlink = 0;                             /* Reset link pointer */
    r = tpl->links + saveSubtree(minTreeTip(tr->start), tpl);  /* Save tree */
    r->sibling = 0;

    tpl->likelihood = tr->likelihood;
    tpl->start      = tr->start;
    tpl->ntips      = tr->ntips;
    tpl->nextnode   = tr->nextnode;
    tpl->opt_level  = tr->opt_level;
    tpl->smoothed   = tr->smoothed;

    tpl_log_f = tpl->log_f;
    if (tpl_log_f) {
      tr_log_f  = tr->log_f;
      i = tpl->log_f_valid = tr->log_f_valid;
      while (--i >= 0)  *tpl_log_f++ = *tr_log_f++;
      }
    else {
      tpl->log_f_valid = 0;
      }
  } /* saveTree */


static void restoreTree (topol* tpl, tree* tr)
  { /* restoreTree */
    /*void  hookup();
    void  initrav();*/

    connptr  r;
    nodeptr  p, p0;
    double  *tr_log_f, *tpl_log_f;
    int  i;

/*  Clear existing connections */

    for (i = 1; i <= 2*(tr->mxtips) - 2; i++) {  /* Uses p = p->next at tip */
      p0 = p = tr->nodep[i];
      do {
        p->back = (nodeptr) NULL;
        p = p->next;
        } while (p != p0);
      }

/*  Copy connections from topology */

    for (r = tpl->links, i = 0; i < tpl->nextlink; r++, i++) {
      hookup(r->p, r->q, r->z);
      }

    tr->likelihood = tpl->likelihood;
    tr->start      = tpl->start;
    tr->ntips      = tpl->ntips;
    tr->nextnode   = tpl->nextnode;
    tr->opt_level  = tpl->opt_level;
    tr->smoothed   = tpl->smoothed;

    tpl_log_f = tpl->log_f;
    if (tpl_log_f) {
      tr_log_f = tr->log_f;
      i = tr->log_f_valid = tpl->log_f_valid;
      while (--i >= 0)  *tr_log_f++ = *tpl_log_f++;
      }
    else {
      tr->log_f_valid = 0;
      }

    initrav(tr->start);
    initrav(tr->start->back);
  } /* restoreTree */


static int initBestTree (bestlist* bt, int newKeep, int nsites)
  { /* initBestTree */
    int  i, nlogf;

    if (newKeep < 1) newKeep = 1;
    if (newKeep > maxkeep) newKeep = maxkeep;

    bt->best     = unlikely;
    bt->worst    = unlikely;
    bt->nvalid   = 0;
    bt->ninit    = -1;
    bt->numtrees = 0;
    bt->improved = FALSE;
   if (! (bt->start = setupTopol(numsp, nsites)))  return  0;
    for (i = bt->ninit + 1; i <= newKeep; i++) {
      nlogf = (i <= maxlogf) ? nsites : 0;
      if (! (bt->byScore[i] = setupTopol(numsp, nlogf)))  break;
      bt->byTopol[i] = bt->byScore[i];
      bt->ninit = i;
      }
    bt->nkeep = bt->ninit;
    return  (bt->nkeep);
  } /* initBestTree */


static void resetBestTree (bestlist* bt)
  { /* resetBestTree */
    bt->best = unlikely;
    bt->worst = unlikely;
    bt->nvalid = 0;
    bt->improved = FALSE;
  } /* resetBestTree */


static void  freeBestTree(bestlist* bt)
  { /* freeBestTree */
    while (bt->ninit >= 0)  freeTopol(bt->byScore[(bt->ninit)--]);
    freeTopol(bt->start);
  } /* freeBestTree */


static int  saveBestTree (bestlist* bt, tree* tr)
  { /* saveBestTree */
    topol  *tpl;
    int  scrNum, tplNum;

    if ((bt->nvalid < bt->nkeep) || (tr->likelihood > bt->worst)) {
      scrNum = 1;
      tplNum = 1;
      tpl = bt->byScore[scrNum];
      saveTree(tr, tpl);
      tpl->scrNum = scrNum;
      tpl->tplNum = tplNum;

      bt->byTopol[tplNum] = tpl;
      bt->nvalid          = 1;
      bt->best            = tr->likelihood;
      bt->worst           = tr->likelihood;
      bt->improved        = TRUE;
      }

    else
      scrNum = 0;

    return  scrNum;
  } /* saveBestTree */


static int  startOpt (bestlist* bt, tree* tr)
  { /* startOpt */
    int  scrNum;

    bt->nvalid = 0;
    scrNum = saveBestTree (bt, tr);
    bt->improved = FALSE;
    return  scrNum;
  } /* startOpt */


static int  setOptLevel (bestlist* bt, int level)
  { /* setOptLevel */
    int  scrNum;

    if (! bt->improved) {
      bt->byScore[1]->opt_level = level;
      scrNum = 1;
      }
    else
      scrNum = 0;

    return  scrNum;
  } /* setOptLevel */


static int  recallBestTree (bestlist* bt, int rank, tree* tr)
  { /* recallBestTree */
    if (rank < 1)  rank = 1;
    if (rank > bt->nvalid)  rank = bt->nvalid;
    if (rank > 0)  restoreTree(bt->byScore[rank], tr);
    return  rank;
  } /* recallBestTree */

/*=======================================================================*/
/*                       End of best tree routines                       */
/*=======================================================================*/





static boolean white (int ch)  { return (ch == ' ' || ch == '\n' || ch == '\t'); }


static void getoptions (tree* tr)
  { /* getoptions */
    int     i;

    bootstrap    =  FALSE;  /* Don't bootstrap column weights */
    categs       =      0;  /* Number of rate categories */
    extrainfo    =      0;  /* No extra runtime info unless requested */
    freqsfrom    =   TRUE;  /* Use empirical base frequencies */
    tr->global   = Gafter;  /* Default search locale for optimum */
    tr->partswap =Gduring;  /* Default to swap locally after insert */
    interleaved  =   TRUE;  /* By default, data format is interleaved */
    jumble       =  FALSE;  /* Use random addition sequence */
    nkeep        =      1;  /* Keep only the one best tree */
    tr->outgr    =      1;  /* Outgroup number */
    outgropt     =  FALSE;  /* User-defined outgroup rooting */
    printdata    =  FALSE;  /* Don't echo data to output stream */
    fastadd      = Master;  /* Smooth branches globally in add */
    rate[1]      =    1.0;  /* Rate values */
    restart      =  FALSE;  /* Restart from user tree */
    treeprint    =  FALSE;  /* Print tree to output stream */
    trout        =      0;  /* Output tree file */
    ttratio      =  ttrat;  /* Transition/transversion rate ratio */
    tr->userlen  =  FALSE;  /* User-supplied branch lengths */
    tr->usertree =  FALSE;  /* User-defined tree topologies */
    tr->userwgt  =  FALSE;  /* User-defined position weights */


      for (i = 1; i <= sites; i++) weight[i] = 1;
      weightsum = sites;
      for (i = 1; i <= sites; i++) category[i] = 1;
      categs = 1;

return;

  } /* getoptions */


static void getbasefreqs (void)
  { /* getbasefreqs */
    double  suma, sumb;

    freqr = freqa + freqg;
    invfreqr = 1.0/freqr;
    freqar = freqa * invfreqr;
    freqgr = freqg * invfreqr;
    freqy = freqc + freqt;
    invfreqy = 1.0/freqy;
    freqcy = freqc * invfreqy;
    freqty = freqt * invfreqy;
    suma = ttratio*freqr*freqy - (freqa*freqg + freqc*freqt);
    sumb = freqa*freqgr + freqc*freqty;
    xi = suma/(suma+sumb);
    xv = 1.0 - xi;
    if (xi <= 0.0) {
/*    (void) printf("WARNING: This transition/transversion ratio\n");
      (void) printf("         is impossible with these base frequencies!\n");
      (void) printf("Transition/transversion parameter reset\n\n");
      xi = 3./5.;
      xv = 2./5.;*/
      anerror=TRUE;
      return;
      }
    fracchange = 2.0 * xi*(freqa*freqgr + freqc*freqty)
                     + xv*(1.0 - freqa*freqa - freqc*freqc
                               - freqg*freqg - freqt*freqt);
    /*  fracchange = 2.0 * (xi * (freqa*freqgr + freqc*freqty)
                            xv * (freqa*freqg + freqc*freqt + freqr*freqy)
                           )
                   = 2.0 * (xi * sumb
                            xv * (freqa*freqg + freqc*freqt + freqr*freqy)
                           )
     */
  } /* getbasefreqs */


static void getyspace (void)
  { /* getyspace */
    long size;
    int  i;
    char *y0;

    size = 4 * (sites/4 + 1);
    if (! (y0 = (char*) malloc((unsigned) ((numsp+1) * size * sizeof(char))))) {
      (void) printf("ERROR: Unable to obtain space for data array\n");
      anerror = TRUE;
      return;
      }

    for (i = 0; i <= numsp; i++) {
      y[i] = y0;
      y0 += size;
      }
  } /* getyspace */


static void setupTree (tree* tr)
  { /* setupTree */
    nodeptr  p0, p, q;
    int  i, j, tips, inter;

    tips  = tr->mxtips;
    inter = tr->mxtips - 1;


    if (!(p0 = (nodeptr) malloc((unsigned) ((tips + 3*inter) *
                                             sizeof(node))))) {
      (void) printf("ERROR: Unable to obtain sufficient tree memory\n");
      anerror = TRUE;
      return;
      }

    tr->nodep[0] = (node *) NULL;    /* Use as 1-based array */

    for (i = 1; i <= tips; i++) {   /* Set-up tips */
      p = p0++;
      p->x      = (xarray *) NULL;
      p->tip    = (char *) NULL;
      p->number = i;
      p->next   = p;
      p->back   = (node *) NULL;

      tr->nodep[i] = p;
      }


    for (i = tips + 1; i <= tips + inter; i++) { /* Internal nodes */
      q = (node *) NULL;
      for (j = 1; j <= 3; j++) {
        p = p0++;
        p->x      = (xarray *) NULL;
        p->tip    = (char *) NULL;
        p->number = i;
        p->next   = q;
        p->back   = (node *) NULL;
        q = p;
        }
      p->next->next->next = p;
      tr->nodep[i] = p;
      }

    tr->likelihood  = unlikely;
    tr->start       = (node *) NULL;
    tr->outgrnode   = tr->nodep[tr->outgr];
    tr->ntips       = 0;
    tr->nextnode    = 0;
    tr->opt_level   = 0;
    tr->smoothed    = FALSE;
    tr->log_f_valid = 0;
  } /* setupTree */


static void freeTreeNode (nodeptr p)       /* Free tree node (sector) associated data */
  { /* freeTreeNode */
    if (p) {
      if (p->x) {
        if (p->x->a) free((char *) p->x->a);
        free((char *) p->x);
        }
      }
  } /* freeTreeNode */


static void freeTree (tree* tr)
  { /* freeTree */
    nodeptr  p, q;
    int  i, tips, inter;

    tips  = tr->mxtips;
    inter = tr->mxtips - 1;

    for (i = 1; i <= tips; i++) freeTreeNode(tr->nodep[i]);

    for (i = tips + 1; i <= tips + inter; i++) {
      p = tr->nodep[i];
      if (p) {
        q = p->next;
        if (q) {
          freeTreeNode(q->next);
          freeTreeNode(q);
          }
        freeTreeNode(p);
        }
      }

    free((char *) tr->nodep[1]);       /* Free the actual nodes */
  } /* freeTree */




static void getinput (tree* tr)
  { /* getinput */
    getoptions(tr);                   if (anerror) return;
    getyspace();                      if (anerror) return;
    setupTree(tr);                    if (anerror) return;
  } /* getinput */


static void sitesort (void)
    /* Shell sort keeping sites, weights in same order */
  { /* sitesort */
    int  gap, i, j, jj, jg, k;
    boolean  flip, tied;

    for (gap = sites/2; gap > 0; gap /= 2) {
      for (i = gap + 1; i <= sites; i++) {
        j = i - gap;

        do {
          jj = alias[j];
          jg = alias[j+gap];
          flip = (category[jj] >  category[jg]);
          tied = (category[jj] == category[jg]);
          for (k = 1; (k <= numsp) && tied; k++) {
            flip = (y[k][jj] >  y[k][jg]);
            tied = (y[k][jj] == y[k][jg]);
            }
          if (flip) {
            alias[j]     = jg;
            alias[j+gap] = jj;
            j -= gap;
            }
          } while (flip && (j > 0));

        }  /* for (i ...   */
      }    /* for (gap ... */
  } /* sitesort */


static void sitecombcrunch (void)
    /* combine sites that have identical patterns (and nonzero weight) */
  { /* sitecombcrunch */
    int  i, sitei, j, sitej, k;
    boolean  tied;

    i = 0;
    alias[0] = alias[1];
    aliasweight[0] = 0;

    for (j = 1; j <= sites; j++) {
      sitei = alias[i];
      sitej = alias[j];
      tied = (category[sitei] == category[sitej]);

      for (k = 1; tied && (k <= numsp); k++)
        tied = (y[k][sitei] == y[k][sitej]);

      if (tied) {
        aliasweight[i] += weight[sitej];
        }
      else {
        if (aliasweight[i] > 0) i++;
        aliasweight[i] = weight[sitej];
        alias[i] = sitej;
        }
      }

    endsite = i;
    if (aliasweight[i] > 0) endsite++;
  } /* sitecombcrunch */


static void makeweights (void)
    /* make up weights vector to avoid duplicate computations */
  { /* makeweights */
    int  i;

    for (i = 1; i <= sites; i++)  alias[i] = i;
    sitesort();
    sitecombcrunch();
  } /* makeweights */


static void makevalues (tree* tr)
    /* set up fractional likelihoods at tips */
  { /* makevalues */
    double  temp;
    int  i, j, k;

    for (i = 1; i <= tr->mxtips; i++) {    /* Pack and move tip data */
      for (j = 0; j < endsite; j++)  y[i-1][j] = y[i][alias[j]];
      tr->nodep[i]->tip = &(y[i-1][0]);
      }

    totalwrate = 0.0;
    for (k = 0; k < endsite; k++) {
      catnumb[k] = i = category[alias[k]];
      ratvalue[k] = temp = rate[i];
      totalwrate += wgt_rate[k] = temp * aliasweight[k];
      wgt_rate2[k] = temp * temp * aliasweight[k];
      }
  } /* makevalues */


static void empiricalfreqs (tree* tr)
    /* Get empirical base frequencies from the data */
  { /* empiricalfreqs */
    double  sum, suma, sumc, sumg, sumt, wj, fa, fc, fg, ft;
    int  i, j, k, code;
    char *yptr;

    freqa = 0.25;
    freqc = 0.25;
    freqg = 0.25;
    freqt = 0.25;
    for (k = 1; k <= 8; k++) {
      suma = 0.0;
      sumc = 0.0;
      sumg = 0.0;
      sumt = 0.0;
      for (i = 1; i <= tr->mxtips; i++) {
        yptr = tr->nodep[i]->tip;
        for (j = 0; j < endsite; j++) {
          code = *yptr++;
          fa = freqa * ( code       & 1);
          fc = freqc * ((code >> 1) & 1);
          fg = freqg * ((code >> 2) & 1);
          ft = freqt * ((code >> 3) & 1);
          wj = aliasweight[j] / (fa + fc + fg + ft);
          suma += wj * fa;
          sumc += wj * fc;
          sumg += wj * fg;
          sumt += wj * ft;
          }
        }
      sum = suma + sumc + sumg + sumt;
      freqa = suma / sum;
      freqc = sumc / sum;
      freqg = sumg / sum;
      freqt = sumt / sum;
      }
  } /* empiricalfreqs */


static xarray *setupxarray (void)
  { /* setupxarray */
    xtype  *data;
    struct xmantyp  *x;

    x = (xarray *) malloc((unsigned) sizeof(xarray));
    if (x) {
      data = (xtype *) malloc((unsigned) (4 * endsite * sizeof(xtype)));
      if (data) {
        x->a = data;
        x->c = data += endsite;
        x->g = data += endsite;
        x->t = data +  endsite;
        x->prev = x->next = x;
        x->owner = (node *) NULL;
        }
      else {
        free((char *) x);
        return (xarray *) NULL;
        }
      }
    return x;
  } /* setupxarray */


static void linkxarray (int req, int min, xarray** freexptr, xarray** usedxptr)
    /*  Link a set of xarrays */
  { /* linkxarray */
    xarray  *first, *prev, *x;
    int  i;

    first = prev = (xarray *) NULL;
    i = 0;

    do {
      x = setupxarray();
      if (x) {
        if (! first) first = x;
        else {
          prev->next = x;
          x->prev = prev;
          }
        prev = x;
        i++;
        }
      else {
        (void) printf("ERROR: Failure to get requested xarray memory\n");
        if (i < min) {
          anerror = TRUE;
          return;
          }
        }
      } while ((i < req) && x);

    if (first) {
      first->prev = prev;
      prev->next = first;
      }

    *freexptr = first;
    *usedxptr = (xarray *) NULL;
  } /* linkxarray */


static void setupnodex (tree* tr)
  { /* setupnodex */
    nodeptr  p;
    int  i;

    for (i = tr->mxtips + 1; (i <= 2*(tr->mxtips) - 2); i++) {
      p = tr->nodep[i];
      anerror = !(p->x = setupxarray());
      if (anerror) {
        (void) printf("ERROR: Failure to get internal node xarray memory\n");
        return;
        }
      }
  } /* setupnodex */


static xarray *getxtip (nodeptr p)
  { /* getxtip */
    xarray  *new2;
    boolean  splice;

    if (! p) return (xarray *) NULL;

    splice = FALSE;

    if (p->x) {                  /* array is there; move to tail of list */
      new2 = p->x;
      if (new2 == new2->prev) ;             /* linked to self; leave it */
      else if (new2== usedxtip) usedxtip = usedxtip->next; /* at head */
      else if (new2 == usedxtip->prev) ;   /* already at tail */
      else {                              /* move to tail of list */
        new2->prev->next = new2->next;
        new2->next->prev = new2->prev;
        splice = TRUE;
        }
      }

    else if (freextip) {                 /* take from unused list */
      p->x = new2 = freextip;
      new2->owner = p;
      if (new2->prev != new2) {            /* not only member of freelist */
        new2->prev->next = new2->next;
        new2->next->prev = new2->prev;
        freextip = new2->next;
        }
      else
        freextip = (xarray *) NULL;

      splice = TRUE;
      }

    else if (usedxtip) {                 /* take from head of used list */
      usedxtip->owner->x = (xarray *) NULL;
      p->x = new2 = usedxtip;
      new2->owner = p;
      usedxtip = usedxtip->next;
      }

    else {
      (void) printf("ERROR: Unable to locate memory for tip %d.\n", p->number);
      anerror = TRUE;
      exit(1);
      }

    if (splice) {
      if (usedxtip) {                  /* list is not empty */
        usedxtip->prev->next = new2;
        new2->prev = usedxtip->prev;
        usedxtip->prev = new2;
        new2->next = usedxtip;
        }
      else
        usedxtip = new2->prev = new2->next = new2;
      }

    return  new2;
  } /* getxtip */


static xarray *getxnode (nodeptr p)
    /* Ensure that internal node p has memory */
  { /* getxnode */
    nodeptr  s;

    if (! (p->x)) {  /*  Move likelihood array on this node to sector p */
      if ((s = p->next)->x || (s = s->next)->x) {
        p->x = s->x;
        s->x = (xarray *) NULL;
        }
      else {
        (void) printf("ERROR: Unable to locate memory at node %d.\n", p->number);
        exit(1);
        }
      }
    return  p->x;
  } /* getxnode */


static void newview (nodeptr p)                      /*  Update likelihoods at node */
  { /* newview */
    double   z1, lz1, xvlz1, z2, lz2, xvlz2;
    nodeptr  q, r;
    xtype   *x1a, *x1c, *x1g, *x1t, *x2a, *x2c, *x2g, *x2t,
            *x3a, *x3c, *x3g, *x3t;
    int  i;

    if (p->tip) {             /*  Make sure that data are at tip */
      int code;
      char *yptr;

      if (p->x) return;       /*  They are already there */

      (void) getxtip(p);      /*  They are not, so get memory */
      x3a = &(p->x->a[0]);    /*  Move tip data to xarray */
      x3c = &(p->x->c[0]);
      x3g = &(p->x->g[0]);
      x3t = &(p->x->t[0]);
      yptr = p->tip;
      for (i = 0; i < endsite; i++) {
        code = *yptr++;
        *x3a++ =  code       & 1;
        *x3c++ = (code >> 1) & 1;
        *x3g++ = (code >> 2) & 1;
        *x3t++ = (code >> 3) & 1;
        }
      return;
      }

/*  Internal node needs update */

    q = p->next->back;
    r = p->next->next->back;

    while ((! p->x) || (! q->x) || (! r->x)) {
      if (! q->x) newview(q);
      if (! r->x) newview(r);
      if (! p->x) (void) getxnode(p);
      }

    x1a = &(q->x->a[0]);
    x1c = &(q->x->c[0]);
    x1g = &(q->x->g[0]);
    x1t = &(q->x->t[0]);
    z1 = q->z;
    lz1 = (z1 > zmin) ? log(z1) : log(zmin);
    xvlz1 = xv * lz1;

    x2a = &(r->x->a[0]);
    x2c = &(r->x->c[0]);
    x2g = &(r->x->g[0]);
    x2t = &(r->x->t[0]);
    z2 = r->z;
    lz2 = (z2 > zmin) ? log(z2) : log(zmin);
    xvlz2 = xv * lz2;

    x3a = &(p->x->a[0]);
    x3c = &(p->x->c[0]);
    x3g = &(p->x->g[0]);
    x3t = &(p->x->t[0]);

    { double  *zz1table, *zv1table,
              *zz2table, *zv2table,
              *zz1ptr, *zv1ptr, *zz2ptr, *zv2ptr, *rptr;
      double  fx1r, fx1y, fx1n, suma1, sumg1, sumc1, sumt1,
              fx2r, fx2y, fx2n, ki, tempi, tempj;
      int  *cptr;



#     ifdef Vectorize
        double zz1[maxsites], zv1[maxsites], zz2[maxsites], zv2[maxsites];
        int  cat;
#     else
        double zz1, zv1, zz2, zv2;
        int cat;
#     endif
zz1table=(double*)malloc((maxcategories+1)*sizeof(double));
zz2table=(double*)malloc((maxcategories+1)*sizeof(double));
zv1table=(double*)malloc((maxcategories+1)*sizeof(double));
zv2table=(double*)malloc((maxcategories+1)*sizeof(double));
if(zz1table==NULL || zz2table==NULL || zv1table==NULL || zv2table==NULL){
  anerror=TRUE;
  return;
}
      rptr   = &(rate[1]);
      zz1ptr = &(zz1table[1]);
      zv1ptr = &(zv1table[1]);
      zz2ptr = &(zz2table[1]);
      zv2ptr = &(zv2table[1]);

#     ifdef Vectorize
#       pragma IVDEP
#     endif

      for (i = 1; i <= categs; i++) {   /* exps for each category */
        ki = *rptr++;
        *zz1ptr++ = exp(ki *   lz1);
        *zv1ptr++ = exp(ki * xvlz1);
        *zz2ptr++ = exp(ki *   lz2);
        *zv2ptr++ = exp(ki * xvlz2);
        }

      cptr = &(catnumb[0]);

#     ifdef Vectorize
#       pragma IVDEP
        for (i = 0; i < endsite; i++) {
          cat = *cptr++;
          zz1[i] = zz1table[cat];
          zv1[i] = zv1table[cat];
          zz2[i] = zz2table[cat];
          zv2[i] = zv2table[cat];
        }

#       pragma IVDEP
        for (i = 0; i < endsite; i++) {
          fx1r = freqa * *x1a + freqg * *x1g;
          fx1y = freqc * *x1c + freqt * *x1t;
          fx1n = fx1r + fx1y;
          tempi = fx1r * invfreqr;
          tempj = zv1[i] * (tempi-fx1n) + fx1n;
          suma1 = zz1[i] * (*x1a++ - tempi) + tempj;
          sumg1 = zz1[i] * (*x1g++ - tempi) + tempj;
          tempi = fx1y * invfreqy;
          tempj = zv1[i] * (tempi-fx1n) + fx1n;
          sumc1 = zz1[i] * (*x1c++ - tempi) + tempj;
          sumt1 = zz1[i] * (*x1t++ - tempi) + tempj;

          fx2r = freqa * *x2a + freqg * *x2g;
          fx2y = freqc * *x2c + freqt * *x2t;
          fx2n = fx2r + fx2y;
          tempi = fx2r * invfreqr;
          tempj = zv2[i] * (tempi-fx2n) + fx2n;
          *x3a++ = suma1 * (zz2[i] * (*x2a++ - tempi) + tempj);
          *x3g++ = sumg1 * (zz2[i] * (*x2g++ - tempi) + tempj);
          tempi = fx2y * invfreqy;
          tempj = zv2[i] * (tempi-fx2n) + fx2n;
          *x3c++ = sumc1 * (zz2[i] * (*x2c++ - tempi) + tempj);
          *x3t++ = sumt1 * (zz2[i] * (*x2t++ - tempi) + tempj);
          }

#     else  /*  Not Vectorize  */
        for (i = 0; i < endsite; i++) {
          cat = *cptr++;
          zz1 = zz1table[cat];
          zv1 = zv1table[cat];
          fx1r = freqa * *x1a + freqg * *x1g;
          fx1y = freqc * *x1c + freqt * *x1t;
          fx1n = fx1r + fx1y;
          tempi = fx1r * invfreqr;
          tempj = zv1 * (tempi-fx1n) + fx1n;
          suma1 = zz1 * (*x1a++ - tempi) + tempj;
          sumg1 = zz1 * (*x1g++ - tempi) + tempj;
          tempi = fx1y * invfreqy;
          tempj = zv1 * (tempi-fx1n) + fx1n;
          sumc1 = zz1 * (*x1c++ - tempi) + tempj;
          sumt1 = zz1 * (*x1t++ - tempi) + tempj;

          zz2 = zz2table[cat];
          zv2 = zv2table[cat];
          fx2r = freqa * *x2a + freqg * *x2g;
          fx2y = freqc * *x2c + freqt * *x2t;
          fx2n = fx2r + fx2y;
          tempi = fx2r * invfreqr;
          tempj = zv2 * (tempi-fx2n) + fx2n;
          *x3a++ = suma1 * (zz2 * (*x2a++ - tempi) + tempj);
          *x3g++ = sumg1 * (zz2 * (*x2g++ - tempi) + tempj);
          tempi = fx2y * invfreqy;
          tempj = zv2 * (tempi-fx2n) + fx2n;
          *x3c++ = sumc1 * (zz2 * (*x2c++ - tempi) + tempj);
          *x3t++ = sumt1 * (zz2 * (*x2t++ - tempi) + tempj);
          }
#     endif  /*  Vectorize or not  */

free(zz1table); free(zz2table); free(zv1table); free(zv2table);
      }
  } /* newview */


static double evaluate (tree* tr, nodeptr p)
  { /* evaluate */
    double   sum, z, lz, xvlz,
             ki, fx1a, fx1c, fx1g, fx1t, fx1r, fx1y, fx2r, fx2y,
             suma, sumb, sumc, term;

       double   zz,zv;

    double   *zztable, *zvtable,
            *zzptr, *zvptr;
    double  *log_f, *rptr;
    xtype   *x1a, *x1c, *x1g, *x1t, *x2a, *x2c, *x2g, *x2t;
    nodeptr  q;
    int  cat, *cptr, i, *wptr;


zztable=(double*)malloc((maxcategories+1)*sizeof(double));
zvtable=(double*)malloc((maxcategories+1)*sizeof(double));
if(zztable==NULL || zvtable==NULL) {
  anerror=TRUE;
  return 0.;
}
    q = p->back;
    while ((! p->x) || (! q->x)) {
      if (! (p->x)) newview(p);
      if (! (q->x)) newview(q);
      }

    x1a = &(p->x->a[0]);
    x1c = &(p->x->c[0]);
    x1g = &(p->x->g[0]);
    x1t = &(p->x->t[0]);

    x2a = &(q->x->a[0]);
    x2c = &(q->x->c[0]);
    x2g = &(q->x->g[0]);
    x2t = &(q->x->t[0]);

    z = p->z;
    if (z < zmin) z = zmin;
    lz = log(z);
    xvlz = xv * lz;

    rptr  = &(rate[1]);
    zzptr = &(zztable[1]);
    zvptr = &(zvtable[1]);
    for (i = 1; i <= categs; i++) {
      ki = *rptr++;
      *zzptr++ = exp(ki *   lz);
      *zvptr++ = exp(ki * xvlz);
      }
    wptr = &(aliasweight[0]);
    cptr = &(catnumb[0]);
    log_f = tr->log_f;
    tr->log_f_valid = TRUE;
    sum = 0.0;

      for (i = 0; i < endsite; i++) {
        cat  = *cptr++;
        zz   = zztable[cat];
        zv   = zvtable[cat];
        fx1a = freqa * *x1a++;
        fx1g = freqg * *x1g++;
        fx1c = freqc * *x1c++;
        fx1t = freqt * *x1t++;
        suma = fx1a * *x2a + fx1c * *x2c + fx1g * *x2g + fx1t * *x2t;
        fx2r = freqa * *x2a++ + freqg * *x2g++;
        fx2y = freqc * *x2c++ + freqt * *x2t++;
        fx1r = fx1a + fx1g;
        fx1y = fx1c + fx1t;
        sumc = (fx1r + fx1y) * (fx2r + fx2y);
        sumb = fx1r * fx2r * invfreqr + fx1y * fx2y * invfreqy;
        suma -= sumb;
        sumb -= sumc;
        term = log(zz * suma + zv * sumb + sumc);
        sum += *wptr++ * term;
        *log_f++ = term;
        }

    tr->likelihood = sum;
free(zztable); free(zvtable);
    return  sum;
  } /* evaluate */




static double makenewz (nodeptr p, nodeptr q, double z0, int maxiter)
  { /* makenewz */
    double   *abi, *bci, *sumci,
            *abptr, *bcptr, *sumcptr;
    double   dlnLidlz, dlnLdlz, d2lnLdlz2, z, zprev, zstep, lz, xvlz,
             ki, suma, sumb, sumc, ab, bc, inv_Li, t1, t2,
             fx1a, fx1c, fx1g, fx1t, fx1r, fx1y, fx2r, fx2y;
    double   *zztable, *zvtable,
            *zzptr, *zvptr;
    double  *rptr, *wrptr, *wr2ptr;
    xtype   *x1a, *x1c, *x1g, *x1t, *x2a, *x2c, *x2g, *x2t;
    int    cat, *cptr, i, curvatOK;
    int condition; /*added for MacIntosh*/


abi=(double*)malloc(endsite * sizeof(double));
bci=(double*)malloc(endsite * sizeof(double));
sumci=(double*)malloc(endsite * sizeof(double));
if(abi==NULL || bci==NULL || sumci==NULL) {
  anerror=TRUE;
  return 0.;
}
zztable=(double*)malloc((maxcategories+1)*sizeof(double));
zvtable=(double*)malloc((maxcategories+1)*sizeof(double));
if(zztable==NULL || zvtable==NULL) {
  anerror=TRUE;
  return 0.;
}

    while ((! p->x) || (! q->x)) {
      if (! (p->x)) newview(p);
      if (! (q->x)) newview(q);
      }
  
    x1a = &(p->x->a[0]);
    x1c = &(p->x->c[0]);
    x1g = &(p->x->g[0]);
    x1t = &(p->x->t[0]);
    x2a = &(q->x->a[0]);
    x2c = &(q->x->c[0]);
    x2g = &(q->x->g[0]);
    x2t = &(q->x->t[0]);

    abptr = &(abi[0]);
    bcptr = &(bci[0]);
    sumcptr = &(sumci[0]);
  
    for (i = 0; i < endsite; i++) {
      fx1a = freqa * *x1a++;
      fx1g = freqg * *x1g++;
      fx1c = freqc * *x1c++;
      fx1t = freqt * *x1t++;
      suma = fx1a * *x2a + fx1c * *x2c + fx1g * *x2g + fx1t * *x2t;
      fx2r = freqa * *x2a++ + freqg * *x2g++;
      fx2y = freqc * *x2c++ + freqt * *x2t++;
      fx1r = fx1a + fx1g;
      fx1y = fx1c + fx1t;
      *sumcptr++ = sumc = (fx1r + fx1y) * (fx2r + fx2y);
      sumb       = fx1r * fx2r * invfreqr + fx1y * fx2y * invfreqy;
      *abptr++   = suma - sumb;
      *bcptr++   = sumb - sumc;
      }
  
    z = z0;
    do {
      zprev = z;
      zstep = (1.0 - zmax) * z + zmin;
      curvatOK = FALSE;

      do {
        if (z < zmin) z = zmin;
        else if (z > zmax) z = zmax;

        lz    = log(z);
        xvlz  = xv * lz;
        rptr  = &(rate[1]);
        zzptr = &(zztable[1]);
        zvptr = &(zvtable[1]);

        for (i = 1; i <= categs; i++) {
          ki = *rptr++;
          *zzptr++ = exp(ki *   lz);
          *zvptr++ = exp(ki * xvlz);
          }

        abptr   = &(abi[0]);
        bcptr   = &(bci[0]);
        sumcptr = &(sumci[0]);
        cptr    = &(catnumb[0]);
        wrptr   = &(wgt_rate[0]);
        wr2ptr  = &(wgt_rate2[0]);
        dlnLdlz = 0.0;                 /*  = d(ln(likelihood))/d(lz) */
        d2lnLdlz2 = 0.0;               /*  = d2(ln(likelihood))/d(lz)2 */

        for (i = 0; i < endsite; i++) {
          cat    = *cptr++;              /*  ratecategory(i) */
          ab     = *abptr++ * zztable[cat];
          bc     = *bcptr++ * zvtable[cat];
          sumc   = *sumcptr++;
          inv_Li = 1.0/(ab + bc + sumc);
          t1     = ab * inv_Li;
          t2     = xv * bc * inv_Li;
	  dlnLidlz   = t1 + t2;
          dlnLdlz   += *wrptr++  * dlnLidlz;
          d2lnLdlz2 += *wr2ptr++ * (t1 + xv * t2 - dlnLidlz * dlnLidlz);
          }

#ifdef WIN_MAC 
  condition=(z - zmax<-1.0E-6);
#else
  condition=z<zmax;
#endif 

        if ((d2lnLdlz2 >= 0.0) && condition)
          zprev = z = 0.37 * z + 0.63;  /*  Bad curvature, shorten branch */
        else
          curvatOK = TRUE;

        } while (! curvatOK);

      if (d2lnLdlz2 < 0.0) {
        z *= exp(-dlnLdlz / d2lnLdlz2);
        if (z < zmin) z = zmin;
        if (z > 0.25 * zprev + 0.75)    /*  Limit steps toward z = 1.0 */
          z = 0.25 * zprev + 0.75;
        }

      if (z > zmax) z = zmax;
      } while ((--maxiter > 0) && (ABS(z - zprev) > zstep));

free(abi); free(bci); free(sumci); free(zztable); free(zvtable);
    return  z;
  } /* makenewz */


static void update (tree* tr, nodeptr p)
  { /* update */
    nodeptr  q;
    double   z0, z;

    q = p->back;
    z0 = q->z;
    p->z = q->z = z = makenewz(p, q, z0, newzpercycle);
    if (ABS(z - z0) > deltaz)  tr->smoothed = FALSE;
  } /* update */


static void smooth (tree* tr, nodeptr p)
  { /* smooth */
  
    update(tr, p);                       /*  Adjust branch */
    if (! p->tip) {                      /*  Adjust "descendents" */
      smooth(tr, p->next->back);
      smooth(tr, p->next->next->back);

#     if ReturnSmoothedView
        newview(p);
#     endif
      }
  } /* smooth */


static void smoothTree (tree* tr, int maxtimes)
  { /* smoothTree */
    nodeptr  p;

    p = tr->start;

    while (--maxtimes >= 0 && ! anerror) {
      tr->smoothed = TRUE;
      smooth(tr, p->back);
      if (! p->tip) {
        smooth(tr, p->next->back);
        smooth(tr, p->next->next->back);
        }
      if (tr->smoothed)  break;
      }
  } /* smoothTree */


static void localSmooth (tree* tr, nodeptr p, int maxtimes)    /* Smooth branches around p */
  { /* localSmooth */
    nodeptr  pn, pnn;

    if (p->tip) return;               /* Should actually be an error */

    pn  = p->next;
    pnn = pn->next;
    while (--maxtimes >= 0) {
      tr->smoothed = TRUE;
      update(tr, p);     if (anerror) break;
      update(tr, pn);    if (anerror) break;
      update(tr, pnn);   if (anerror) break;
      if (tr->smoothed)  break;
      }
    tr->smoothed = FALSE;             /* Only smooth locally */
  } /* localSmooth */



static void insert (tree* tr, nodeptr p, nodeptr q, boolean glob)   
/* Insert node p into branch q <-> q->back */
/* Smooth tree globally? */

/*                q
                 /.
             add/ .
               /  .
             pn   .
    s ---- p      .remove
             pnn  .
               \  .
             add\ .
                 \.      pn  = p->next;
                  r      pnn = p->next->next;
 */

  { /* insert */
    nodeptr  r, s;

    r = q->back;
    s = p->back;


#if BestInsertAverage && ! Master
    { double  zqr, zqs, zrs, lzqr, lzqs, lzrs, lzsum, lzq, lzr, lzs, lzmax;

      zqr = makenewz(q, r, q->z,     iterations);
      zqs = makenewz(q, s, defaultz, iterations);
      zrs = makenewz(r, s, defaultz, iterations);

      lzqr = (zqr > zmin) ? log(zqr) : log(zmin);  /* long branches */
      lzqs = (zqs > zmin) ? log(zqs) : log(zmin);
      lzrs = (zrs > zmin) ? log(zrs) : log(zmin);
      lzsum = 0.5 * (lzqr + lzqs + lzrs);
      lzq = lzsum - lzrs;
      lzr = lzsum - lzqs;
      lzs = lzsum - lzqr;
      lzmax = log(zmax);
      if      (lzq > lzmax) {lzq = lzmax; lzr = lzqr; lzs = lzqs;} /* short */
      else if (lzr > lzmax) {lzr = lzmax; lzq = lzqr; lzs = lzrs;}
      else if (lzs > lzmax) {lzs = lzmax; lzq = lzqs; lzr = lzrs;}

      hookup(p->next,       q, exp(lzq));
      hookup(p->next->next, r, exp(lzr));
      hookup(p,             s, exp(lzs));

      }

#   else
    { double  z;
      z = sqrt(q->z);
      hookup(p->next,       q, z);
      hookup(p->next->next, r, z);
      }

#   endif

    newview(p);         /*  Required so that sector p is valid at update */
    tr->opt_level = 0;

#   if ! Master         /*  Smoothings are done by slave */
      if (! glob)  localSmooth(tr, p, smoothings);  /* Smooth locale of p */
      if (glob)    smoothTree(tr, smoothings);      /* Smooth whole tree */

#   else
      tr->likelihood = unlikely;
#   endif

return;

  } /* insert */


static nodeptr  removeNode (tree* tr, nodeptr p)

/*                q
                 .|
          remove. |
               .  |
             pn   |
    s ---- p      |add
             pnn  |
               .  |
          remove. |
                 .|      pn  = p->next;
                  r      pnn = p->next->next;
 */

    /* remove p and return where it was */
  { /* removeNode */
    double   zqr;
    nodeptr  q, r;

    q = p->next->back;
    r = p->next->next->back;
    zqr = q->z * r->z;
#   if ! Master
      hookup(q, r, makenewz(q, r, zqr, iterations));
#   else
      hookup(q, r, zqr);
#   endif

    p->next->next->back = p->next->back = (node *) NULL;
    return  q;
  } /* removeNode */



static nodeptr buildNewTip (tree* tr, nodeptr p)
  { /* buildNewTip */
    nodeptr  q;

    q = tr->nodep[(tr->nextnode)++];
    hookup(p, q, defaultz);
    return  q;
  } /* buildNewTip */
  
  



static void buildSimpleTree (tree* tr, int ip, int iq, int ir)
  { /* buildSimpleTree */
    /* p, q and r are tips meeting at s */
    nodeptr  p, s;
    int  i;
    
    i = MIN(ip, iq);
    if (ir < i)  i = ir; 
    tr->start = tr->nodep[i];
    tr->ntips = 3;
    p = tr->nodep[ip];
    hookup(p, tr->nodep[iq], defaultz);
    s = buildNewTip(tr, tr->nodep[ir]);

    insert(tr, s, p, FALSE);           /* Smoothing is local to s */
  } /* buildSimpleTree */

/*
static char * strchr (char* str, int chr)
 { 
    int  c;

    while (c = *str)  {if (c == chr) return str; str++;}
    return  (char *) NULL;
 } 


static char * strstr (char* str1, char* str2)
 { 
    char *s1, *s2;
    int  c;

    while (*(s1 = str1)) {
      s2 = str2;
      do {
        if (! (c = *s2++))  return str1;
        } 
        while (*s1++ == c);
      str1++;
      }
    return  (char *) NULL;
 } 

*/
static boolean readKeyValue (char* string, char* key, char* format, void* value)
  { /* readKeyValue */

    if (! (string = strstr(string, key)))  return FALSE;
    string += (int)strlen(key);
    if (! (string = strchr(string, '=')))  return FALSE;
    string++;
    return  sscanf(string, format, value);  /* 1 if read, otherwise 0 */
  } /* readKeyValue */


static void cacheZ (tree* tr)
  { /* cacheZ */
    nodeptr  p;
    int  nodes;

    nodes = tr->mxtips  +  3 * (tr->mxtips - 2);
    p = tr->nodep[1];
    while (nodes-- > 0) {p->z0 = p->z; p++;}
  } /* cacheZ */


static void restoreZ (tree* tr)
  { /* restoreZ */
    nodeptr  p;
    int  nodes;
    nodes = tr->mxtips  +  3 * (tr->mxtips - 2);
    p = tr->nodep[1];

    while (nodes-- > 0) {p->z = p->z0; p++;}
  } /* restoreZ */


static void testInsert (tree* tr, nodeptr p, nodeptr q, bestlist* bt, boolean fast)
  { /* testInsert */
    double  qz;
    nodeptr  r;

    r = q->back;             /* Save original connection */
    qz = q->z;
    insert(tr, p, q, ! fast);
#   if ! Master
      (void) evaluate(tr, fast ? p->next->next : tr->start);
      (void) saveBestTree(bt, tr);
#   else  /* Master */
      tr->likelihood = unlikely;
      sendTree(&comm_slave, tr);
#   endif
    /* remove p from this branch */


    hookup(q, r, qz);
    p->next->next->back = p->next->back = (nodeptr) NULL;
    if (! fast) {            /* With fast add, other values are still OK */
      restoreZ(tr);          /*   Restore branch lengths */
#     if ! Master            /*   Regenerate x values */
        initrav(p->back);
        initrav(q);
       initrav(r);

#     endif
      }
    return;
  } /* testInsert */


static int addTraverse (tree* tr, nodeptr p, nodeptr q, int mintrav, int maxtrav, bestlist* bt, boolean fast)
  { /* addTraverse */
    int  tested;
    tested = 0;
    if (--mintrav <= 0) {           /* Moved minimum distance? */
      testInsert(tr, p, q, bt, fast);
      tested++;
      }
    if ((! q->tip) && (--maxtrav > 0)) {    /* Continue traverse? */
      tested += addTraverse(tr, p, q->next->back,
                            mintrav, maxtrav, bt, fast);
      tested += addTraverse(tr, p, q->next->next->back,
                            mintrav, maxtrav, bt, fast);
      }
    return tested;
  } /* addTraverse */


static int  rearrange (tree* tr, nodeptr p, int mintrav, int maxtrav,bestlist*  bt)
    /* rearranges the tree, globally or locally */
  { /* rearrange */
    double   p1z, p2z, q1z, q2z;
    nodeptr  p1, p2, q, q1, q2;
    int      tested, mintrav2;

    tested = 0;
    if (maxtrav < 1 || mintrav > maxtrav)  return tested;

/* Moving subtree forward in tree. */

    if (! p->tip) {
      p1 = p->next->back;
      p2 = p->next->next->back;
      if (! p1->tip || ! p2->tip) {
        p1z = p1->z;
        p2z = p2->z;
        (void) removeNode(tr, p);
        cacheZ(tr);
        if (! p1->tip) {
          tested += addTraverse(tr, p, p1->next->back,
                                mintrav, maxtrav, bt, FALSE);
          tested += addTraverse(tr, p, p1->next->next->back,
                                mintrav, maxtrav, bt, FALSE);
          }

        if (! p2->tip) {
          tested += addTraverse(tr, p, p2->next->back,
                                mintrav, maxtrav, bt, FALSE);
          tested += addTraverse(tr, p, p2->next->next->back,
                                mintrav, maxtrav, bt, FALSE);
          }

        hookup(p->next,       p1, p1z);  /*  Restore original tree */
        hookup(p->next->next, p2, p2z);
        initrav(tr->start);
        initrav(tr->start->back);
        }
      }   /* if (! p->tip) */

/* Moving subtree backward in tree.  Minimum move is 2 to avoid duplicates */

    q = p->back;
    if (! q->tip && maxtrav > 1) {
      q1 = q->next->back;
      q2 = q->next->next->back;
      if (! q1->tip && (!q1->next->back->tip || !q1->next->next->back->tip) ||
          ! q2->tip && (!q2->next->back->tip || !q2->next->next->back->tip)) {
        q1z = q1->z;
        q2z = q2->z;
        (void) removeNode(tr, q);
        cacheZ(tr);
        mintrav2 = mintrav > 2 ? mintrav : 2;

        if (! q1->tip) {
          tested += addTraverse(tr, q, q1->next->back,
                                mintrav2 , maxtrav, bt, FALSE);
          tested += addTraverse(tr, q, q1->next->next->back,
                                mintrav2 , maxtrav, bt, FALSE);
          }

        if (! q2->tip) {
          tested += addTraverse(tr, q, q2->next->back,
                                mintrav2 , maxtrav, bt, FALSE);
          tested += addTraverse(tr, q, q2->next->next->back,
                                mintrav2 , maxtrav, bt, FALSE);
          }

        hookup(q->next,       q1, q1z);  /*  Restore original tree */
        hookup(q->next->next, q2, q2z);
        initrav(tr->start);
        initrav(tr->start->back);
        }
      }   /* if (! q->tip && maxtrav > 1) */

/* Move other subtrees */

    if (! p->tip) {
      tested += rearrange(tr, p->next->back,       mintrav, maxtrav, bt);
      tested += rearrange(tr, p->next->next->back, mintrav, maxtrav, bt);
      }

    return  tested;
  } /* rearrange */


static FILE *fopen_pid (char* filenm, char* mode, char* name_pid)
  { /* fopen_pid */

    /*(void) sprintf(name_pid, "%s.%d", filenm, getpid());
    return  fopen(name_pid, mode);*/
    return NULL;
  } /* fopen_pid */


static node * findAnyTip(nodeptr p)
  { /* findAnyTip */
    return  p->tip ? p : findAnyTip(p->next->back);
  } /* findAnyTip */


static void  optimize (tree* tr, int maxtrav, bestlist* bt)
  { /* optimize */
    nodeptr  p;
    int    mintrav, tested;

    if (tr->ntips < 4)  return;

    if (maxtrav > tr->ntips - 3)  maxtrav = tr->ntips - 3;
    if (maxtrav <= tr->opt_level)  return;

    /* loop while tree gets better  */

    do {
      (void) startOpt(bt, tr);
      mintrav = tr->opt_level + 1;

      /* rearrange must start from a tip or it will miss some trees */

      p = findAnyTip(tr->start);
      tested = rearrange(tr, p->back, mintrav, maxtrav, bt);


      if (anerror)  return;
      bt->numtrees += tested;
      (void) setOptLevel(bt, maxtrav);
      (void) recallBestTree(bt, 1, tr);     /* recover best found tree */


      } while (maxtrav > tr->opt_level);

  } /* optimize */


static void coordinates (tree* tr, nodeptr p, double lengthsum, drawdata* tdptr)
  { /* coordinates */
    /* establishes coordinates of nodes */
    double  x, z;
    nodeptr  q, first, last;

    if (p->tip) {
      p->xcoord = nint(over * lengthsum);
      p->ymax = p->ymin = p->ycoord = tdptr->tipy;
      tdptr->tipy += down;
      if (lengthsum > tdptr->tipmax) tdptr->tipmax = lengthsum;
      }

    else {
      q = p->next;
      do {
        z = q->z;
        if (z < zmin) z = zmin;
        x = lengthsum - fracchange * log(z);
        coordinates(tr, q->back, x, tdptr);
        q = q->next;
        } while (p == tr->start->back ? q != p->next : q != p);

      first = p->next->back;
      q = p;
      while (q->next != p) q = q->next;
      last = q->back;
      p->xcoord = nint(over * lengthsum);
      p->ycoord = (first->ycoord + last->ycoord)/2;
      p->ymin = first->ymin;
      p->ymax = last->ymax;
      }
  } /* coordinates */


static void copyTrimmedName (char  * cp1, char  * cp2)
 { /* copyTrimmedName */
   char *ep;

   ep = cp1;
   while (*ep)  ep++;                              /* move forward to end */
   ep--;                                           /* move back to last */
   while (ep >= cp1 && white((int) *(ep)))  ep--;  /* trim white */
   while (cp1 <= ep)  *cp2++ = *cp1++;             /* copy to new end */
   *cp2 = 0;
 } /* copyTrimmedName */


static double sigma (nodeptr p, double* sumlrptr)
    /* compute standard deviation */
  { /* sigma */
    double  slope, sum, sumlr, z, zv, zz, lz,
            rat, suma, sumb, sumc, d2, d, li, temp, abzz, bczv, t3,
            fx1a, fx1c, fx1g, fx1t, fx1r, fx1y, fx2r, fx2y, w;
    double  *rptr;
    xtype   *x1a, *x1c, *x1g, *x1t, *x2a, *x2c, *x2g, *x2t;
    nodeptr  q;
    int  i, *wptr;

    q = p->back;
    while ((! p->x) || (! q->x)) {
      if (! (p->x)) newview(p);
      if (! (q->x)) newview(q);
      }

    x1a = &(p->x->a[0]);
    x1c = &(p->x->c[0]);
    x1g = &(p->x->g[0]);
    x1t = &(p->x->t[0]);

    x2a = &(q->x->a[0]);
    x2c = &(q->x->c[0]);
    x2g = &(q->x->g[0]);
    x2t = &(q->x->t[0]);

    z = p->z;
    if (z < zmin) z = zmin;
    lz = log(z);

    wptr = &(aliasweight[0]);
    rptr = &(ratvalue[0]);
    sum = sumlr = slope = 0.0;

    for (i = 0; i < endsite; i++) {
      rat  = *rptr++;
      zz   = exp(rat    * lz);
      zv   = exp(rat*xv * lz);

      fx1a = freqa * *x1a++;
      fx1g = freqg * *x1g++;
      fx1c = freqc * *x1c++;
      fx1t = freqt * *x1t++;
      fx1r = fx1a + fx1g;
      fx1y = fx1c + fx1t;
      suma = fx1a * *x2a + fx1c * *x2c + fx1g * *x2g + fx1t * *x2t;
      fx2r = freqa * *x2a++ + freqg * *x2g++;
      fx2y = freqc * *x2c++ + freqt * *x2t++;
      sumc = (fx1r + fx1y) * (fx2r + fx2y);
      sumb = fx1r * fx2r * invfreqr + fx1y * fx2y * invfreqy;
      abzz = zz * (suma - sumb);
      bczv = zv * (sumb - sumc);
      li = sumc + abzz + bczv;
      t3 = xv * bczv;
      d  = abzz + t3;
      d2 = rat * (abzz*(rat-1.0) + t3*(rat*xv-1.0));

      temp = rat * d / li;
      w = *wptr++;
      slope += w *  temp;
      sum   += w * (temp * temp - d2/li);
      sumlr += w * log(li/(suma+1.0E-300));
      }

    *sumlrptr = sumlr;
    return (sum > 1.0E-300) ? z*(-slope + sqrt(slope*slope + 3.841*sum))/sum
                            : 1.0;
  } /* sigma */


static void describe (tree* tr, nodeptr p)
    /* print out information for one branch */
  { /* describe */
    double   sumlr;
    nodeptr  q;
    double z, s;



    q = p->back;

    z = q->z;
    s = sigma(q, &sumlr);
    if (! p->tip) {
      describe(tr, p->next->back);
      describe(tr, p->next->next->back);
      }
  } /* describe */


static void summarize (tree* tr)
    /* print out branch length information and node numbers */
  { /* summarize */

    describe(tr, tr->start->back->next->back);
    describe(tr, tr->start->back->next->next->back);
    describe(tr, tr->start);
   } /* summarize */


/*===========  This is a problem if tr->start->back is a tip!  ===========*/
/*  All routines should be contrived so that tr->start->back is not a tip */

static char *treeString (char* treestr, tree*  tr, nodeptr p, int form)
    /* write string with representation of tree */
    /* form == 1 -> Newick tree */
    /* form == 2 -> Prolog fact */
  { /* treeString */
    double  x, z;
    char  *nameptr;
    int  n, c;

    if (p == tr->start->back) {
      if (form == 2) {
        (void) sprintf(treestr, "phylip_tree(");
        while (*treestr) treestr++;            /* move pointer to null */
        }

      (void) sprintf(treestr, "[&&%s: version = '%s'",
                               programName, programVersion);
      while (*treestr) treestr++;

      (void) sprintf(treestr, ", %s = %15.13g",
                               likelihood_key, tr->likelihood);
      while (*treestr) treestr++;

      (void) sprintf(treestr, ", %s = %d", ntaxa_key, tr->ntips);
      while (*treestr) treestr++;

      (void) sprintf(treestr,", %s = %d", opt_level_key, tr->opt_level);
      while (*treestr) treestr++;

      (void) sprintf(treestr, ", %s = %d", smoothed_key, tr->smoothed);
      while (*treestr) treestr++;

      (void) sprintf(treestr, "]%s", form == 2 ? ", " : " ");
      while (*treestr) treestr++;
      }

    if (p->tip) {
      *treestr++ = '\'';
      n = nmlngth;
      nameptr = p->name + nmlngth - 1;
      while (*nameptr-- == ' ' && n) n--;    /*  Trim trailing spaces */
      nameptr = p->name;
      while (n--) {
        if ((c = *nameptr++) == '\'')  *treestr++ = '\'';
        *treestr++ = c;
        }
      *treestr++ = '\'';
      }

    else {
      *treestr++ = '(';
      treestr = treeString(treestr, tr, p->next->back, form);
      *treestr++ = ',';
      treestr = treeString(treestr, tr, p->next->next->back, form);
      if (p == tr->start->back) {
        *treestr++ = ',';
        treestr = treeString(treestr, tr, p->back, form);
        }
      *treestr++ = ')';
      }

    if (p == tr->start->back) {
      (void) sprintf(treestr, ":0.0%s\n", (form != 2) ? ";" : ").");
      }
    else {
      z = p->z;
      if (z < zmin) z = zmin;
      x = -log(z) * fracchange;
      (void) sprintf(treestr, ": %8.6f", x);  /* prolog needs the space */
      }

    while (*treestr) treestr++;     /* move pointer up to null termination */
    return  treestr;
  } /* treeString */


static void treeOut (FILE* treefile, tree* tr, int form, char* treestr)
    /* write out file with representation of final tree */
  { /* treeOut */

    (void) treeString(treestr, tr, tr->start->back, 1);
  } /* treeOut */


static void  showBestTrees (bestlist* bt, tree* tr, FILE* treefile, char* treestr)
  { /* showBestTrees */
    int     rank;

    for (rank = 1; rank <= bt->nvalid; rank++) {

      if (rank > 1) {
        if (rank != recallBestTree(bt, rank, tr))  break;
        }
      (void) evaluate(tr, tr->start);

      if (anerror)  return;
      if (tr->outgrnode->back)  tr->start = tr->outgrnode;
      summarize(tr);

      treeOut(treefile, tr, trout, treestr);
      }
  } /* showBestTrees */


static void cmpBestTrees (bestlist *bt, tree* tr)
  { /* cmpBestTrees */
    double  sum, sum2, sd, temp, bestscore;
    double  *log_f0;		      /* Save a copy of best log_f */
    double *log_f_ptr, *log_f0_ptr;
    int     i, j, num, besttips;

log_f0=(double*)malloc(endsite*sizeof(double));
if(log_f0==NULL){
  anerror=TRUE;
  return;
}
    num = bt->nvalid;
    if ((num <= 1) || (weightsum <= 1)) goto fincmpbesttrees;

    (void) printf("Tree      Ln L        Diff Ln L       Its S.D.");
    (void) printf("   Significantly worse?\n\n");

    for (i = 1; i <= num; i++) {
      if (i != recallBestTree(bt, i, tr))  break;
      if (! (tr->log_f_valid))  (void) evaluate(tr, tr->start);

      (void) printf("%3d%14.5f", i, tr->likelihood);
      if (i == 1) {
        (void) printf("  <------ best\n");
        besttips = tr->ntips;
        bestscore = tr->likelihood;
        log_f0_ptr = log_f0;
        log_f_ptr  = tr->log_f;
        for (j = 0; j < endsite; j++)  *log_f0_ptr++ = *log_f_ptr++;
        }
      else if (tr->ntips != besttips)
        (void) printf("  (different number of species)\n");
      else {
        sum = sum2 = 0.0;
        log_f0_ptr = log_f0;
        log_f_ptr  = tr->log_f;
        for (j = 0; j < endsite; j++) {
          temp  = *log_f0_ptr++ - *log_f_ptr++;
          sum  += aliasweight[j] * temp;
          sum2 += aliasweight[j] * temp * temp;
          }
        sd = sqrt( weightsum * (sum2 - sum*sum/weightsum) / (weightsum-1) );
        (void) printf("%14.5f%14.4f", tr->likelihood - bestscore, sd);
        (void) printf("           %s\n", (sum > 1.95996 * sd) ? "Yes" : " No");
        }
      }

    (void) printf("\n\n");
fincmpbesttrees:
free(log_f0);
return;
  } /* cmpBestTrees */






static void makeDenovoTree (tree* tr, bestlist* bt, char* treestr)
  { /* makeDenovoTree */
    char   filename[128];
    FILE  *treefile;
    nodeptr  p;
    int  enterorder[maxsp+1];      /*  random entry order */
    int  i, nextsp, newsp, maxtrav, tested;


      tr->ntips = 0;
      for (i = 1; i <= tr->mxtips; i++) enterorder[i] = i;
    bt->numtrees = 1;
    if (tr->ntips == 0) {
      for (i = 1; i <= 3; i++) {
        char  trimmed[nmlngth + 1];

        copyTrimmedName(tr->nodep[enterorder[i]]->name, trimmed);
        }
      tr->nextnode = tr->mxtips + 1;
      buildSimpleTree(tr, enterorder[1], enterorder[2], enterorder[3]);
      }
    if (anerror)  return;

    while (tr->ntips < tr->mxtips || tr->opt_level < tr->global) {
      maxtrav = (tr->ntips == tr->mxtips) ? tr->global : tr->partswap;
      if (maxtrav > tr->ntips - 3)  maxtrav = tr->ntips - 3;

      if (tr->opt_level >= maxtrav) {
        char  trimmed[nmlngth + 1];


        nextsp = ++(tr->ntips);
        newsp = enterorder[nextsp];
        p = tr->nodep[newsp];
        copyTrimmedName(p->name, trimmed);
        (void) buildNewTip(tr, p);

        resetBestTree(bt);
        cacheZ(tr);
	/*boucle infinie dans addTraverse*/
        tested = addTraverse(tr, p->back, findAnyTip(tr->start)->back,
                             1, tr->ntips - 2, bt, fastadd);
        bt->numtrees += tested;
        if (anerror)  return;

        (void) recallBestTree(bt, 1, tr);
        if (! tr->smoothed) {
          smoothTree(tr, smoothings);         if (anerror)  return;
          (void) evaluate(tr, tr->start);     if (anerror)  return;
          (void) saveBestTree(bt, tr);
          }

        if (tr->ntips == 4)  tr->opt_level = 1;  /* All 4 taxon trees done */
        maxtrav = (tr->ntips == tr->mxtips) ? tr->global : tr->partswap;
        if (maxtrav > tr->ntips - 3)  maxtrav = tr->ntips - 3;
        }

      optimize(tr, maxtrav, bt);

      if (anerror)  return;
      }

     treefile = trout ? fopen_pid("treefile", "w", filename) : (FILE *) NULL;
    showBestTrees(bt, tr, treefile, treestr);
    if (treefile) {
      (void) fclose(treefile);
      (void) printf("Tree also written to %s\n\n", filename);
      }
    cmpBestTrees(bt, tr);

  } /* makeDenovoTree */




void uprootTree (tree*  tr, nodeptr p)
   { /* uprootTree */
    nodeptr  q, r, s;
    int  n;

    if (p->tip || p->back) {
      (void) printf("ERROR: Unable to uproot tree.\n");
      (void) printf("       Inappropriate node marked for removal.\n");
      anerror = TRUE;
      return;
      }

    n = --(tr->nextnode);               /* last internal node added */
    if (n != tr->mxtips + tr->ntips - 1) {
      (void) printf("ERROR: Unable to uproot tree.  Inconsistent\n");
      (void) printf("       number of tips and nodes for rooted tree.\n");
      anerror = TRUE;
      return;
      }

    q = p->next->back;                  /* remove p from tree */
    r = p->next->next->back;
    hookup(q, r, tr->userlen ? (q->z * r->z) : defaultz);

    q = tr->nodep[n];
    r = q->next;
    s = q->next->next;
    if (tr->ntips > 2 && p != q && p != r && p != s) {
      hookup(p,             q->back, q->z);   /* move connections to p */
      hookup(p->next,       r->back, r->z);
      hookup(p->next->next, s->back, s->z);
      }

    q->back = r->back = s->back = (nodeptr) NULL;
    tr->rooted = FALSE;
  } /* uprootTree */




int str_findch (char** treestrp, int c)
  { /* str_findch */
    int ch;

    while ((ch = *(*treestrp)++) != 0 && ch != c) ;
    return  ch;
  } /* str_findch */


int str_treeFinishCom (char** treestrp)
/*  tree string pointer */
  { /* str_treeFinishCom */
    int ch;
    boolean  inquote;

    inquote = FALSE;
    while ((ch = *(*treestrp)++) != 0 && (inquote || ch != ']')) {
      if      (ch == '[' && ! inquote) {         /* comment; find its end */
        if ((ch = str_treeFinishCom(treestrp)) == 0)  break;
        }
      else if (ch == '\'') inquote = ! inquote;  /* start or end of quote */
      }
    return  ch;
  } /* str_treeFinishCom */


int str_treeGetCh (char** treestrp)
    /* get next nonblank, noncomment character */
  { /* str_treeGetCh */
    int  ch;
    
    while ((ch = *(*treestrp)++) != 0) {
      if (white(ch)) ;
      else if (ch == '[') {                   /* comment; find its end */
        if ((ch = str_treeFinishCom(treestrp)) == 0)  break;
        }
      else  break;
      }

    return  ch;
  } /* str_treeGetCh */


void  str_treeFlushLabel (char** treestrp)
  { /* str_treeFlushLabel */
    int      ch;
    boolean  done, quoted;

    if ((ch = str_treeGetCh(treestrp)) == 0)  done = TRUE;
    else {
      done = (ch == ':' || ch == ',' || ch == ')'  || ch == '[' || ch == ';');
      if (! done && (quoted = (ch == '\'')))  ch = *(*treestrp)++;
      }

    while (! done) {
      if (quoted) {
        if ((ch = str_findch(treestrp, '\'')) == 0)  done = TRUE;
        else {
          ch = *(*treestrp)++;                        /* check next char */
          if (ch != '\'') done = TRUE;                /* not doubled quote */
          }
        }
      else if (ch == ':' || ch == ',' || ch == ')'  || ch == '['
                         || ch == ';' || ch == '\n' || ch == 0) {
        done = TRUE;
        }
      if (! done)  done = ((ch = *(*treestrp)++) == 0);
      }

    (*treestrp)--;
  } /* str_treeFlushLabel */


int  str_findTipName (char** treestrp, tree* tr, int ch)
  { /* str_findTipName */
    nodeptr  q;
    char  *nameptr, str[nmlngth+1];
    int  i, n;
    boolean  found, quoted, done;

    i = 0;
    quoted = (ch == '\'');
    if (quoted) ch = *(*treestrp)++;
    done = FALSE;

    do {
      if (quoted) {
        if (ch == '\'') {
          ch = *(*treestrp)++;
          if (ch != '\'') done = TRUE;
          }
        else if (ch == 0)
          done = TRUE;
        else if (ch == '\n' || ch == '\t')
          ch = ' ';
        }
      else if (ch == ':' || ch == ','  || ch == ')'  || ch == '['
                         || ch == '\n' || ch == 0)
        done = TRUE;
      else if (ch == '_' || ch == '\t')
        ch = ' ';

      if (! done) {
        if (i < nmlngth)  str[i++] = ch;
        ch = *(*treestrp)++;
        }
      } while (! done);

    (*treestrp)--;
    if (ch == 0) {
      (void) printf("ERROR: NULL in tree species name\n");
      return  0;
      }

    while (i < nmlngth)  str[i++] = ' ';     /*  Pad name */

    n = 1;
    do {
      q = tr->nodep[n];
      if (! (q->back)) {          /*  Only consider unused tips */
        i = 0;
        nameptr = q->name;
        do {found = str[i] == *nameptr++;}  while (found && (++i < nmlngth));
        }
      else
        found = FALSE;
      } while ((! found) && (++n <= tr->mxtips));

    if (! found) {
      i = nmlngth;
      do {str[i] = '\0';} while (i-- && (str[i] <= ' '));
      (void) printf("ERROR: Cannot find data for tree species: %s\n", str);
      }

    return  (found ? n : 0);
  } /* str_findTipName */


double str_processLength (char** treestrp)
  { /* str_processLength */
    double  branch;
    int     used;

    (void) str_treeGetCh(treestrp);                /*  Skip comments */
    (*treestrp)--;

    if (sscanf(*treestrp, "%lf%n", &branch, &used) != 1) {
      (void) printf("ERROR: Problem reading branch length in str_processLength:\n");
      (void) printf("%40s\n", *treestrp);
      anerror = TRUE;
      branch = 0.0;
      }
    else {
      *treestrp += used;
      }

    return  branch;
  } /* str_processLength */


void  str_treeFlushLen (char** treestrp)
  { /* str_treeFlushLen */
    int  ch;

    if ((ch = str_treeGetCh(treestrp)) == ':')
      (void) str_processLength(treestrp);
    else
      (*treestrp)--;

  } /* str_treeFlushLen */


void  str_treeNeedCh (char** treestrp, int c1, char* where)
  { /* str_treeNeedCh */
    int  c2, i;

    if ((c2 = str_treeGetCh(treestrp)) == c1)  return;

    (void) printf("ERROR: Missing '%c' %s tree; ", c1, where);
    if (c2 == 0) 
      (void) printf("NULL");
    else {
      (void) putchar('\'');
      for (i = 24; i-- && (c2 != 0); c2 = *(*treestrp)++)  (void) putchar(c2);
      (void) putchar('\'');
      }
    (void) printf(" found instead\n");
    anerror = TRUE;
  } /* str_treeNeedCh */


void  str_addElementLen (char** treestrp, tree* tr, nodeptr p)
  { /* str_addElementLen */
    double   z, branch;
    nodeptr  q;
    int      n, ch;

    if ((ch = str_treeGetCh(treestrp)) == '(') { /*  A new internal node */
      n = (tr->nextnode)++;
      if (n > 2*(tr->mxtips) - 2) {
        if (tr->rooted || n > 2*(tr->mxtips) - 1) {
          (void) printf("ERROR: too many internal nodes.  Is tree rooted?\n");
          (void) printf("Deepest splitting should be a trifurcation.\n");
          anerror = TRUE;
          return;
          }
        else {
          tr->rooted = TRUE;
          }
        }
      q = tr->nodep[n];
      str_addElementLen(treestrp, tr, q->next);        if (anerror)  return;
      str_treeNeedCh(treestrp, ',', "in");             if (anerror)  return;
      str_addElementLen(treestrp, tr, q->next->next);  if (anerror)  return;
      str_treeNeedCh(treestrp, ')', "in");             if (anerror)  return;
      str_treeFlushLabel(treestrp);                    if (anerror)  return;
      }

    else {                           /*  A new tip */
      n = str_findTipName(treestrp, tr, ch);
      if (n <= 0) {anerror = TRUE; return; }
      q = tr->nodep[n];
      if (tr->start->number > n)  tr->start = q;
      (tr->ntips)++;
      }                              /* End of tip processing */


    if (tr->userlen){
      str_treeNeedCh(treestrp, ':', "in");               if (anerror)  return;
      branch = str_processLength(treestrp);              if (anerror)  return;
      z = exp(-branch / fracchange);
      if (z > zmax)  z = zmax;
      hookup(p, q, z);
      }
    else{
      str_treeFlushLen(treestrp);
      hookup(p, q, defaultz);
      }

    return;
  } /* str_addElementLen */


boolean str_processTreeCom(tree* tr,char**  treestrp)
  { /* str_processTreeCom */
    char  *com, *com_end;
    int  text_started, functor_read, com_open;

    com = *treestrp;

    functor_read = text_started = 0;
    (void) sscanf(com, " p%nhylip_tree(%n", &text_started, &functor_read);
    if (functor_read) {
      com += functor_read;
      }
    else if (text_started) {
      com += text_started;
      (void) sscanf(com, "seudoNewick(%n", &functor_read);
      if (! functor_read) {
        (void) printf("Start of tree 'p...' not understood.\n");
        anerror = TRUE;
        return FALSE;
        }
      else {
        com += functor_read;
        }
      }

    com_open = 0;
    (void) sscanf(com, " [%n", &com_open);
    com += com_open;

    if (com_open) {                              /* comment; read it */
	if (!(com_end = strchr(com, ']'))) {
        (void) printf("Missing end of tree comment.\n");
        anerror = TRUE;
        return FALSE;
        }

      *com_end = 0;
      (void) readKeyValue(com, likelihood_key, "%lg",
                               (void *) &(tr->likelihood));
      (void) readKeyValue(com, opt_level_key,  "%d",
                               (void *) &(tr->opt_level));
      (void) readKeyValue(com, smoothed_key,   "%d",
                               (void *) &(tr->smoothed));
      *com_end = ']';
      com_end++;

      if (functor_read) {                          /* remove trailing comma */
        text_started = 0;
        (void) sscanf(com_end, " ,%n", &text_started);
        com_end += text_started;
        }

      *treestrp = com_end;
      }

    return (functor_read > 0);
  } /* str_processTreeCom */


void str_treeReadLen (char* treestr, tree* tr)
     /* read string with representation of tree */
  { /* str_treeReadLen */
    nodeptr  p;
    int  i;
    boolean  is_fact;

    for (i = 1; i <= tr->mxtips; i++) tr->nodep[i]->back = (node *) NULL;
    tr->start       = tr->nodep[tr->mxtips];
    tr->ntips       = 0;
    tr->nextnode    = tr->mxtips + 1;
    tr->opt_level   = 0;
    tr->log_f_valid = 0;
    tr->smoothed    = Master;
    tr->rooted      = FALSE;

    is_fact = str_processTreeCom(tr, &treestr);

    p = tr->nodep[(tr->nextnode)++];
    str_treeNeedCh(&treestr, '(', "at start of");    if (anerror)  return;
    str_addElementLen(&treestr, tr, p);              if (anerror)  return;
    str_treeNeedCh(&treestr, ',', "in");             if (anerror)  return;
    str_addElementLen(&treestr, tr, p->next);        if (anerror)  return;
    if (! tr->rooted) {
      if (str_treeGetCh(&treestr) == ',') {        /*  An unrooted format */
        str_addElementLen(&treestr, tr, p->next->next);
        if (anerror)  return;
        }
      else {                                       /*  A rooted format */
        p->next->next->back = (nodeptr) NULL;
        tr->rooted = TRUE;
        treestr--;
        }
      }
    str_treeNeedCh(&treestr, ')', "in");             if (anerror)  return;
    str_treeFlushLabel(&treestr);                    if (anerror)  return;
    str_treeFlushLen(&treestr);                      if (anerror)  return;
    if (is_fact) {
      str_treeNeedCh(&treestr, ')', "at end of");    if (anerror)  return;
      str_treeNeedCh(&treestr, '.', "at end of");    if (anerror)  return;
      }
    else {
      str_treeNeedCh(&treestr, ';', "at end of");    if (anerror)  return;
      }

    if (tr->rooted)  uprootTree(tr, p->next->next);  if (anerror)  return;
    tr->start = p->next->next->back;  /* This is start used by treeString */

    initrav(tr->start);
    initrav(tr->start->back);
  } /* str_treeReadLen */



/*==========================================================================*/
/*                         "main" ml function                               */
/*==========================================================================*/


char* ml(char** seq, char** seqname, int notu, char* treestrnew, double* likel, char* toevaluate, char* err_mess)
  { /* DNA Maximum Likelihood */

    tree      curtree, *tr;   /*  current tree */
    bestlist  bestree, *bt;   /*  topology of best found tree */

    int i,j;
    int  *meaning;          /*  meaning of input characters */ 
    char* error;
    char* chpt;

    char* newbltree=NULL;


meaning=(int*)malloc(256*sizeof(int));
if(meaning==NULL) return NULL;
/* dynamic allocation needed on Mac because struct cannot exceed 32k */
curtree.log_f=(double*)malloc(maxpatterns * sizeof(double));
curtree.nodep=(node**)malloc(2*maxsp * sizeof(node*));
if(curtree.log_f==NULL || curtree.nodep==NULL) {
  (void) printf("not enough memory\n");
  exit(0);
}
error=(char*)malloc(10*sizeof(char));
(void) strcpy(error,"erreur");



    anerror = FALSE;
    tr = & curtree;
    bt = & bestree;
    bt->ninit = 0;


    for (i = 0; i <= 255; i++) meaning[i] = 0;
    meaning['A'] =  1;
    meaning['B'] = 14;
    meaning['C'] =  2;
    meaning['D'] = 13;
    meaning['G'] =  4;
    meaning['H'] = 11;
    meaning['K'] = 12;
    meaning['M'] =  3;
    meaning['N'] = 15;
    meaning['O'] = 15;
    meaning['R'] =  5;
    meaning['S'] =  6;
    meaning['T'] =  8;
    meaning['U'] =  8;
    meaning['V'] =  7;
    meaning['W'] =  9;
    meaning['X'] = 15;
    meaning['Y'] = 10;
    meaning['?'] = 15;
    meaning['-'] = 15;


    numsp=notu; sites=(int)strlen(seq[0]);
    tr->mxtips = numsp;


alias=(int*)malloc((sites+1)*sizeof(int));
aliasweight=(int*)malloc((sites+1)*sizeof(int));
category=(int*)malloc((sites+1)*sizeof(int));
weight=(int*)malloc((sites+1)*sizeof(int));
if(alias==NULL || aliasweight==NULL || weight==NULL || category==NULL)
  return NULL;


    getinput(tr);

    


    for(i=0;i<numsp;i++){
	for(j=0;j<sites;j++){
	     y[i+1][j+1]=seq[i][j];
	}

	(void) strncpy(tr->nodep[i+1]->name,seqname[i],nmlngth);
	chpt=tr->nodep[i+1]->name;
	for(j=(int)strlen(chpt);j<nmlngth;j++) chpt[j]=' ';
    }


    for (j = 1; j <= tr->mxtips; j++)    /* Convert characters to meanings */
      for (i = 1; i <= sites; i++)  y[j][i] = meaning[y[j][i]];


/*  The material below would be a loop over jumbles and/or boots */

    makeweights();                              if (anerror)  return NULL;
 
   
ratvalue=(double*)malloc(endsite*sizeof(double));
wgt_rate=(double*)malloc(endsite*sizeof(double));
wgt_rate2=(double*)malloc(endsite*sizeof(double));
catnumb=(int*)malloc(endsite*sizeof(int));
if(ratvalue==NULL || wgt_rate==NULL || wgt_rate2==NULL || catnumb==NULL)
  return NULL;
 
    makevalues(tr);                             if (anerror)  return NULL;
    empiricalfreqs(tr);                         if (anerror)  return NULL;
    getbasefreqs();				
    if (anerror) {
      (void) sprintf(err_mess, "Transition/transversion ratio impossible with these base frequencies");
      return NULL;
    }

    linkxarray(3, 3, & freextip, & usedxtip);   if (anerror)  return NULL;
    setupnodex(tr);                             if (anerror)  return NULL;
    

   (void) initBestTree(bt, nkeep, endsite);    if (anerror)  return NULL;

     if (!toevaluate){
       makeDenovoTree(tr, bt, treestrnew);  /*** tree-inf ****/
       if (anerror)  return NULL; 
     }
     else {
       tr->usertree=TRUE;
       tr->userlen=FALSE;
       str_treeReadLen(toevaluate, tr);
       smoothTree(tr, 4 * smoothings);
       (void) evaluate(tr, tr->start);
newbltree=(char*)malloc((50*notu+500)*sizeof(char));
if(newbltree==NULL) return NULL;
       treeOut(NULL, tr, trout, newbltree);
     }

     if (likel) *likel=tr->likelihood;

     freeBestTree(bt);
     if (anerror)  return NULL;
     freeTree(tr);
   
free(meaning);
free(alias); free(aliasweight); free(category); free(weight);
free(ratvalue); free(wgt_rate); free(wgt_rate2);  free(catnumb); 

/* remove final 0.0 branch length */
if(!toevaluate){
  chpt=treestrnew;
  while(*chpt!=';') chpt++;
  while(*chpt!=')') chpt--;
  chpt++;
  *chpt=';';
  chpt++;
  while(*chpt) {*chpt='\0'; chpt++;}

}


    return newbltree;


  } /* DNA Maximum Likelihood */



/*
main(){

char* arbre, *seq[100], *seqname[100], **comments=NULL, filename[50];
int notu;

(void) printf("Fichier de sequences? (mase)  ");
gets(filename);
notu=readmaseseqs(filename, seq, seqname, comments, 100);
arbre=(char*)malloc(10000*sizeof(char));
ml(seq, seqname, notu, arbre);

(void) printf("%s\n",arbre);
}
*/

