//========================================================
//
// PipeList
//
// Mark Russinovich
// Systems Internals
// http://www.sysinternals.com
//
// This program lists the named pipes active on a system.
// Note that we cannot use the Win32 FindFile functions
// since we have to open the root directory of the named
// pipe driver for an enumeration and the Win32 functions
// don't do that.
//
//========================================================
#include <windows.h>
#include "stdio.h"
#include "stdlib.h"
#include "pipelist.h"


//
// Native functions we use
//
NTSTATUS (_stdcall * NtQueryDirectoryFile)( 
					IN HANDLE FileHandle,
					IN HANDLE Event,
					IN PIO_APC_ROUTINE ApcRoutine,
					IN PVOID ApcContext,
					OUT PIO_STATUS_BLOCK IoStatusBlock,
					OUT PVOID FileInformation,
					IN ULONG Length,
					IN FILE_INFORMATION_CLASS FileInformationClass,
					IN BOOLEAN ReturnSingleEntry,
					IN PUNICODE_STRING FileName,
					IN BOOLEAN RestartScan );

ULONG (__stdcall *RtlNtStatusToDosError) (
		IN NTSTATUS Status
		);

//----------------------------------------------------------------------
//
// PrintNtError
//
// Formats an error message for the last native error.
//
//----------------------------------------------------------------------
void PrintNtError( char *message, NTSTATUS status )
{
	char *errMsg;

	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL, RtlNtStatusToDosError( status ), 
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
			(LPTSTR) &errMsg, 0, NULL );
	printf("%s: %s\n", message, errMsg );
	LocalFree( errMsg );
}

//----------------------------------------------------------------------
//
// PrintWin32Error
//
// Formats an error message for the last Win32 error.
//
//----------------------------------------------------------------------
void PrintWin32Error( char *message, DWORD error )
{
	char *errMsg;

	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL, error, 
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
			(LPTSTR) &errMsg, 0, NULL );
	printf("%s: %s\n", message, errMsg );
	LocalFree( errMsg );
}



//----------------------------------------------------------------------
//
// Main
//
// Does all the work.
//
//----------------------------------------------------------------------
int main( int argc, char *argv[] )
{
	HANDLE pipes;
	NTSTATUS ntStatus;
	PFILE_DIRECTORY_INFORMATION dirInfo, curEntry;
	BOOLEAN firstQuery = TRUE;
	IO_STATUS_BLOCK ioStatus;
	WCHAR instances[80], maxinstances[80], pipeName[80];

	//
	// Print the banner
	//
	printf("\nPipeList v1.01\n" );
	printf("by Mark Russinovich\n");
	printf("http://www.sysinternals.com\n\n");

	//
	// Locate NTDLL entry points
	//
	if( !(NtQueryDirectoryFile = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
							"NtQueryDirectoryFile" )) ) {

		printf("\nCould not find NtQueryDirectoryFile entry point in NTDLL.DLL\n");
		exit(1);
	}
	if( !(RtlNtStatusToDosError = (void *) GetProcAddress( GetModuleHandle("ntdll.dll"),
							"RtlNtStatusToDosError" )) ) {

		printf("\nCould not find RtlNtStatusToDosError entry point in NTDLL.DLL\n");
		exit(1);
	}
	
	//
	// Perform a directory listing of the existing pipe
	//
	pipes = CreateFile( "\\\\.\\Pipe\\", 
			  GENERIC_READ,
			  FILE_SHARE_READ|FILE_SHARE_WRITE|FILE_SHARE_DELETE,
			  NULL,
			  OPEN_EXISTING,
			  0, 
			  NULL );
	if( pipes == INVALID_HANDLE_VALUE ) {

		PrintWin32Error("Pipe error", GetLastError());
		return 0;
	}

	//
	// Print the header
	// 
	printf("%-30s%14s%20s\n", "Pipe Name", "Instances", "Max Instances" );
	printf("%-30s%14s%20s\n", "---------", "---------", "-------------" );

	//
	// Query until no more files
	//
	dirInfo = (PFILE_DIRECTORY_INFORMATION) malloc( DIRBUFSIZE );
	while( 1 ) {

		//
		// Do a native directory listing
		//
		ntStatus = NtQueryDirectoryFile( pipes, NULL, NULL, 0, &ioStatus,
								dirInfo, DIRBUFSIZE, FileDirectoryInformation,
								FALSE, NULL, firstQuery );
		if( !NT_SUCCESS( ntStatus )) {

			if( ntStatus != STATUS_NO_MORE_FILES ) {

				PrintNtError("Error querying pipe directory:", ntStatus );
			}
			break;
		}

		//
		// Dump the information nicely formatted
		//
		curEntry = dirInfo;
		while( 1 ) {

			swprintf( instances,    L"%d   ", curEntry->EndOfFile.LowPart );
			swprintf( maxinstances, L"%d      ", curEntry->AllocationSize.LowPart );
			wcsncpy( pipeName, curEntry->FileName, curEntry->FileNameLength/2 ); 
			pipeName[ curEntry->FileNameLength/2 ] = 0;

			wprintf(L"%-30s%14s%20s\n", pipeName,
										instances,
										maxinstances );

			if( curEntry->NextEntryOffset == 0 ) break;

			curEntry = (PFILE_DIRECTORY_INFORMATION) ((PCHAR) curEntry + curEntry->NextEntryOffset );
		}

		firstQuery = FALSE;
	}

	CloseHandle( pipes );
	return 0;
}
