/* $Id: BaseRSAPrivateKey.java,v 1.1 1999/08/13 17:39:31 gelderen Exp $
 *
 * Copyright (C) 1995-1999 Systemics Ltd.
 * on behalf of the Cryptix Development Team. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General License along with this library;
 * if not, you can download a copy from http://www.cryptix.org/ .
 */
package cryptix.jce12.provider.rsa;


// + Now u = q^-1 (mod p), not p^-1 (mod q). Apart from PGP, this is
//   more commonly used (e.g. see the P1363 draft).
// + Added check that uq = 1 (mod p). If this check fails, a debugging
//   message is printed, and the given value of u is ignored. This is
//   worthwhile because uq (mod p) is much faster to calculate than
//   q^-1 (mod p), and it makes the code robust against errors where
//   p and q are incorrectly swapped.


import java.math.BigInteger;
import java.security.InvalidParameterException;
import java.security.interfaces.RSAPrivateKey;


/**
 * An abstract class representing an RSA private key.
 *
 * @version $Revision: 1.1 $
 * @author  Raif S. Naffah
 * @author  David Hopwood
 * @author  Jeroen C. van Gelderen
 * @since   Cryptix 2.2.2
 */
public abstract class BaseRSAPrivateKey
implements RSAPrivateKey
{

// Variables
//...........................................................................

    private static final BigInteger ZERO = BigInteger.valueOf(0L);
    private static final BigInteger ONE =  BigInteger.valueOf(1L);

    /**
     * Public decryption modulus. It is the product of the two <i>p</i>
     * and <i>q</i> factors.
     */
    private BigInteger n;

    /**
     * Private encryption exponent. Traditionally referred to as <i>d</i>.
     */
    private BigInteger d;

    /**
     * The first factor of the public modulus <i>n</i> traditionally
     * referred to as <i>p</i>.
     */
    private BigInteger p;

    /**
     * The second factor of the public modulus <i>n</i> traditionally
     * referred to as <i>q</i>.
     */
    private BigInteger q;

    /**
     * The result of <i>q</i>^-1 (mod <i>p</i>), called the 'multiplicative
     * inverse' and traditionally referred to as <i>u</i>. This is used in
     * modular exponentiation operations using the Chinese Remainder
     * Theorem (CRT).
     */
    private BigInteger u;


// Constructor
//...........................................................................

    /**
     * Constructs an RSA private key, without setting the parameters.
     * Subclasses should call one of the setRsaParams methods in each of
     * their constructors.
     */
    protected BaseRSAPrivateKey() {}


// RSAKey interface methods implementation
//...........................................................................

    /**
     * Return the public modulus <i>n</i>: the product of both <i>p</i>
     * and <i>q</i>.
     *
     * @return the public modulus <i>n</i>: the product of both <i>p</i>
     *         and <i>q</i>.
     */
    public BigInteger getModulus() { return n; }

    /**
     * Return the private exponent <i>d</i>.
     *
     * @return the private exponent <i>d</i>.
     */
    public BigInteger getExponent() { return d; }


// RSAFactors interface methods implementation
//...........................................................................

    /**
     * Returns <i>p</i>, the first factor of the public modulus.
     *
     * @return the first factor <i>p</i>
     */
    public BigInteger getP() { return p; }

    /**
     * Return <i>q</i>, the second factor of the public modulus.
     *
     * @return the second factor <i>q</i>
     */
    public BigInteger getQ() { return q; }

    /**
     * Returns the multiplicative inverse of <i>q</i> modulo <i>p</i>. The
     * values <i>p</i> and <i>q</i> are those returned by the <i>getP()</i>
     * and <i>getQ()</i> methods respectively.
     *
     * @return the multiplicative inverse of <i>q</i> modulo <i>p</i>.
     */
    public BigInteger getInverseOfQModP() { return u; }


// Key interface methods implementation
//...........................................................................

    /**
     * Returns the name of the algorithm, for this class always "RSA".
     *
     * @return the name of the algorithm, "RSA".
     */
    public String getAlgorithm() { return "RSA"; }


// Own methods
//...........................................................................

    /**
     * Sets the RSA parameters <i>n</i> and <i>d</i>.
     *
     * @exception NullPointerException if n == null || d == null
     */
    protected void setRsaParams(BigInteger n, BigInteger d) {
        if (n == null) throw new NullPointerException("n == null");
        if (d == null) throw new NullPointerException("d == null");

        this.n = n;
        this.d = d;
    }

    /**
     * Sets the RSA parameters <i>d</i>, <i>p</i>, <i>q</i>, and <i>u</i>,
     * to allow fast execution of mathematical operations performed later
     * on during the life of this key. <i>u</i> may be null, in which case
     * it is calculated automatically.
     *
     * @exception NullPointerException if d == null || p == null || q == null
     * @exception InvalidParameterException if u must be calculated, and
     *              gcd(q, p) != 1
     */
    protected void setRsaParams(BigInteger d, BigInteger p, BigInteger q,
                                BigInteger u) {
        if (d == null) throw new NullPointerException("d == null");

        this.n = p.multiply(q);
        this.d = d;
        this.p = p;
        this.q = q;

        if (u != null && !u.multiply(q).mod(p).equals(ONE)) {
            u = null;
        }
        if (u == null) {
            try {
                u = q.modInverse(p);
            } catch (ArithmeticException ae) {
                throw new InvalidParameterException("gcd(q, p) != 1");
            }
        }
        this.u = u;
    }

    /**
     * Returns a string representation of this key. This may reveal
     * private information when debugging is enabled, and should be used
     * with care.
     *
     * @return a string representation of this key.
     */
    public String toString() {
        return "BaseRSAPrivateKey";
    }
}
