/* $Id: Cipher.java,v 1.1.1.1 1999/07/13 23:15:32 gelderen Exp $
 *
 * Copyright (C) 1995-1998 Systemics Ltd. 
 * on behalf of the Cryptix Development Team. All rights reserved.
 * 
 * Use, modification, copying and distribution of this software is subject 
 * the terms and conditions of the Cryptix General Licence. You should have 
 * received a copy of the Cryptix General License along with this library; 
 * if not, you can download a copy from http://www.cryptix.org/ .
 */
package javax.crypto;


import java.security.AlgorithmParameters;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.security.spec.AlgorithmParameterSpec;


// XXX TODO: implement state
public class Cipher {
    
    public static final int DECRYPT_MODE = 2;
    public static final int ENCRYPT_MODE = 1;


    private final CipherSpi spi_;
    private final Provider  provider_;
    private final String    transformation_;
        
    
    protected Cipher(CipherSpi cipherSpi, Provider provider, String transformation) {
        
        spi_            = cipherSpi;
        provider_       = provider;
        transformation_ = transformation;
    }
    
    
    // FIXME: 
    // 1. Clean up and 
    // 2. check for compatibility. 
    // 3. Fix ignored exceptions
    // 4. Possibly merge with Support.java
    private static Object[] getCipherImplementation(String transformation, Provider p) {
        
        //
        // Extract name components from 'transformation'
        //
        String part_alg;
        String part_mode = "//"; // Default is invalid name
        String part_pad  = "//";
        
        int index_1 = transformation.indexOf('/');
        int index_2 = transformation.indexOf('/', index_1+1);
        
        if(index_1==-1) {
            part_alg  = transformation;
        } 
        else if(index_2==-1) {
            return null;
//            part_alg  = transformation.substring(0, index_1);
//            part_mode = transformation.substring(index_1+1);
        }
        else {
            part_alg  = transformation.substring(0, index_1);
            part_mode = transformation.substring(index_1+1, index_2);
            part_pad  = transformation.substring(index_2+1);
        }
        
        
        //
        // Try and get a class instance
        //
        try {
            CipherSpi spi;            
            String    class_name;
            Object[]  res = new Object[2];

            res[1] = p;
            
            class_name = Support.getClassName("Cipher", transformation, p);
            if(class_name!=null) {
                res[0] = Class.forName(class_name).newInstance();
                return res;
            }
            class_name = Support.getClassName("Cipher", part_alg + "/" + part_mode, p);
            if(class_name!=null) {
                spi = (CipherSpi)Class.forName(class_name).newInstance();
                spi.engineSetPadding(part_pad);
                res[0] = spi;
                return res;
            }
            class_name = Support.getClassName("Cipher", part_alg + "//" + part_alg, p);
            if(class_name!=null) {
                spi = (CipherSpi)Class.forName(class_name).newInstance();
                spi.engineSetMode(part_mode);
                res[0] = spi;
                return res;
            }
            class_name = Support.getClassName("Cipher", part_alg, p);
            if(class_name!=null) {
                spi = (CipherSpi)Class.forName(class_name).newInstance();
                spi.engineSetMode(part_mode);
                spi.engineSetPadding(part_pad);
                res[0] = spi;
                return res;
            }
        }
        catch(NoSuchPaddingException e) {
        }
        catch(NoSuchAlgorithmException e) {
        }
        catch(LinkageError e) {
            // FIXME: Throw a RuntimeException(?) with a sensible message????
        } 
        catch(ClassNotFoundException e) {
            // FIXME: Throw a RuntimeException(?) with a sensible message????
        } 
        catch(InstantiationException e) {
            // FIXME: Throw a RuntimeException(?) with a sensible message????
        } 
        catch(IllegalAccessException e) {
            // FIXME: Throw a RuntimeException(?) with a sensible message????
        }
        return null;
    }
    
    
    public static Cipher getInstance(String transformation)
    throws NoSuchAlgorithmException, NoSuchPaddingException {
        
        Provider[] providers = Security.getProviders();
        if( (providers==null) || (providers.length==0) )
            throw new NoSuchAlgorithmException("No providers installed");
            
        for(int i=0; i<providers.length; i++) {
            Object[] res = getCipherImplementation(transformation, providers[i]);
            if(res != null)
                return new Cipher((CipherSpi)res[0], (Provider)res[1], transformation);
        }
        throw new NoSuchAlgorithmException("Algorithm not found. [Cipher." + transformation + "]");
    }
    
    
    public static Cipher getInstance(String transformation, String provider)
    throws NoSuchAlgorithmException, NoSuchProviderException, NoSuchPaddingException {
        
        Provider p = Security.getProvider(provider);
        if(p==null)
            throw new NoSuchProviderException("Provider not found. [" + provider + "]");
            
        Object[] res = getCipherImplementation(transformation, p);
        if(res != null)
            return new Cipher((CipherSpi)res[0], (Provider)res[1], transformation);
            
        throw new NoSuchAlgorithmException("Algorithm not found. [Cipher." + transformation + "]");
    }
    
           
    
    public final Provider getProvider() {
        return provider_;
    }
    
    
    public final String getAlgorithm() {
        return transformation_;
    }
    
    
    public final int getBlockSize() {
        return spi_.engineGetBlockSize();
    }
    
    
    public final int getOutputSize(int inputLen)
    throws IllegalStateException {
        
        if(inputLen < 0)
            throw new IllegalArgumentException("Input size must be >= 0");
            
        return spi_.engineGetOutputSize(inputLen);
    }
    
    
    public final byte[] getIV() {
        return spi_.engineGetIV();
    }
    
    
    public final AlgorithmParameters getParameters() {
        return spi_.engineGetParameters();
    }
    
    
    public final void init(int opmode, Key key) 
    throws InvalidKeyException {
        spi_.engineInit(opmode, key, new SecureRandom());
    }
    
    
    public final void init(int opmode, Key key, SecureRandom random)
    throws InvalidKeyException {
        spi_.engineInit(opmode, key, random);
    }
    
    
    public final void init(int opmode, Key key, AlgorithmParameterSpec params)
    throws InvalidKeyException, InvalidAlgorithmParameterException {
        spi_.engineInit(opmode, key, params, new SecureRandom());
    }
    
    
    public final void init(int opmode, Key key, AlgorithmParameterSpec params, SecureRandom random)
    throws InvalidKeyException, InvalidAlgorithmParameterException {
        spi_.engineInit(opmode, key, params, random);
    }
    
    
    public final void init(int opmode, Key key, AlgorithmParameters params) 
    throws InvalidKeyException, InvalidAlgorithmParameterException {
        spi_.engineInit(opmode, key, params, new SecureRandom());
    }
    
    
    public final void init(int opmode, Key key, AlgorithmParameters params, SecureRandom random)
    throws InvalidKeyException, InvalidAlgorithmParameterException {
        spi_.engineInit(opmode, key, params, random);
    }
    
    
    public final byte[] update(byte[] input) 
    throws IllegalStateException {
        if(input == null)
            throw new IllegalArgumentException();
            
        return spi_.engineUpdate(input, 0, input.length);
    }
    
    
    public final byte[] update(byte[] input, int inputOffset, int inputLen)
    throws IllegalStateException {
        return spi_.engineUpdate(input, inputOffset, inputLen);
    }
    
    
    public final int update(byte[] input, int inputOffset, int inputLen, byte[] output)
    throws IllegalStateException, ShortBufferException {
        return spi_.engineUpdate(input, inputOffset, inputLen, output, 0);
    }
    
    
    public final int update(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset)
    throws IllegalStateException, ShortBufferException {
        return spi_.engineUpdate(input, inputOffset, inputLen, output, outputOffset);
    }
    
    
    public final byte[] doFinal() 
    throws IllegalStateException, IllegalBlockSizeException, BadPaddingException {
        return spi_.engineDoFinal(null, 0, 0);
    }
    
    
    public final int doFinal(byte[] output, int outputOffset)
    throws IllegalStateException, IllegalBlockSizeException, ShortBufferException, BadPaddingException {
        return spi_.engineDoFinal(null, 0, 0, output, outputOffset);
    }
    
    
    public final byte[] doFinal(byte[] input)
    throws IllegalStateException, IllegalBlockSizeException, BadPaddingException {
        return spi_.engineDoFinal(input, 0, input.length);
    }
    
    
    public final byte[] doFinal(byte[] input, int inputOffset, int inputLen)
    throws IllegalStateException, IllegalBlockSizeException, BadPaddingException {
        return spi_.engineDoFinal(input, inputOffset, inputLen);
    }
    
    
    public final int doFinal(byte[] input, int inputOffset, int inputLen, byte[] output) 
    throws IllegalStateException, ShortBufferException, IllegalBlockSizeException, BadPaddingException {
        return spi_.engineDoFinal(input, inputOffset, inputLen, output, 0);
    }
    
    
    public final int doFinal(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset)
    throws IllegalStateException, ShortBufferException, IllegalBlockSizeException, BadPaddingException {
        return spi_.engineDoFinal(input, inputOffset, inputLen, output, outputOffset);
    }
}












