/* $Id: Mac.java,v 1.1.1.1 1999/07/13 23:15:32 gelderen Exp $
 *
 * Copyright (C) 1995-1998 Systemics Ltd. 
 * on behalf of the Cryptix Development Team. All rights reserved.
 * 
 * Use, modification, copying and distribution of this software is subject 
 * the terms and conditions of the Cryptix General Licence. You should have 
 * received a copy of the Cryptix General License along with this library; 
 * if not, you can download a copy from http://www.cryptix.org/ .
 */
package javax.crypto;


import java.security.InvalidKeyException;
import java.security.InvalidAlgorithmParameterException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.spec.AlgorithmParameterSpec;


public class Mac {
    
    private static final int STATE_UNDEFINED   = 0;
    private static final int STATE_INITIALIZED = 1;
    
    private       int      state_;
    private final MacSpi   spi_;
    private final Provider provider_;
    private final String   algorithm_;
    
    
    protected Mac(MacSpi macSpi, Provider provider, String algorithm) {
        
        this(macSpi, provider, algorithm, STATE_UNDEFINED);
    }
    

    /**
     * Construct a Mac with a given initial state.
     *
     * This constructor comes in handy when cloning.
     */ 
    private Mac(MacSpi macSpi, Provider provider, String algorithm, int state) {
        spi_       = macSpi;
        provider_  = provider;
        algorithm_ = algorithm;
        state_     = state;
    }
    
    
    public final String getAlgorithm() {
        return algorithm_;
    }
    
    
    public static final Mac getInstance(String algorithm) 
    throws NoSuchAlgorithmException {
        
        Object[] o = Support.getImplementation("Mac", algorithm);
        return new Mac((MacSpi)o[0], (Provider)o[1], algorithm);
    }
    
    
    public static final Mac getInstance(String algorithm, String provider)
    throws NoSuchAlgorithmException, NoSuchProviderException {
        
        Object[] o = Support.getImplementation("Mac", algorithm, provider);
        return new Mac((MacSpi)o[0], (Provider)o[1], algorithm);
    }
    
    
    public final Provider getProvider() {
        return provider_;
    }
    
    
    public final int getMacLength() {
        return spi_.engineGetMacLength();
    }
    
    
    public final void init(Key key) 
    throws InvalidKeyException {
        try {
            spi_.engineInit(key, null);
        }
        catch(InvalidAlgorithmParameterException e) {
            throw new InvalidKeyException("This Mac requires an AlgorithmParameterSpec");
        }
        state_ = STATE_INITIALIZED;
    }
    
    
    public final void init(Key key, AlgorithmParameterSpec params)
    throws InvalidKeyException, InvalidAlgorithmParameterException {
        spi_.engineInit(key, params);
        state_ = STATE_INITIALIZED;
    }
    
    
    public final void update(byte input)
    throws IllegalStateException {
        
        if(state_ != STATE_INITIALIZED)
            throw new IllegalStateException();
            
        spi_.engineUpdate(input);
    }
    
    
    public final void update(byte[] input)
    throws IllegalStateException {

        if(state_ != STATE_INITIALIZED)
            throw new IllegalStateException();

        spi_.engineUpdate(input, 0, input.length);
    }
    
    
    public final void update(byte[] input, int offset, int len)
    throws IllegalStateException {

        if(state_ != STATE_INITIALIZED)
            throw new IllegalStateException();

        spi_.engineUpdate(input, offset, len);
    }
    
    
    public final byte[] doFinal() 
    throws IllegalStateException {

        if(state_ != STATE_INITIALIZED)
            throw new IllegalStateException();

        byte[] res = spi_.engineDoFinal();
        reset();
        return res;
    }
    
    
    public final void doFinal(byte[] output, int outOffset)
    throws ShortBufferException, IllegalStateException {

        if(state_ != STATE_INITIALIZED)
            throw new IllegalStateException();
        
        int my_len = getMacLength();
        
        if( (output.length-outOffset) < my_len )
            throw new ShortBufferException("Buffer too short");
            
        byte[] res = spi_.engineDoFinal();
        System.arraycopy(res, 0, output, outOffset, my_len);
        
        reset();
    }


    public final byte[] doFinal(byte[] input) 
    throws IllegalStateException {

        if(state_ != STATE_INITIALIZED)
            throw new IllegalStateException();
        
        update(input);
        return doFinal();
    }
    
    
    public final void reset() {
        
        spi_.engineReset();
    }
    
    
    public final Object clone()
    throws CloneNotSupportedException {
        
        MacSpi clone_spi = (MacSpi)spi_.clone();
        Mac    clone_mac = new Mac(clone_spi, provider_, algorithm_, state_);
        return clone_mac;
    }
    
}