/* cc.c -- closed caption decoder 
 * Mike Baker (mbm@linux.com)
 * (based on code by timecop@japan.co.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>

/* #define XWIN 1   /* */

#ifdef XWIN
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
Display *dpy;
Window Win,Root;
char dpyname[256] = "";
GC WinGC;
GC WinGC0;
GC WinGC1;
int x;
#endif

char    *ratings[] = {"(NOT RATED)","TV-Y","TV-Y7","TV-G","TV-PG","TV-14","TV-MA","(NOT RATED)"};
int     rowdata[] = {11,-1,1,2,3,4,12,13,14,15,5,6,7,8,9,10};
char	*specialchar[] = {"","","","","(TM)","","","o/~ ",""," ","","","","","",""};
char	*modes[]={"current","future","channel","miscellanious","public service","reserved","invalid","invalid","invalid","invalid"};

char	info[8][25][256]; 
char	newinfo[8][25][256];
char	*infoptr=newinfo[0][0];
int	mode,type;
char	infochecksum;

int	ccmode=1;		//cc1 or cc2
int	ccdbl=1;		//some captions are double buffered
char	ccbuf[2][2][255];	//cc is 32 columns per row, this allows for extra characters

int column;
int lastcode;

int parityok(int n)	/* check parity for 2 bytes packed in n */
{
    int mask=0;
    int j, k;
    for (k = 1, j = 0; j < 7; j++)
	if (n & (1<<j)) k++;
    if ((k & 1) == ((n>>7)&1))
	    mask|=0x00FF;
    for (k = 1, j = 8; j < 15; j++)
	if (n & (1<<j)) k++;
    if ((k & 1) == ((n>>15)&1))
	    mask|=0xFF00;
   return mask;
}

int decodebit(unsigned char *data, int threshold)
{
    int i, sum = 0;
    for (i = 0; i < 23; i++)
	sum += data[i];
    return (sum > threshold*23);
}

int decode(unsigned char *vbiline)
{
    int max[7], min[7], val[7], i, clk, tmp, sample, packedbits = 0;
    
    for (clk=0; clk<7; clk++)
	max[clk] = min[clk] = val[clk] = -1;
    clk = tmp = 0;
    i=30;

    while (i < 600 && clk < 7) {	/* find and lock all 7 clocks */
	sample = vbiline[i];
	if (max[clk] < 0) { /* find maximum value before drop */
	    if (sample > 85 && sample > val[clk])
		(val[clk] = sample, tmp = i);	/* mark new maximum found */
	    else if (val[clk] - sample > 30)	/* far enough */
		(max[clk] = tmp, i = tmp + 10);
	} else { /* find minimum value after drop */
	    if (sample < 85 && sample < val[clk])
		(val[clk] = sample, tmp = i);	/* mark new minimum found */
	    else if (sample - val[clk] > 30)	/* searched far enough */
		(min[clk++] = tmp, i = tmp + 10);
	}
	i++;
    } 

   i=min[6]=min[5]-max[5]+max[6]; 
   
    if (clk != 7 || vbiline[max[3]] - vbiline[min[5]] < 45)		/* failure to locate clock lead-in */
	return -1;

#ifdef XWIN
   for (clk=0;clk<7;clk++)
   {
	   XDrawLine(dpy,Win,WinGC,min[clk]/2,0,min[clk]/2,128);
	   XDrawLine(dpy,Win,WinGC1,max[clk]/2,0,max[clk]/2,128);
   }
   XFlush(dpy);
#endif
 
    
    /* calculate threshold */
    for (i=0,sample=0;i<7;i++)
	    sample=(sample + vbiline[min[i]] + vbiline[max[i]])/3;

    for(i=min[6];vbiline[i]<sample;i++);

#ifdef XWIN
   for (clk=i;clk<i+57*18;clk+=57)
	   XDrawLine(dpy,Win,WinGC,clk/2,0,clk/2,128);
   XFlush(dpy);
#endif
 
    
    tmp = i+57;
    for (i = 0; i < 16; i++)
	if(decodebit(&vbiline[tmp + i * 57], sample))
	    packedbits |= 1<<i;
    return packedbits&parityok(packedbits);
} /* decode */

int XDSdecode(int data)
{
	int b1, b2, length;
	if (data == -1)
		return -1;
	
	b1 = data & 0x7F;
	b2 = (data>>8) & 0x7F;

	if (b1 < 15) // start packet 
	{
		mode = b1;
		type = b2;
		infochecksum = b1 + b2 + 15;
		if (mode > 8 || type > 20)
		{
//			printf("%% Unsupported mode %s(%d) [%d]\n",modes[(mode-1)>>1],mode,type);
			mode=0; type=0;
		}
			infoptr = newinfo[mode][type];
	}
	else if (b1 == 15) // eof (next byte is checksum)
	{
//		if (mode == 0)
//		{
//			length=infoptr - newinfo[0][0];
//			infoptr[1]=0;
//			printf("LEN: %d\n",length);
//			for (y=0;y<length;y++)
//				printf(" %03d",newinfo[0][0][y]);
//			printf(" --- %s\n",newinfo[0][0]);
//		}
		if (mode == 0) return 0;
		if (b2 != 128-((infochecksum%128)&0x7F)) return 0;

		length = infoptr - newinfo[mode][type];

		//don't bug the user with repeated data
		//only parse it if it's different
		if (strncmp(info[mode][type],newinfo[mode][type],length))
		{
			infoptr = info[mode][type];
			memcpy(infoptr,newinfo[mode][type],length);
			switch ((mode<<8) + type)
			{
				case 0x0101:
					printf("%% TIMECODE: %d/%02d %d:%02d\n",
					infoptr[3]&0x0f,infoptr[2]&0x1f,infoptr[1]&0x1f,infoptr[0]&0x3f);
				case 0x0102:
					if ((infoptr[1]&0x3f)>5)
						break;
					printf("%%   LENGTH: %d:%02d:%02d of %d:%02d:00\n",
					infoptr[3]&0x3f,infoptr[2]&0x3f,infoptr[4]&0x3f,infoptr[1]&0x3f,infoptr[0]&0x3f);
					break;
				case 0x0103:
					infoptr[length] = 0;
					printf("%%    TITLE: %s\n",infoptr);
					break;
				case 0x0105:
					printf("%%   RATING: %s (%d)",ratings[infoptr[0]&0x07],infoptr[0]);
					if ((infoptr[0]&0x07)>0)
					{
						if (infoptr[0]&0x20) printf(" VIOLENCE");
						if (infoptr[0]&0x10) printf(" SEXUAL");
						if (infoptr[0]&0x08) printf(" LANGUAGE");
					}
					printf("\n");
					break;
				case 0x0501:
					infoptr[length] = 0;
					printf("%%  NETWORK: %s\n",infoptr);
					break;
				case 0x0502:
					infoptr[length] = 0;
					printf("%%     CALL: %s\n",infoptr);
					break;
				case 0x0701:
					printf("%% CUR.TIME: %d:%02d %d/%02d/%04d UTC\n",infoptr[1]&0x1F,infoptr[0]&0x3f,infoptr[3]&0x0f,infoptr[2]&0x1f,(infoptr[5]&0x3f)+1990);
					break;
				case 0x0704: //timezone
					printf("%% TIMEZONE: UTC-%d\n",infoptr[0]&0x1f);
					break;
				case 0x0104: //program genere
					break;
				case 0x0110:
				case 0x0111:
				case 0x0112:
				case 0x0113:
				case 0x0114:
				case 0x0115:
				case 0x0116:
				case 0x0117:
					infoptr[length+1] = 0;
					printf("%%     DESC: %s\n",infoptr);
					break;
			}
			fflush(stdout);
		}
		mode = 0; type = 0;
	}
	else if( (infoptr - newinfo[mode][type]) < 250 ) // must be a data packet, check if we're in a supported mode
	{
		infoptr[0] = b1; infoptr++;
		infoptr[0] = b2; infoptr++;
		infochecksum += b1 + b2;
	}
	return 0;
}
void webtv_check(char * buf,int len)
{
	unsigned long   sum;
	unsigned long   nwords;
	unsigned short  csum=0;
	char temp[9];
	int nbytes=0;
	
	while (buf[0]!='<' && len > 0)
	{
		buf++; len--;
	}
	while (nbytes+6 <= len)
	{
		if (buf[nbytes] == '[' && buf[nbytes+5] == ']' && buf[nbytes+6] != ']')
			break;
		else
			nbytes++;
	}
	nwords = nbytes >> 1;
	sum = 0;
	while (nwords-- > 0) {
		sum += *buf++ << 8;
		sum += *buf++;
	}
	if (nbytes & 1) {
		sum += *buf << 8;
	}
	csum = (unsigned short)(sum >> 16);
	while(csum !=0) {
		sum = csum + (sum & 0xffff);
		csum = (unsigned short)(sum >> 16);
	}
	sprintf(temp,"%04X\n",(int)~sum&0xffff);
	buf++;
	if(!strncmp(buf,temp,4))
	{
		buf[5]=0;
		printf("WEBTV: %s\n",buf-nbytes-1);
		buf[-nbytes]=0;
	}
}
int CCdecode(int data)
{
	int b1, b2, row, len, x;
	if (data == -1)
		return -1;
	b1 = data & 0x7f;
	b2 = (data>>8) & 0x7f;
//	if (b1 || b2)
//		fprintf(stderr,"%c%c",b1,b2);
	len = strlen(ccbuf[ccmode][ccdbl]);


	
	if (b1&0x60 && data != lastcode) // text
	{
		ccbuf[ccmode][ccdbl][len++]=b1;
		if (b2&0x60) ccbuf[ccmode][ccdbl][len++]=b2;
		if (b1 == ']' || b2 == ']')
			webtv_check(ccbuf[ccmode][ccdbl],len);
	}
	else if ((b1&0x10) && (b2>0x1F) && (data != lastcode)) //codes are always transmitted twice
	{
		if (ccmode!=((b1>>3)&1)+1)
		{
			ccmode=((b1>>3)&1)+1;
			len = strlen(ccbuf[ccmode][ccdbl]);
		}
		if (b2 & 0x40)	//preamble address code
		{
			row=rowdata[((b1<<1)&14)|((b2>>5)&1)];
			if (len!=0)
				ccbuf[ccmode][ccdbl][len++]='\n';
			if (b2&0x10)
				for (x=0;x<(b2&0x0F)<<1;x++)
					ccbuf[ccmode][ccdbl][len++]=' ';
		}
		else
		{
			switch (b1 & 0x07)
			{
				case 0x00:	//attribute
					printf("<ATTRIBUTE %d %d>\n",b1,b2);
					break;
				case 0x01:	//midrow or char
					switch (b2&0x70)
					{
						case 0x20: //midrow attribute change
							switch (b2&0x0e)
							{
								case 0x00:
									strcat(ccbuf[ccmode][ccdbl],"\33[0m ");
									break;
								case 0x0e:
									strcat(ccbuf[ccmode][ccdbl],"\33[36m ");
									break;
							}
							if (b2&0x01)
								strcat(ccbuf[ccmode][ccdbl],"\33[4m");
							else
								strcat(ccbuf[ccmode][ccdbl],"\33[24m");
							break;
						case 0x30: //special character..
							strcat(ccbuf[ccmode][ccdbl],specialchar[b2&0x0f]);
							break;
					}
					break;
				case 0x04:	//misc
				case 0x05:	//misc + F
					switch (b2)
					{
						case 0x20: //resume caption (new caption)
							ccdbl=2;	//double buffer
							memset(ccbuf[ccmode][ccdbl],0,255);
							break;
						case 0x21: //backspace
							ccbuf[ccmode][ccdbl][len--]=0;
							break;
						case 0x25: //rollup
						case 0x26:
						case 0x27:
						case 0x29: //resume direct caption
							ccdbl=1;
							break;
						case 0x2A: // test restart
							ccdbl=1;
							memset(ccbuf[ccmode][ccdbl],0,255);
							break;
						case 0x2B: //resume text display
							ccdbl=1;
							break;
						case 0x2C: //erase displayed memory
							//printf("%s\33[m\n",ccbuf[ccmode][ccdbl]);
							//memset(ccbuf[ccmode][1],0,255);
							break;
						case 0x2D: //carriage return
							printf("%s\33[m\n",ccbuf[ccmode][ccdbl]);
							memset(ccbuf[ccmode][ccdbl],0,255);
							break;
						case 0x2E: //erase non-displayed memory
							memset(ccbuf[ccmode][2],0,255);
							break;
						case 0x2F: //end caption + swap memory
							memcpy(ccbuf[ccmode][1],ccbuf[ccmode][2],255);
							ccdbl=1;
							printf("%s\33[m\n",ccbuf[ccmode][ccdbl]);
							memset(ccbuf[ccmode][ccdbl],0,255);
							break;
					}
					break;
				case 0x07:	//misc (TAB)
					for(x=0;x<(b2&0x03);x++)
						ccbuf[ccmode][ccdbl][len++]=' ';
					break;
			}
		}
	}
	lastcode=data;
	return 0;
}
int RAW(int data)
{
	int b1, b2;
	if (data == -1)
		return -1;
	b1 = data & 0x7f;
	b2 = (data>>8) & 0x7f;
	fprintf(stderr,"%c%c",b1,b2);
	return 0;
}
#ifdef XWIN
unsigned long getColor(char *colorName, float dim)
{
	XColor Color;
	XWindowAttributes Attributes;

	XGetWindowAttributes(dpy, Root, &Attributes);
	Color.pixel = 0;

	XParseColor (dpy, Attributes.colormap, colorName, &Color);
	Color.red=(unsigned short)(Color.red/dim);
	Color.blue=(unsigned short)(Color.blue/dim);
	Color.green=(unsigned short)(Color.green/dim);
	Color.flags=DoRed | DoGreen | DoBlue;
	XAllocColor (dpy, Attributes.colormap, &Color);

	return Color.pixel;
}
#endif
int main(int argc,char **argv)
{
   unsigned char buf[65536];
   int vbifd;
   fd_set rfds;


#ifdef XWIN
int y;
dpy=XOpenDisplay(dpyname);
Root=DefaultRootWindow(dpy);
Win = XCreateSimpleWindow(dpy, Root, 10, 10, 1024, 128,0,0,0);
WinGC = XCreateGC(dpy, Win, 0, NULL);
WinGC0 = XCreateGC(dpy, Win, 0, NULL);
WinGC1 = XCreateGC(dpy, Win, 0, NULL);
XSetForeground(dpy, WinGC, getColor("blue",1));
XSetForeground(dpy, WinGC0, getColor("green",1));
XSetForeground(dpy, WinGC1, getColor("red",1));
XMapWindow(dpy, Win);
if (!argv[1]) {
	printf("required line argument missing\n");
	return 0;
}
y=atoi(argv[1]);
#endif
	
   if ((vbifd = open("/dev/vbi", O_RDONLY)) < 0) {
	perror("/dev/vbi");
	exit(1);
   }
   
   //mainloop
   while(1){
	FD_ZERO(&rfds);
	FD_SET(vbifd, &rfds);
	select(FD_SETSIZE, &rfds, NULL, NULL, NULL);
	if (FD_ISSET(vbifd, &rfds)) {
	    if (read(vbifd, buf , 65536)!=65536)
		    printf("read error\n");
#ifdef XWIN
		XClearArea(dpy,Win,0,0,1024,128,0);
		XDrawLine(dpy,Win,WinGC1,0,128-85/2,1024,128-85/2);
		for (x=0;x<1024;x++)
			XDrawLine(dpy,Win,WinGC0,x,128-buf[2048 * y+x*2]/2,x+1,128-buf[2048 * y+x*2+2]/2);
		RAW(decode(&buf[2048 * y]));
#else
		XDSdecode(decode(&buf[2048 * 27]));
		CCdecode(decode(&buf[2048 * 11]));
#endif
#ifdef XWIN	
		XFlush(dpy);
		usleep(100);
#endif
	    
	}
   }
   return 0;
}
