//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.cms.v1;

import java.util.Vector;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.OctetString;
import com.dstc.security.x509.AlgorithmId;

/**
 * <p> Implements the ASN.1 structure RecipientInfo.
 *
 * <pre>
 *     RecipientInfo ::= CHOICE {
 *        ktri KeyTransRecipientInfo,
 *        kari [1] KeyAgreeRecipientInfo, 
 *        mlri [2] MailListRecipientInfo }
 *         
 *     EncryptedKey ::= OCTET STRING
 *
 *     KeyTransRecipientInfo ::= SEQUENCE {
 *        version CMSVersion,  -- always set to 0 or 2
 *        rid RecipientIdentifier, 
 *        keyEncryptionAlgorithm KeyEncryptionAlgorithmIdentifier,
 *        encryptedKey EncryptedKey }
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class RecipientInfo extends Sequence
{
  private int version;
  private IssuerAndSerialNumber rid;
  private AlgorithmId keyEncryptionAlgo;
  private byte[] encryptedKey;

  /**
   * Constructs a RecipientInfo from an IssuerAndSerialNumber,
   * an encryption AlgorithmId and the encrypted key bytes
   */
  public RecipientInfo(IssuerAndSerialNumber recipientId,
                        AlgorithmId encAlgo, byte[] encKey)
  {
    addComponent(new com.dstc.security.asn1.Integer(version));

    this.rid = recipientId;
    addComponent(recipientId);

    this.keyEncryptionAlgo = encAlgo;
    addComponent(encAlgo);

    this.encryptedKey = encKey;
    addComponent(new OctetString(encKey));
  }

  /**
   * Constructs a RecipientInfo from a DER encoding
   */
  public RecipientInfo(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);
    int i = 0;
    
    this.version = 
      ((com.dstc.security.asn1.Integer)components.elementAt(i++)).getInt();

    this.rid = 
      new IssuerAndSerialNumber(((Asn1)components.elementAt(i++)).encode());

    this.keyEncryptionAlgo = 
      new AlgorithmId(((Asn1)components.elementAt(i++)).encode());

    this.encryptedKey = ((OctetString)components.elementAt(i)).getBytes();
  }

  /**
   * Returns the encrypted key bytes for this RecipientInfo
   */
  public byte[] getEncryptedKey()
  {
    return (byte[])encryptedKey.clone();
  }

  /**
   * Returns the IssuerAndSerialNumber for this RecipientInfo
   */
  public IssuerAndSerialNumber getRecipientIdentifier()
  {
    return this.rid;
  }

  /**
   * Returns the encryption AlgorithmId for this RecipientInfo
   */
  public AlgorithmId getKeyEncryptionAlgo()
  {
    return this.keyEncryptionAlgo;
  }
}
