////////////////////////////////////////////////////////////////////////////
//
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
//
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC
// file. If your distribution of this software does not contain
// a LICENSE.DOC file then you have no rights to use this
// software in any manner and should contact DSTC at the address
// below to determine an appropriate licensing arrangement.
//
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
//
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
//
////////////////////////////////////////////////////////////////////////////

package com.dstc.security.kerberos;

import java.util.Date;

import com.dstc.security.kerberos.crypto.KerberosCipher;
import com.dstc.security.kerberos.crypto.KeyMaterial;
import com.dstc.security.kerberos.crypto.CryptoException;
import com.dstc.security.kerberos.creds.CredentialInfo;
import com.dstc.security.kerberos.creds.KerberosTicket;
import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptedData;
import com.dstc.security.kerberos.v5.EncKDCRepPart;
import com.dstc.security.kerberos.v5.KrbTGSRep;
import com.dstc.security.kerberos.v5.EncTGSRepPart;
import com.dstc.security.kerberos.v5.BadMessageTypeException;
import com.dstc.security.kerberos.v5.BadVersionException;

import com.dstc.security.asn1.Asn1Exception;

/**
 * Representation of a Ticket Granting Service response message, used by
 * the KDC and its clients.
 *
 * @author Ming Yung
 */
public class TGSResponse extends KerberosMessage
{
  private KrbTGSRep rep = null;
  private KerberosCipher cipher = null;
  private EncTGSRepPart encTGSRepPart = null;
  private EncryptedData encData = null;

  /**
   * Constructs a TGSResponse from a DER encoding of a KrbTGSRep
   */
  public TGSResponse(byte[] encoded) 
    throws Asn1Exception, BadVersionException, BadMessageTypeException
  {
    this.rep = new KrbTGSRep(encoded);
    this.encData = rep.getKDCRep().getEncrypted();
  }

  public void initDecrypt(KeyMaterial cred) throws CryptoException
  {
    KerberosCipher cipher = new KerberosCipher(cred, null, null);
    cipher.initDecrypt();

    this.cipher = cipher;
  }

  /**
   * Decrypts this TGSResponse
   */
  public void decrypt() 
    throws Asn1Exception, CryptoException, KerberosException, 
           BadMessageTypeException
  {
    if (this.cipher == null)
      throw new KerberosException("Cipher not yet set");

    this.encTGSRepPart 
      = new EncTGSRepPart(this.cipher.decrypt(encData));
  }

  /**
   * Returns the ticket sent back by the KDC
   */
  public KerberosTicket getTicket()
  {
    return new KerberosTicket(rep.getKDCRep().getTicket());
  }

  /**
   * Returns the nonce sent back by the KDC
   */
  public int getNonce()
  {
    return this.encTGSRepPart.getEncKDCRepPart().getNonce();
  }

  /**
   * Returns the key expiration time
   */
  public Date getKeyExpiryTime()
  {
    return this.encTGSRepPart.getEncKDCRepPart().getKeyExpiryTime().getDate();
  }

  /**
   * Returns the server name for this EncKDCRepPart
   */
  public String getServerName()
  {
    return this.encTGSRepPart.getEncKDCRepPart().getServerName().toString();
  }

  /**
   * Returns the server realm for this EncKDCRepPart
   */
  public String getServerRealm()
  {
    return this.encTGSRepPart.getEncKDCRepPart().getServerRealm().getName();
  }

  /**
   *
   */
  public CredentialInfo getCredentialInfo()
  {
    EncKDCRepPart kdcRepPart = this.encTGSRepPart.getEncKDCRepPart();

    return new CredentialInfo(
        kdcRepPart.getEncryptionKey(), 
        this.rep.getKDCRep().getClientRealm(),
        this.rep.getKDCRep().getClientName(),
        kdcRepPart.getTicketFlags(),
        kdcRepPart.getAuthTime(),
        kdcRepPart.getStartTime(),
        kdcRepPart.getEndTime(),
        kdcRepPart.getRenewTill(),
        kdcRepPart.getServerRealm(),
        kdcRepPart.getServerName(),
        kdcRepPart.getClientAddresses());
  }

  public byte[] getEncoded()
  {
    return this.rep.encode();
  }
}
