//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.creds;

import java.io.*;
import java.util.*;

import com.dstc.security.kerberos.crypto.KerberosCipher;
import com.dstc.security.kerberos.crypto.KeyMaterial;

/**
 * Representation of a credential entry in a Kerberos keytab
 *
 * @version 0.98, 98/12/22
 * @author Ming Yung
 */
public class KeyTabEntry implements KeyMaterial
{
  private CredentialName name;
  private int principalType;
  private int timestamp;
  private byte kvno;
  private int keyType;
  private byte[] key;

  /**
   * Constructs a KeyTabEntry from the InputStream
   */
  public KeyTabEntry(CCInputStream cis) throws IOException
  {
    this.name = new CredentialName(cis); 
    this.principalType = cis.readInt();
    this.timestamp = cis.readInt();
    this.kvno = cis.readByte();
    this.keyType = cis.readShort();
    this.key = new byte[cis.readShort()];
    cis.read(this.key);
  }

  /**
   * Returns the service name for this KeyTabEntry
   */
  public String getServiceName()
  {
    return this.name.print();
  }

  /**
   * Returns the key for the given service in this keyTabEntry
   */
  public byte[] getKeyBytes()
  {
    return this.key;
  }

  public int getKeyType()
  {
    return this.keyType;
  }

  /**
   * Returns a String representation of this KeyTabEntry
   */
  public String toString()
  {
    StringBuffer sb = new StringBuffer();
    sb.append(this.name.print());
    sb.append("\nPrincipalType: " + this.principalType);
    sb.append("\nTimeStamp: ");
    sb.append((new Date((long)this.timestamp*1000)).toString());
    sb.append("\nKVNO: " + this.kvno);
    sb.append("\nKey Type: " + this.keyType);
    sb.append("\nKey: " + toHexString(this.key));
    sb.append("\n");
    return sb.toString();
  }

  private String toHexString(byte bytes[])
  {
    StringBuffer sb = new StringBuffer();
    for (int i=0; i<bytes.length; i++)
    {
      sb.append(java.lang.Integer.toHexString(bytes[i] & 0xff));
      sb.append("  ");
    }
    return sb.toString();
  }

  protected class CredentialName
  {
    private String realm;
    private Vector nameComponents;

    protected CredentialName()
    {
      this.nameComponents = new Vector();
    }

    protected CredentialName(CCInputStream cis) throws IOException
    {
      this();
      int numComps = cis.readShort();
  
      byte realmBytes[] = new byte[cis.readShort()];
      cis.read(realmBytes);
      this.realm = new String(realmBytes);
  
      for (int i=0; i<numComps; i++)
      {
       byte comp[] = new byte[cis.readShort()];
       cis.read(comp);
       this.nameComponents.addElement(new String(comp));
      }
    }

    protected String print()
    {
      StringBuffer sb = new StringBuffer();

      int i = 0;
      while (true)
      {
         sb.append((String)nameComponents.elementAt(i++));
         if (i == nameComponents.size()) break;
         sb.append("/");
      }
      sb.append("@" + this.realm);
      return sb.toString();
    }
  }
}
