//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.gssapi;

import java.util.Arrays;
import java.net.InetAddress;

/**
 * The GSS-API accommodates the concept of caller-provided channel
 * binding information. Channel bindings are used to strengthen the
 * quality with which peer entity authentication is provided during
 * context establishment. They enable the GSS-API callers to bind the
 * establishment of the security context to relevant characteristics
 * like addresses or to application specific data.
 *
 * <p>The caller initiating the security context must determine the
 * appropriate channel binding values to set in the GSSContext object.
 * The acceptor must provide an identical binding in order to validate
 * that received tokens possess correct channel-related characteristics.
 *
 * <p>Use of channel bindings is optional in GSS-API. Since channel-binding
 * information may be transmitted in context establishment tokens,
 * applications should therefore not use confidential data as channel-
 * binding components.
 *
 * @author Anthony Ho
 */
public class ChannelBinding
{
  private InetAddress initAddr;
  private InetAddress acceptAddr;
  private byte[] appData;

  /**
   * Create a ChannelBinding object with user supplied address information
   * and data.  "null" values can be used for any fields which the
   * application does not want to specify.
   */
  public ChannelBinding(InetAddress initAddr, InetAddress acceptAddr,
                        byte[] appData)
  {
    this.initAddr = initAddr;
    this.acceptAddr = acceptAddr;
    this.appData = appData;
  }

  /**
   * Creates a ChannelBinding object without any addressing information.
   */
  public ChannelBinding(byte[] appData)
  {
    this.appData = appData;
  }

  /**
   * Returns the initiator's address for this channel binding. "null" is
   * returned if the address has not been set.
   */
  public InetAddress getInitiatorAddress()
  {
    return this.initAddr;
  }

  /**
   * Returns the acceptor's address for this channel binding. "null" is
   * returned if the address has not been set.
   */
  public InetAddress getAcceptorAddress()
  {
    return this.acceptAddr;
  }

  /**
   * Returns application data being used as part of the ChannelBinding.
   * "null" is returned if no application data has been specified for the
   * channel binding.
   */
  public byte[] getApplicationData()
  {
    return appData;
  }

  /**
   * Returns "true" if two channel bindings match.
   */
  public boolean equals(Object obj)
  {
    return ((obj instanceof ChannelBinding)
            && this.initAddr.equals(((ChannelBinding)obj).initAddr)
            && this.acceptAddr.equals(((ChannelBinding)obj).acceptAddr)
            && Arrays.equals(this.appData, ((ChannelBinding)obj).appData));
  }
}
