//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.gssapi;

import java.io.InputStream;
import com.dstc.security.asn1.Asn1;

/** This class represents Universal Object Identifiers (Oids) and their
 *  associated operations.
 *
 * <p>Oids are hierarchically globally-interpretable identifiers used
 * within the GSS-API framework to identify mechanisms and name formats.
 *
 * @author Anthony Ho
 */
public class Oid
{
  private String strOid;
  private com.dstc.security.asn1.Oid asn1Oid;

  /**
   * Creates an Oid object from a string representation of its integer
   * components (e.g. "1.2.840.113554.1.2.2").
   */
  public Oid(String strOid)
  {
    this.strOid = strOid;
    this.asn1Oid = new com.dstc.security.asn1.Oid(this.strOid);
  }

  /**
   * Creates an Oid object from its DER encoding. This refers to the full
   * encoding including tag and length.  The structure and encoding of
   * Oids is defined in ISOIEC-8824 and ISOIEC-8825.  This method is
   * identical in functionality to its byte array counterpart.
   */
  public Oid(InputStream derOid) throws GSSException
  {
    try
    {
      byte[] temp = new byte[Token.readLength(derOid)];
      derOid.read(temp);
      this.asn1Oid = (com.dstc.security.asn1.Oid)Asn1.decode(temp);
      this.strOid = this.asn1Oid.getOid();
    }
    catch (Exception e)
    {
      throw new GSSException(GSSException.FAILURE);
    }
  }

  /**
   * Creates an Oid object from its DER encoding. This refers to the full
   * encoding including tag and length.  The structure and encoding of
   * Oids is defined in ISOIEC-8824 and ISOIEC-8825.  This method is
   * identical in functionality to its stream counterpart.
   */
  public Oid(byte[] DEROid) throws GSSException
  {
    try
    {
      this.asn1Oid = (com.dstc.security.asn1.Oid)Asn1.decode(DEROid);
      this.strOid = this.asn1Oid.getOid();
    }
    catch (Exception e)
    {
      throw new GSSException(GSSException.FAILURE);
    }
  }

  /**
   * Returns a string representation of the oid's integer components in
   * dot separated notation (e.g. "1.2.840.113554.1.2.2").
   */
  public String toString()
  {
    return this.strOid;
  }

  /**
   * Returns a string representation of the Oid's integer components in
   * the format specified within RFC 2078 (e.g. "{ 1 2 840 113554 1 2 2
   * }").
   */
  public String toRFC2078String()
  {
    StringBuffer sb = new StringBuffer();
    
    sb.append("{ ");
    sb.append(this.strOid.replace('.', ' '));
    sb.append(" }");
    return sb.toString();
  }

  /**
   * Returns "true" if the two Oid objects represent the same oid value.
   */
  public boolean equals(Object Obj)
  {
    return (Obj instanceof Oid
            && ((Oid)Obj).strOid.compareTo(this.strOid) == 0);
  }

  /**
   * Returns the full ASN.1 DER encoding for this oid object, which
   * includes the tag and length.
   */
  public byte[] getDER()
  {
    return this.asn1Oid.encode();
  }

  /**
   * A utility method to test if an Oid object is contained within the
   * supplied Oid object array.
   */
  public boolean containedIn(Oid[] oids)
  {
    for (int i = 0; i < oids.length; i++)
    {
      if (this.equals(oids[i]))
        return true;
    }
    return false;  
  }
}
