//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5;

import java.util.Vector;

import com.dstc.security.kerberos.v5.BadVersionException;
import com.dstc.security.kerberos.v5.BadMessageTypeException;
import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptedData;
import com.dstc.security.kerberos.v5.creds.Ticket;

import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Explicit;
import com.dstc.security.asn1.Asn1;

/**
 * <p>A class representing the Kerberos ASN.1 structure AP-REQ.
 *
 * <pre>
 *   AP-REQ ::=  [APPLICATION 14] SEQUENCE {
 *                   pvno[0]                       INTEGER,
 *                   msg-type[1]                   INTEGER,
 *                   ap-options[2]                 APOptions,
 *                   ticket[3]                     Ticket,
 *                   authenticator[4]              EncryptedData
 *               }
 * </pre>
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class APReq extends Explicit
{
  public static final int TAG_KRB_AP_REQ = 14;
  private static final int pvno = 5;

  private int msgType = TAG_KRB_AP_REQ;
  private APOptions apOptions;
  private Ticket ticket;
  private EncryptedData encAuthenticator;

  /**
   * Constructs an APReq from an APOptions, a ticket and
   * an encrypted authenticator
   */
  public APReq(APOptions apOptions, Ticket ticket, 
               EncryptedData encAuthenticator)
  {
    super(Asn1.TAG_CLASS_APPLICATION, TAG_KRB_AP_REQ);
    Sequence seq = new Sequence();

    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(pvno));
    seq.addComponent(exp);

    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    exp.addComponent(new com.dstc.security.asn1.Integer(msgType));
    seq.addComponent(exp);

    this.apOptions = apOptions;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 2);
    exp.addComponent(apOptions);
    seq.addComponent(exp);

    this.ticket = ticket;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 3);
    exp.addComponent(ticket);
    seq.addComponent(exp);

    this.encAuthenticator = encAuthenticator;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 4);
    exp.addComponent(encAuthenticator);
    seq.addComponent(exp);

    addComponent(seq);
  }

  /**
   * Constructs an APReq from a DER encoding
   */
  public APReq(byte[] encoded) 
    throws Asn1Exception, BadVersionException, BadMessageTypeException
  {
    doDecode(encoded);
    if (this.getTagNumber() != TAG_KRB_AP_REQ)
      throw new BadMessageTypeException("Wrong tag for an APReq");

    Sequence seq = (Sequence)this.getComponent();
    Vector comps = seq.getComponents();

    int i = 0;

    Explicit exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 0)
      throw new Asn1Exception("Bad Tag Number");
    if (((com.dstc.security.asn1.Integer)exp.getComponent()).getInt() != pvno)
      throw new BadVersionException("Not version 5");

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 1)
      throw new Asn1Exception("Bad Tag number");
    if (((com.dstc.security.asn1.Integer)exp.getComponent()).getInt() 
        != msgType)
      throw new BadMessageTypeException("Wrong msgType for an APReq");

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 2)
      throw new Asn1Exception("Bad Tag number");
    this.apOptions = new APOptions(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 3)
      throw new Asn1Exception("Bad Tag number");
    this.ticket = new Ticket(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(i++);
    if (exp.getTagNumber() != 4)
      throw new Asn1Exception("Bad Tag number");
    this.encAuthenticator = new EncryptedData(exp.getComponent().encode());
  }

  /**
   * Returns the APOptions object for this request
   */
  public APOptions getAPOptions() 
  {
    return this.apOptions;
  }

  /**
   * Returns the Ticket for this Request
   */
  public Ticket getTicket()
  {
    return this.ticket;
  }

  /**
   * Returns the Authenticator for this Request as an EncryptedData
   */
  public EncryptedData getEncrypted()
  {
    return this.encAuthenticator;
  }
}
