//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5;

import java.util.Date;
import java.util.Vector;

import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptionKey;

import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Explicit;
import com.dstc.security.asn1.Asn1;

/**
 * <p>A class representing the Kerberos ASN.1 structure EncAPRepPart.
 *
 * <pre>
 *   EncAPRepPart ::=  [APPLICATION 27]  SEQUENCE {
 *                        ctime[0]            KerberosTime,
 *                        cusec[1]            INTEGER,
 *                        subkey[2]           EncryptionKey OPTIONAL,
 *                        seq-number[3]       INTEGER OPTIONAL
 *                      }
 * <pre>
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class EncAPRepPart extends Explicit
{
  public static final int TAG_KRB_ENC_AP_REP_PART = 27;

  private KerberosTime ctime;
  private int cusec;
  private EncryptionKey key = null;
  private int seqNum;

  /**
   * Constructs an EncAPRepPart 
   */
  public EncAPRepPart(KerberosTime ctime, int cusec, EncryptionKey key,
                      int seqNum)
  {
    super(Asn1.TAG_CLASS_APPLICATION, TAG_KRB_ENC_AP_REP_PART);

    Sequence seq = new Sequence();

    this.ctime = ctime;
    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(ctime);
    seq.addComponent(exp);

    this.cusec = cusec;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    exp.addComponent(new com.dstc.security.asn1.Integer(cusec));
    seq.addComponent(exp);

    this.key = key;
    if (key != null)
    {
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 2);
      exp.addComponent(key);
      seq.addComponent(exp);
    }

    this.seqNum = seqNum;
    if (seqNum != -1)
    {
      exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 3);
      exp.addComponent(new com.dstc.security.asn1.Integer(seqNum));
      seq.addComponent(exp);
    }

    addComponent(seq);
  }

  /**
   * Constructs an EncAPRepPart from a DER encoding
   */
  public EncAPRepPart(byte encoded[]) 
    throws Asn1Exception, BadMessageTypeException
  {
    doDecode(encoded);
    if (this.getTagNumber() != TAG_KRB_ENC_AP_REP_PART)
      throw new BadMessageTypeException("Wrong tag for an EncAPRepPart");

    int i=0;
    
    Sequence seq = (Sequence)this.getComponent();
    Vector comps = seq.getComponents();

    Explicit exp = (Explicit)comps.elementAt(i++);
    this.ctime = new KerberosTime(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(i++);
    this.cusec 
      = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();

    if (i >= comps.size()) return;

    exp = (Explicit)comps.elementAt(i);
    if (exp.getTagNumber() == 2)
    {
      this.key = new EncryptionKey(exp.getComponent().encode());
      i++;
    }
    
    if (i >= comps.size()) return;

    exp = (Explicit)comps.elementAt(i);
    if (exp.getTagNumber() == 3)
    {
      this.seqNum 
        = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();
    }
  }

  /**
   * Returns the client's time for this EncAPRepPart
   */
  public KerberosTime getClientTime()
  {
    return this.ctime;
  }

  /**
   * Returns the cusec for this EncAPRepPart
   */
  public int getCusec()
  {
    return this.cusec;
  }

  /**
   * Returns the encryption sub-session key for this EncAPRepPart
   */
  public EncryptionKey getEncryptionKey()
  {
    return this.key;
  }

  /**
   * Returns the sequence number for this EncAPRepPart
   */
  public int getSequenceNumber()
  {
    return this.seqNum;
  }
}
