//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5.base;

import java.util.StringTokenizer;
import java.util.Iterator;

import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.OctetString;
import com.dstc.security.asn1.Explicit;

/**
 * <p>Class representing the Kerberos ASN.1 structure HostAddress.
 *
 * <pre>
 *    HostAddress ::=  SEQUENCE  {
 *                        addr-type[0]             INTEGER,
 *                        address[1]               OCTET STRING
 *                     }
 * </pre>
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class HostAddress extends Sequence
{
  private int addressType = 2;
  private byte[] address;

  /**
   * Constructs a HostAddress from a DER encoding
   */
  public HostAddress(byte[] encoded) throws Asn1Exception
  {
    doDecode(encoded);

    Explicit exp = (Explicit)components.elementAt(0);
    if (exp.getTagNumber() != 0)
      throw new Asn1Exception("Bad Tag number");
    this.addressType 
      = ((com.dstc.security.asn1.Integer)exp.getComponent()).getInt();

    exp = (Explicit)components.elementAt(1);
    if (exp.getTagNumber() != 1)
      throw new Asn1Exception("Bad Tag number");
    this.address = ((OctetString)exp.getComponent()).getBytes();
  }
    
  /**
   * Constructs a HostAddress from an address string
   */
  public HostAddress(int type, byte[] address)
  {
    this.addressType = type;
    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(this.addressType));
    addComponent(exp);

    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    this.address = address;
    exp.addComponent(new OctetString(address));
    addComponent(exp);
  }

  /**
   * Constructs a HostAddress from an address string
   */
  public HostAddress(String addressString)
  {
    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(addressType));
    addComponent(exp);

    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    address = toIP(addressString);
    exp.addComponent(new OctetString(address));
    addComponent(exp);
  }

  /**
   * Converts a stringified address with "." as the delimiter into a
   * corresponding array of bytes
   */
  private static final byte[] toIP(String add)
  {
    StringTokenizer tknz = new StringTokenizer (add, ".", false);
    byte retval[] = new byte[tknz.countTokens ()];
    for (int i=0; i<retval.length; i++)
    {
      retval[i] = (byte)java.lang.Integer.parseInt(tknz.nextToken ());
    }
    return retval;
  }

  /**
   */
  private static final String ipToString(byte[] add)
  {
    StringBuffer sb = new StringBuffer(String.valueOf(add[0]));
    for (int i=1; i<add.length; i++)
    {
      sb.append(".").append(String.valueOf(add[i]));
    }
    return sb.toString();
  }

  /**
   * Returns the address type for this HostAddress
   */
  public int getAddressType()
  {
    return this.addressType;
  }

  /**
   * Returns the address for this HostAddress as a byte array
   */
  public byte[] getAddress()
  {
    return this.address;
  }

  public String toString()
  {
    return ipToString(this.address);
  }
}
