//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.kerberos.v5.creds;

import java.util.Vector;

import com.dstc.security.kerberos.v5.BadVersionException;
import com.dstc.security.kerberos.v5.BadMessageTypeException;
import com.dstc.security.kerberos.v5.base.*;
import com.dstc.security.kerberos.v5.crypto.EncryptedData;

import com.dstc.security.asn1.Explicit;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;

/**
 * <p>A class representing the Kerberos ASN.1 structure Ticket.
 *
 * <pre>
 *  Ticket ::=   [APPLICATION 1] SEQUENCE {
 *                    tkt-vno[0]                   INTEGER,
 *                    realm[1]                     Realm,
 *                    sname[2]                     PrincipalName,
 *                   enc-part[3]                  EncryptedData
 *               }
 *
 *  -- Encrypted part of ticket
 *  EncTicketPart ::= [APPLICATION 3] SEQUENCE {
 *                      flags[0]             TicketFlags,
 *                      key[1]               EncryptionKey,
 *                      crealm[2]            Realm,
 *                      cname[3]             PrincipalName,
 *                      transited[4]         TransitedEncoding,
 *                      authtime[5]          KerberosTime,
 *                      starttime[6]         KerberosTime OPTIONAL,
 *                      endtime[7]           KerberosTime,
 *                      renew-till[8]        KerberosTime OPTIONAL,
 *                      caddr[9]             HostAddresses OPTIONAL,
 *                      authorization-data[10]   AuthorizationData OPTIONAL
 *                    }
 * </pre>
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public final class Ticket extends Explicit
{
  public static final int TAG_KRB_TICKET = 1;
  public static final int tktVno = 5;

  private Realm realm;
  private PrincipalName sName;
  private EncryptedData encPart;

  /**
   * Constructs a Ticket from a realm, server name and EncryptedData
   */
  public Ticket(Realm realm, PrincipalName sName, EncryptedData encPart)
  {
    super(Asn1.TAG_CLASS_APPLICATION, TAG_KRB_TICKET);

    Sequence seq = new Sequence();

    Explicit exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 0);
    exp.addComponent(new com.dstc.security.asn1.Integer(tktVno));
    seq.addComponent(exp);

    this.realm = realm;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 1);
    exp.addComponent(realm);
    seq.addComponent(exp);

    this.sName = sName;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 2);
    exp.addComponent(sName);
    seq.addComponent(exp);

    this.encPart = encPart;
    exp = new Explicit(Asn1.TAG_CLASS_CONTEXT, 3);
    exp.addComponent(encPart);
    seq.addComponent(exp);

    addComponent(seq);
  }

  /**
   * Constructs a Ticket from a DER encoding
   */
  public Ticket(byte encoded[]) throws Asn1Exception, BadVersionException
  {
    doDecode(encoded);

    Sequence seq = (Sequence)this.getComponent();
    Vector comps = seq.getComponents();

    Explicit exp = (Explicit)comps.elementAt(0);
    if (exp.getTagNumber() != 0)
      throw new Asn1Exception("Bad Tag Number");
    if (((com.dstc.security.asn1.Integer)exp.getComponent()).getInt() != tktVno)
      throw new BadVersionException("Not version 5");

    exp = (Explicit)comps.elementAt(1);
    if (exp.getTagNumber() != 1)
      throw new Asn1Exception("Bad Tag number");
    this.realm = new Realm(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(2);
    if (exp.getTagNumber() != 2)
      throw new Asn1Exception("Bad Tag number");
    this.sName = new PrincipalName(exp.getComponent().encode());

    exp = (Explicit)comps.elementAt(3);
    if (exp.getTagNumber() != 3)
      throw new Asn1Exception("Bad Tag number");
    this.encPart = new EncryptedData(exp.getComponent().encode());
  } 
    
  /**
   * Returns the Realm for this Ticket
   */
  public Realm getRealm()
  {
    return this.realm;
  }

  /**
   * Returns the server name for this Ticket
   */
  public PrincipalName getServerName()
  {
    return this.sName;
  }

  /**
   * Returns the encrypted part of this Ticket
   */
  public EncryptedData getEncrypted()
  {
    return this.encPart;
  }
}
