//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.pki.netscape;

import java.security.Signature;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.security.KeyFactory;
import java.security.spec.X509EncodedKeySpec;

import com.dstc.security.x509.AlgorithmId;
import com.dstc.security.x509.SubjectPublicKeyInfo;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.BitString;
import com.dstc.security.asn1.IA5String;
import com.dstc.security.provider.OID;

/**
 * <p> Implements the ASN.1 structure PublicKeyAndChallenge.
 *
 * <pre>
 *     PublicKeyAndChallenge ::= SEQUENCE {
 *         spki SubjectPublicKeyInfo,
 *         challenge IA5STRING
 *     }
 *
 *     SignedPublicKeyAndChallenge ::= SEQUENCE {
 *         publicKeyAndChallenge PublicKeyAndChallenge,
 *         signatureAlgorithm AlgorithmIdentifier,
 *         signature BIT STRING
 *     }
 * </pre>
 * @version 0.98, 98/11/01
 * @author Ming Yung
 */
public class SignedPubKeyAndChallenge extends Sequence
{
  private PubKeyAndChallenge pkc;
  private AlgorithmId algo;
  private byte[] sig;

  /**
   * Constructs a SignedPubKeyAndChallenge from a DER encoding
   */
  public SignedPubKeyAndChallenge(byte[] encoded) throws Asn1Exception
  {
    doDecode(encoded);

    this.pkc = new PubKeyAndChallenge(((Asn1)components.elementAt(0)).encode());

    this.algo = new AlgorithmId(((Asn1)components.elementAt(1)).encode());

    this.sig = ((BitString)components.elementAt(2)).getBytes();
  }

  /**
   * Returns the SubjectPublicKeyInfo for this SignedPubKeyAndChallenge
   */
  public SubjectPublicKeyInfo getSubjectPublicKeyInfo()
  {
    return pkc.spki;
  }

  /**
   * Returns the signature algorithm name for this SignedPubKeyAndChallenge
   */
  public String getSigAlgName()
  {
    return OID.getAlgName(this.algo.getOid());
  }

  /**
   * Returns the key algorithm name for this SignedPubKeyAndChallenge
   */
  public String getKeyAlgName()
  {
    return OID.getAlgName(pkc.spki.getAlgorithmId().getOid());
  }

  /**
   * Returns the signature for this SignedPubKeyAndChallenge
   */
  public byte[] getSignature()
  {
    return this.sig;
  }

  /**
   * Verifies the signature in this Certification request
   */
  public boolean verifySignature()
  {
    try
    {
      Signature signature =
        Signature.getInstance(getSigAlgName());

      KeyFactory keyFact
        = KeyFactory.getInstance(getKeyAlgName());

      PublicKey pub = keyFact.generatePublic(new X509EncodedKeySpec(
          this.pkc.spki.encode()));

      signature.initVerify(pub);
      signature.update(this.pkc.encode());

      return signature.verify(this.sig);
    }
    catch (Exception e)
    {
      e.printStackTrace();
      return false;
    }
  }

  /**
   * <p>Inner class representing the Netscape ASN.1 structure 
   * PubKeyAndChallenge
   */
  protected class PubKeyAndChallenge extends Sequence
  {
    private SubjectPublicKeyInfo spki;
    private String challenge;

    /**
     * Constructs a PubKeyAndChallenge from a DER encoding
     */
    protected PubKeyAndChallenge(byte[] encoded) throws Asn1Exception
    {
      doDecode(encoded);

      this.spki 
        = new SubjectPublicKeyInfo(((Asn1)components.elementAt(0)).encode());

      this.challenge = ((IA5String)components.elementAt(1)).getName();
    }
  }
}
