//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.provider;

import java.math.BigInteger;
import java.security.SecureRandom;
import java.security.AlgorithmParameterGenerator;
import java.security.spec.AlgorithmParameterSpec;
import java.security.InvalidAlgorithmParameterException;
import javax.crypto.spec.DHParameterSpec;
import java.security.KeyPairGenerator;
import java.security.KeyPairGeneratorSpi;
import java.security.KeyPair;

/**
 * <p>Implements a Key Pair generator for the Diffie-Hellman 
 * key agreement algorithm.
 *
 * @see java.security.KeyPairGenerator
 * @see com.dstc.security.provider.DHPublicKey
 * @see com.dstc.security.provider.DHPrivateKey
 * @see com.dstc.security.provider.DHKeyAgreement
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class DHKeyPairGenerator extends KeyPairGeneratorSpi
{
  private SecureRandom random;
  private int keyLength;
  private BigInteger base, modulus;
  private int len;

  public void initialize(int strength, SecureRandom random)
  {
    this.random = random;
    this.keyLength = strength;
    
    if (strength == 1024)
    {
      //This group was taken from the ISAKMP/Oakley specification, 
      //and was originally generated by Richard Schroeppel at the 
      //University of Arizona.

      this.base = BigInteger.valueOf((long)2);
      this.modulus = new BigInteger(
        "179769313486231590770839156793787453197860296048756011706444" +
        "423684197180216158519368947833795864925541502180565485980503" +
        "646440548199239100050792877003355816639229553136239076508735" +
        "759914822574862575007425302077447712589550957937778424442426" +
        "617334727629299387668709205606050270810842907692932019128194" +
        "467627007", 10);
    }
    else
    {
      //Generate new DH params
      try
      {
        AlgorithmParameterGenerator algGen =
          AlgorithmParameterGenerator.getInstance("DH", "DSTC");
        algGen.init(strength, random);
  
        DHParameterSpec dhParams
           = (DHParameterSpec)algGen.generateParameters().getParameterSpec(
               Class.forName("javax.crypto.spec.DHParameterSpec"));
  
        this.base = dhParams.getG();
        this.modulus = dhParams.getP();
        this.len = dhParams.getL();
      }
      catch (Exception e)
      {
        e.printStackTrace();
      }
    }
  }

  public void initialize(AlgorithmParameterSpec params, SecureRandom random) 
    throws InvalidAlgorithmParameterException
  {
    if (!(params instanceof DHParameterSpec))
      throw new InvalidAlgorithmParameterException("Not DH parameters");

    DHParameterSpec dhParams = (DHParameterSpec)params;
    this.keyLength = dhParams.getP().bitLength();
    this.base = dhParams.getG();
    this.modulus = dhParams.getP();
    this.len = dhParams.getL();
    this.random = random;
  }

  public KeyPair generateKeyPair()
  {
    BigInteger x = new BigInteger(keyLength, random);
    DHPrivateKey priv = new DHPrivateKey(x, modulus, base, len);
    BigInteger y = base.modPow(x, modulus);
    DHPublicKey pub = new DHPublicKey(y, modulus, base, len);
    return new KeyPair(pub, priv);
  }
}
