//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.provider;

import java.math.BigInteger;
import java.io.IOException;
import java.security.AlgorithmParametersSpi;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidParameterSpecException;
import javax.crypto.spec.DHParameterSpec;
import com.dstc.security.asn1.Integer;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;

/**
 * <p>Implements algorithm parameters for the Diffie-Hellman key agreement
 * algorithm following the PKCS#3 standard.
 *
 * @see java.security.AlgorithmParameters
 * @see com.dstc.security.provider.DHKeyAgreement
 * @see javax.crypto.KeyAgreement
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class DHParameters extends AlgorithmParametersSpi
{
  private DHParameterSpec paramsSpec;
  private DHParams dhParams;

  protected void engineInit(AlgorithmParameterSpec paramSpec) 
    throws InvalidParameterSpecException
  {
    if (!(paramSpec instanceof DHParameterSpec))
      throw new InvalidParameterSpecException("Not a DH param spec");

    this.paramsSpec = (DHParameterSpec)paramSpec;
    this.dhParams = new DHParams(paramsSpec.getP(),
                                 paramsSpec.getG());
  }

  protected void engineInit(byte[] params) throws IOException
  {
    try
    {
      this.dhParams = new DHParams(params);
      this.paramsSpec = new DHParameterSpec(dhParams.p, dhParams.g);
    }
    catch (Exception e)
    {
      throw new IOException(e.toString());
    }
  }

  protected void engineInit(byte[] params, String format) throws IOException
  {
    engineInit(params);
  }

  protected AlgorithmParameterSpec engineGetParameterSpec(Class paramSpec) 
    throws InvalidParameterSpecException
  {
    return paramsSpec;
  }

  protected byte[] engineGetEncoded() throws IOException
  {
    return dhParams.encode();
  }

  protected byte[] engineGetEncoded(String format) throws IOException
  {
    return engineGetEncoded();
  }

  protected String engineToString()
  {
    return "DH Parameters";
  }

  /**
   * <p>An Inner class representing the PKCS#3 ASN.1 structure DHParameter.
   *
   * <p>DHParameter ::= SEQUENCE {
   *                   prime INTEGER, -- p
   *                   base INTEGER, -- g
   *                   privateValueLength INTEGER OPTIONAL}
   */

  protected class DHParams extends Sequence
  {
    private BigInteger g, p;
    private int l;
  
    /**
     * Constructs a DHParams given parameters p, g, l
     */
    protected DHParams (BigInteger p, BigInteger g) 
    {
      this.p = p;
      this.g = g;
      this.addComponent(new com.dstc.security.asn1.Integer(p));
      this.addComponent(new com.dstc.security.asn1.Integer(g));
    }
  
    /**
     * Constructs a DHParams from a DER encoding
     */
    protected DHParams(byte encoded[]) throws Asn1Exception
    {
      doDecode(encoded);
      this.p = ((com.dstc.security.asn1.Integer)
                 components.elementAt(0)).getBigInteger();
      this.g = ((com.dstc.security.asn1.Integer)
                 components.elementAt(1)).getBigInteger();
    }
  }
}
  
