//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509;

import java.util.StringTokenizer;
import java.util.Vector;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.SetOf;
import com.dstc.security.asn1.PrintableString;
import com.dstc.security.asn1.T61String;
import com.dstc.security.asn1.BMPString;
import com.dstc.security.asn1.IA5String;
import com.dstc.security.asn1.Oid;
import com.dstc.security.provider.OID;

/**
 * <p>A class representing the x.500 ASN.1 structure AttributeTypeAndValue.
 *
 * <pre>
 *    AttributeTypeAndValue           ::=     SEQUENCE {
 *          type    AttributeType,
 *          value   AttributeValue }
 *  
 *    AttributeType           ::=   OBJECT IDENTIFIER
 *
 *    AttributeValue            ::=   CHOICE {
 *          DirectoryString,
 *          IA5String
 *          }
 * </pre>
 *
 * @see com.dstc.security.x509.X500Name
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class AttributeTypeAndValue extends Sequence
{
  private String type;
  private String value;

  /**
   * Constructs an AttributeTypeAndValue from a type and a value.
   *
   */
  public AttributeTypeAndValue(String type, String value)
  {
    if (type.indexOf('.') != -1)
      this.type = type;
    else
      this.type = OID.getAlgOid(type);
    
    addComponent(new Oid(this.type));

    this.value = value;
    if (!this.type.equals(OID.emailAddress))
      addComponent(new PrintableString(value));
    else
      addComponent(new IA5String(value));
  }

  /**
   * Constructs an AttributeTypeAndValue from a DER encoding
   */
  public AttributeTypeAndValue(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);

    this.type = ((Oid)components.elementAt(0)).getOid();

    try
    {
      this.value = ((PrintableString)components.elementAt(1)).getName();
      return;
    }
    catch (ClassCastException e)
    {
      //Ignore
    }

    try
    {
      this.value = ((T61String)components.elementAt(1)).getName();
      return;
    }
    catch (ClassCastException e)
    {
      //Ignore
    }

    try
    {
      this.value = ((BMPString)components.elementAt(1)).getName();
      return;
    }
    catch (ClassCastException e)
    {
      //Ignore
    }

    this.value = ((IA5String)components.elementAt(1)).getName();
  }

  public String getRawName()
  {
    return (type + "=" + value);
  }

  public String getName()
  {
    return (OID.getAlgName(type) + "=" + value);
  }

  public String getType()
  {
    return OID.getAlgName(type);
  }

  public String getValue()
  {
    return value;
  }
}
