//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509;

import java.util.Set;
import java.math.BigInteger;
import java.util.Date;
import java.security.cert.X509CRLEntry;
import java.security.cert.CRLException;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.UTCTime;

/**
 * <p>Representation of the ASN.1 structure revokedCertificate.
 * 
 * <pre>
 * revokedCertificates    SEQUENCE OF SEQUENCE  {
 *     userCertificate    CertificateSerialNumber,
 *     revocationDate     ChoiceOfTime,
 *     crlEntryExtensions Extensions OPTIONAL
 *                        -- if present, must be v2
 * }  OPTIONAL
 *
 * CertificateSerialNumber  ::=  INTEGER
 *
 * Extensions  ::=  SEQUENCE SIZE (1..MAX) OF Extension
 *
 * Extension  ::=  SEQUENCE  {
 *     extnId        OBJECT IDENTIFIER,
 *     critical      BOOLEAN DEFAULT FALSE,
 *     extnValue     OCTET STRING
 *                   -- contains a Asn1 encoding of a value
 *                   -- of the type registered for use with
 *                   -- the extnId object identifier value
 * }
 * </pre>
 *
 */
public class RevokedCertificate extends Sequence
{
  private BigInteger userCertificate;
  private UTCTime revocationDate;
  private Extensions crlExtns = null;

  /**
   * Constructs a RevokedCertificate from a DER encoding
   */
  public RevokedCertificate(byte[] encoded) throws Asn1Exception
  {
    doDecode(encoded);
    int i = 0;

    this.userCertificate 
      = ((com.dstc.security.asn1.Integer)
          components.elementAt(i++)).getBigInteger();

    this.revocationDate 
      = (UTCTime)components.elementAt(i++);

    if (i == components.size()) return;
    this.crlExtns = 
      new Extensions(((Asn1)components.elementAt(i++)).encode());
  }

  /**
   * Constructs a RevokedCertificate
   */
  public RevokedCertificate(BigInteger userCert, UTCTime rev, 
                               Extensions extns)
  {
    this.userCertificate = userCert;
    addComponent(new com.dstc.security.asn1.Integer(userCert));

    this.revocationDate = rev;
    addComponent(rev);

    if (extns == null) return;
    this.crlExtns = extns;
    addComponent(extns);
  }

  public BigInteger getSerialNumber()
  {
    return this.userCertificate;
  }

  public UTCTime getRevocationDate()
  {
    return this.revocationDate;
  }

  public Extensions getCrlExtns()
  {
    return this.crlExtns;
  }
}
