//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509;

import java.util.Vector;
import java.util.Date;
import java.util.Calendar;
import java.util.TimeZone;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.UTCTime;
import com.dstc.security.asn1.Sequence;

/**
 * <p>A class representing the x.509 ASN.1 structure Validity.
 *
 * <pre>
 *    Validity ::= SEQUENCE {
 *       notBefore      CertificateValidityDate,
 *       notAfter       CertificateValidityDate }
 *  
 *    CertificateValidityDate ::= CHOICE {
 *       utcTime        UTCTime,
 *       generalTime    GeneralizedTime }
 * </pre>
 *
 * <p>Currently only handles GMT
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class Validity extends Sequence
{
  //UTCTime for notBefore
  private String notBefore;

  //UTCTime for notAfter
  private String notAfter;

  /**
   * Constructs a Validity from a period in days
   */
  public Validity(int days) 
  {
    Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    StringBuffer now = new StringBuffer();
    now.append(String.valueOf(cal.get(Calendar.YEAR)).substring(2, 4));

    append(now, cal.get(Calendar.MONTH) + 1);
    append(now, cal.get(Calendar.DATE));
    append(now, cal.get(Calendar.HOUR));
    append(now, cal.get(Calendar.MINUTE));

    now.append("Z");
    this.notBefore = now.toString();
    addComponent(new UTCTime(notBefore));

    cal.add(Calendar.DATE, days);
    now = new StringBuffer();
    now.append(String.valueOf(cal.get(Calendar.YEAR)).substring(2, 4));

    append(now, cal.get(Calendar.MONTH) + 1);
    append(now, cal.get(Calendar.DATE));
    append(now, cal.get(Calendar.HOUR));
    append(now, cal.get(Calendar.MINUTE));

    now.append("Z");
    this.notAfter = now.toString();
    addComponent(new UTCTime(notAfter));
  }

  /**
   * Constructs a Validity given now and notAfter
   */
  Validity(String notBefore, String notAfter) 
  {
    this.notBefore = notBefore;
    addComponent(new UTCTime(notBefore));

    this.notAfter = notAfter;
    addComponent(new UTCTime(notAfter));
  }

  /**
   * Constructs a Validity from a DER encoding
   */
  Validity(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);
    this.notBefore = ((UTCTime)components.elementAt(0)).getTime();
    this.notAfter = ((UTCTime)components.elementAt(1)).getTime();
  }

  /**
   * Returns a string representation of this Validity
   */
  public String toString() 
  {
    return new StringBuffer("Not Before: " + notBefore + "\n"
                            + "Not After:   " + notAfter).toString ();
  }

  /**
   * Returns notBefore as a Java Date
   */
  Date getNotBefore() 
  {
    return getDate(notBefore);
  }

  /**
   * Returns notAfter as a Java Date
   */
  Date getNotAfter() 
  {
    return getDate(notAfter);
  }

  // Only handles GMT time currently
  static final Date getDate(String utc) 
  {
    Date retval = null;
    if (utc.endsWith ("Z")) 
    {
      Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
      int year, month, date, hour, minute, second;
      int twoDigitYear;
      switch (utc.length ()) 
      {
        //YYMMDDhhmmZ
        case (11): 
          twoDigitYear = java.lang.Integer.parseInt (utc.substring (0, 2));
          if(twoDigitYear > 50)
                 year = 1900 + twoDigitYear;
          else
                 year = 2000 + twoDigitYear;

          month = java.lang.Integer.parseInt (utc.substring (2, 4)) - 1;
          date = java.lang.Integer.parseInt (utc.substring (4, 6));
          hour = java.lang.Integer.parseInt (utc.substring (6, 8));
          minute = java.lang.Integer.parseInt (utc.substring (8, 10));
          cal.set (year, month, date, hour, minute);
          break;

        //YYMMDDhhmmssZ
        case (13): 
          twoDigitYear = java.lang.Integer.parseInt (utc.substring (0, 2));
          if(twoDigitYear > 50)
                 year = 1900 + twoDigitYear;
          else
                 year = 2000 + twoDigitYear;

          month = java.lang.Integer.parseInt (utc.substring (2, 4)) - 1;
          date = java.lang.Integer.parseInt (utc.substring (4, 6));
          hour = java.lang.Integer.parseInt (utc.substring (6, 8));
          minute = java.lang.Integer.parseInt (utc.substring (8, 10));
          second = java.lang.Integer.parseInt (utc.substring (10, 12));
          cal.set (year, month, date, hour, minute, second);
          break;

        default: ;
      }
       retval = cal.getTime ();
    }
    return retval;
  } 
      
  private void append(StringBuffer sb, int number)
  {
    if (number < 10)
      sb.append("0" + number);
    else
      sb.append(number);
  }
}
