//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509;

import java.io.IOException;
import java.util.Vector;
import java.util.TreeSet;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.security.Principal;
import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;

/**
 * <p>A class representing the x.509 ASN.1 structure Name.
 *
 * <pre>
 *    Name            ::=   CHOICE {
 *                         -- only one possibility for now --
 *                            rdnSequence  RDNSequence }
 *
 *  RDNSequence     ::=   SEQUENCE OF RelativeDistinguishedName
 * </pre>
 *
 * @see com.dstc.security.x509.RDN
 *
 * @version 0.98, 98/07/01
 * @author Ming Yung
 */
public class X500Name extends Sequence implements Principal
{
  private Vector rdns = new Vector();

  /**
   * Constructor from a string distinguished name 
   */
  public X500Name(String dn) throws IOException
  {
    StringTokenizer tknz = new StringTokenizer(dn, ",", false);

    String current = tknz.nextToken();
    StringBuffer sb = new StringBuffer(current);
    RDN rdn;

    while (tknz.hasMoreTokens())
    {
      current = tknz.nextToken();
      if (current.indexOf('=') == -1)
      {
        sb.append(",").append(current);
      }
      else
      {
        rdn = new RDN(sb.toString());
        rdns.addElement(rdn);
        sb = new StringBuffer(current);
      }
    }
    rdn = new RDN(sb.toString());
    rdns.addElement(rdn);

    Iterator ordered = (new TreeSet(rdns)).iterator();
    while (ordered.hasNext())
    {
      addComponent((RDN)ordered.next());
    }
  }

  /**
   * Constructs an X500Name from a DER encoding
   */
  public X500Name(byte encoded[]) throws Asn1Exception
  {
    doDecode(encoded);
    
    for (int i=0; i<components.size(); i++)
    {
      rdns.addElement(new RDN(((Asn1)components.elementAt(i)).encode()));
    }
  }

  /**
   * Returns a Vector of RDNs
   */
  public Vector getRDNs()
  {
    return this.rdns;
  }

  //Methods defined abstract in Principal 

  public int hashcode() 
  {
    return 0;
  }

  /**
   * Returns the distinguished name as a string
   */
  public String getName() 
  {
    StringBuffer st = new StringBuffer();
    Iterator ordered = (new TreeSet(rdns)).iterator();

    while (ordered.hasNext())
    {
      st.append(((RDN)ordered.next()).getName());
      if (ordered.hasNext())
        st.append(", ");
    }
    return st.toString();
  }

  public boolean equals(Object another)
  {
    try
    {
      String name = ((X500Name)another).getName();
      if (this.getName().equals(name))
      {
        return true;
      }
      else
      {
        return false;
      }
    }
    catch (ClassCastException e)
    {
      return false;
    }
  }
}
