//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package com.dstc.security.x509.extns;

import java.util.Vector;
import java.math.BigInteger;

import com.dstc.security.asn1.Asn1;
import com.dstc.security.asn1.Sequence;
import com.dstc.security.asn1.Asn1Exception;
import com.dstc.security.asn1.Primitive;
import com.dstc.security.asn1.OctetString;
import com.dstc.security.asn1.Oid;
import com.dstc.security.provider.OID;
import com.dstc.security.x509.Extension;
import com.dstc.security.x509.GeneralNames;

/**
 * Class representing the ASN.1 structure AuthorityKeyIdentifier.
 *
 * <pre>
 *  AuthorityKeyIdentifier ::= SEQUENCE {
 *        keyIdentifier             [0] KeyIdentifier            OPTIONAL,
 *        authorityCertIssuer       [1] GeneralNames             OPTIONAL,
 *        authorityCertSerialNumber [2] CertificateSerialNumber  OPTIONAL }
 *      -- authorityCertIssuer and authorityCertSerialNumber shall both
 *      -- be present or both be absent
 *  
 *  KeyIdentifier ::= OCTET STRING
 * </pre>  
 * @author Ming Yung
 */
public class AuthorityKeyIdentifier extends Extension
{
  private byte[] keyId;
  private GeneralNames issuer;
  private BigInteger serialNumber;

  /**
   * Constructs an AuthorityKeyIdentifier extension from a key identifier
   * (Non-critical).
   */
  public AuthorityKeyIdentifier(boolean dummy, byte[] keyId)
  {
    this.extnID = OID.authorityKeyIdentifier;
    this.addComponent(new Oid(extnID));
 
    this.keyId = keyId;

    Sequence seq = new Sequence();
    OctetString oct = new OctetString(keyId);
    oct.setTagClass(Asn1.TAG_CLASS_CONTEXT);
    oct.setTagNumber(0);
    seq.addComponent(oct);
    this.extnValue = seq.encode();
    this.addComponent(new OctetString(this.extnValue));
  }

  /**
   * Constructs an AuthorityKeyIdentifier extension from a cert issuer
   * and serial number (Non-critical).
   */
  public AuthorityKeyIdentifier(GeneralNames issuer, BigInteger serial)
  {
    this.extnID = OID.authorityKeyIdentifier;
    this.addComponent(new Oid(extnID));
 
    this.issuer = issuer;
    this.serialNumber = serial;
    Sequence seq = new Sequence();

    issuer.setTagClass(Asn1.TAG_CLASS_CONTEXT);
    issuer.setTagNumber(1);
    seq.addComponent(issuer);

    com.dstc.security.asn1.Integer intgr= 
      new com.dstc.security.asn1.Integer(serial);
    intgr.setTagClass(Asn1.TAG_CLASS_CONTEXT);
    intgr.setTagNumber(2);

    seq.addComponent(intgr);
    this.extnValue = seq.encode();
    this.addComponent(new OctetString(this.extnValue));
  }

  /**
   * Constructs an AuthorityKeyIdentifier extension from a DER encoding
   */
  public AuthorityKeyIdentifier(byte[] encoded) throws Asn1Exception
  {
    super(encoded);
 
    Sequence seq = (Sequence)Asn1.decode(this.value);
    Vector comps = seq.getComponents();
    int i = 0;

    try
    {
      Primitive imp = (Primitive)comps.elementAt(i);
      this.keyId = imp.getBytes();
    }
    catch (ClassCastException e)
    {
      e.printStackTrace();
    }
  }

  public String toString()
  {
    StringBuffer sb = new StringBuffer();

    sb.append("AuthorityKeyIdentifier extension:\n");
    sb.append("  " + printHex(this.keyId));

    return sb.toString();
  }
}
