//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package javax.net.ssl;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.InetAddress;
import java.net.UnknownHostException;

/**
 * <p> This class is extended by server sockets which return connections
 * which are protected using the Secure Sockets Layer (SSL) protocol, and
 * which extend the SSLSocket class.
 *
 * @version 0.98, 98/09/10
 * @author Ming Yung
 */

public abstract class SSLServerSocket extends ServerSocket
{
  /**
   * Used only by subclasses. Create a TCP server socket on a port, using
   * the default authentication context. The connection backlog
   * defaults to fifty connections queued up before the system
   * starts to reject new connection requests.
   *
   * @param port  the port on which to listen
   */
  protected SSLServerSocket(int port) throws IOException
  {
    super(port);
  }

  /**
   * Used only by subclasses. Create a TCP server socket on a port, using
   * the default authentication context and a specified backlog of
   * connections.
   *
   * @param port  the port on which to listen
   * @param backlog  how many connections may be pending before the
   * system should start rejecting new requests
   */
  protected SSLServerSocket(int port, int backlog) throws IOException
  {
    super(port, backlog);
  }


  /**
   * Used only by subclasses. Create a TCP server socket on a port, using
   * the default authentication context and a specified backlog of
   * connections as well as a particular specified network
   * interface. This constructor is used on multihomed hosts, such
   * as those used for firewalls or as routers, to control through
   * which interface a network service is provided.
   *
   * @param port  the port on which to listen
   * @param backlog  how many connections may be pending before the
   * system should start rejecting new requests
   * @param address  the address of the network interface through
   * which connections will be accepted
   */
  protected SSLServerSocket(int port, int backlog, InetAddress address) 
    throws IOException
  {
    super(port, backlog, address);
  }

  /**
   * Returns the names of the cipher suites which could be enabled for use
   * on an SSL connection. Normally, only a subset of these will
   * actually be enabled by default, since this list may include
   * cipher suites which do not meet quality of service requirements
   * for those defaults. Such cipher suites are useful in
   * specialized applications.
   *
   * @return an array of cipher suite names
   */
  public abstract String[] getSupportedCipherSuites();

  /**
   * Returns the list of cipher suites which are currently enabled for use
   * by newly accepted connections. If this has not been explicitly
   * modified, a system-provided default guarantees a minimum
   * quality of service in all enabled cipher suites. That default
   * is accessible through the SSLSocket API.
   *
   * <p> There are several reasons why an enabled cipher suite might not
   * actually be used. For example: the server socket might not have
   * appropriate private keys available to it or the cipher suite
   * might be anonymous, precluding the use of client
   * authentication, while the server socket has been told to
   * require that sort of authentication.
   *
   * @return an array of cipher suite names
   */
  public abstract String[] getEnabledCipherSuites();

  /**
   * Controls which particular SSL cipher suites are enabled for use by
   * accepted connections. The cipher suites must have been listed
   * by getSupportedCipherSuites() as being supported. Suites which
   * require authentication information which is not available in
   * this ServerSocket's authentication context will not be used in
   * any case, even if they are enabled.
   *
   * @param suites  Names of all the cipher suites to enable; null
   * means to accept system defaults.
   *
   */
  public abstract void setEnabledCipherSuites(String suites[]);

  /**
   * Controls whether accepted connections are in the (default) SSL server
   * mode, or the SSL client mode. In rare cases, TCP servers need
   * to act in the SSL client mode on newly accepted connections in
   * order to reuse sessions. For example, FTP clients acquire
   * server sockets and listen there for reverse connections, from
   * the server. FTP clients would use SSLServerSockets in "client"
   * mode, in order to reuse existing SSL sessions where the FTP
   * client is acting as a client.
   *
   * @param flag  true if newly accepted connections should use SSL
   * client mode.
   */
  public abstract void setUseClientMode(boolean mode);

  /**
   * Returns true if accepted connections will be in SSL client mode.
   *
   * @return true if the connection should use SSL client mode.
   */
  public abstract boolean getUseClientMode();

  /**
   * Controls whether the connections which are accepted must include
   *  client authentication. By default, clients do not need to
   *  provide authentication information. This control requires the
   *  connection to be in the (default) SSL server mode.
   *
   * @param flag  true if the clients must authenticate themselves.
   */
  public abstract void setNeedClientAuth(boolean flag);

  /**
   * Returns true if client authentication is required on newly accepted
   * connections.
   *
   * @return true if the clients must authenticate themselves.
   */
  public abstract boolean getNeedClientAuth();

  /**
   * Controls whether new SSL sessions may be established by the sockets
   * which are created from this server socket.
   *
   * @param flag  true indicates that sessions may be created; this
   * is the default. false indicates that an existing session
   * must be resumed.
   */
  public abstract void setEnableSessionCreation(boolean flag);

  /**
   * Returns true if new SSL sessions may be established by the sockets
   * which are created from this server socket.
   *
   * @return true indicates that sessions may be created; this is the
   * default. false indicates that an existing session must be
   * resumed.
   */
  public abstract boolean getEnableSessionCreation();
}
