//////////////////////////////////////////////////////////////////////////// 
// 
// Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
// Unpublished work.  All Rights Reserved.
// 
// The software contained on this media is the property of the
// DSTC Pty Ltd.  Use of this software is strictly in accordance
// with the license agreement in the accompanying LICENSE.DOC 
// file. If your distribution of this software does not contain 
// a LICENSE.DOC file then you have no rights to use this 
// software in any manner and should contact DSTC at the address 
// below to determine an appropriate licensing arrangement.
// 
//      DSTC Pty Ltd
//      Level 7, GP South
//      University of Queensland
//      St Lucia, 4072
//      Australia
//      Tel: +61 7 3365 4310
//      Fax: +61 7 3365 4311
//      Email: jcsi@dstc.qut.edu.au
// 
// This software is being provided "AS IS" without warranty of
// any kind.  In no event shall DSTC Pty Ltd be liable for
// damage of any kind arising out of or in connection with
// the use or performance of this software.
// 
//////////////////////////////////////////////////////////////////////////// 

package javax.net.ssl;

import java.security.cert.X509Certificate;

/**
 * <p> A SSLSession is a grouping of SSLSessions associated with a
 * single entity. For example, it could be associated with a server or
 * client who participates in many sessions concurrently. This interface
 * provides methods for retrieving a SSLSession based on its ID, and
 * allows such IDs to be listed.
 *
 * @version 0.98, 98/09/10
 * @author Ming Yung
 *
 */

public interface SSLSession 
{
  /**
   * Returns the identifier assigned to this Session.
   */
  public abstract byte[] getId();

  /**
   * Returns the context in which this session is bound. This context may
   * be unavailable in some environments, in which case this method
   * returns null.
   */
  public abstract SSLSessionContext getSessionContext();

  /**
   * Returns the time at which this Session representation was created, in
   * milliseconds since midnight, January 1, 1970 UTC.
   */
  public abstract long getCreationTime();

  /**
   * Returns the last time this Session representation was accessed by the
   * session level infrastructure, in milliseconds since midnight,
   * January 1, 1970 UTC. Access indicates a new connection being
   * established using session data. Application level operations,
   * such as getting or setting a value associated with the session,
   * are not reflected in this access time.
   *
   * <p> This information is particularly useful in session management
   * policies. For example, a session manager thread could leave all
   * sessions in a given context which haven't been used in a long
   * time; or, the sessions might be sorted according to age to
   * optimize some task.
   */
  public abstract long getLastAccessedTime();

  /**
   * Invalidates the session. Future connections will not be able to resume
   * or join this session.
   */
  public abstract void invalidate();

  /**
   * Binds the specified object into the session's application layer data
   * with the given name. Any existing binding with the same name is
   * replaced. If the new (or existing) value implements the
   * SSLSessionBindingListener interface, it is notified appropriately.
   *
   * @param name  the name to which the data object will be bound.
   * This may not be null.
   * @param value  the data object to be bound. This may not be null.
   */
  public abstract void putValue(String name, Object value);

  /**
   * Returns the object bound to the given name in the session's
   * application layer data. Returns null if there is no such binding.
   *
   * @param name  the name of the binding to find.
   * @return the value bound to that name, or null if the binding does
   * not exist.
   */
  public abstract Object getValue(String name);

  /**
   * Removes the object bound to the given name in the session's
   * application layer data. Does nothing if there is no object
   * bound to the given name. If the value implements the
   * SessionBindingListener interface, it is notified appropriately.
   *
   * @param name  the name of the object to remove
   */
  public abstract void removeValue(String name);

  /**
   * Returns an array of the names of all the application layer data
   * objects bound into the Session.
   */
  public abstract String[] getValueNames();

  /**
   * Return the cert chain presented by the peer.
   *
   * @return array of peer X.509 certs, with the peer's own cert first
   * in the chain, and with the "root" CA last.
   */
  public abstract X509Certificate[] getPeerCertificateChain() 
    throws SSLPeerUnverifiedException; 

  /**
   * Returns the name of the SSL cipher suite which is used for all
   * connections in the session. This defines the level of
   * protection provided to the data sent on the connection,
   * including the kind of encryption used and most aspects of how
   * authentication is done..
   *
   * The cipher suite names are as defined in the SSL protocol
   * definition, and include:
   *
   * + SSL_RSA_WITH_RC4_128_MD5 ... a non-exportable cipher suite
   *   supporting 128 bit RC4 encryption keys and full RSA key
   *   sizes.
   * + SSL_DHE_DSS_WITH_3DES_EDE_CBC_SHA ... a non-exportable cipher
   *   suite supporting 168 bit DES encryption keys. (The effective
   *   strength of this cipher is only 112 bits.)
   * + SSL_RSA_EXPORT_WITH_RC4_40_MD5 ... an exportable cipher suite
   *   using weakened RC4 encryption and limited RSA key sizes.
   * + SSL_DH_anon_EXPORT_WITH_DES40_CBC_SHA ... an exportable
   *   cipher suite using weakened DES encryption, and which doesn't
   *   support authentication of servers.
   * + SSL_RSA_WITH_NULL_MD5 ... an exportable cipher suite using no
   *   encryption and full RSA key sizes.
   *
   * @return the name of the session's cipher suite
   */
  public abstract String getCipherSuite();

  /**
   * Returns the host name of the peer in this session. That is, for the
   * server, this is the client's host, and for the client it is the 
   * server's host.
   */
  public abstract String getPeerHost();
}
