/******************************************************************************
Module name: SparseFile.cpp
Written by: Jeffrey Richter
Notices: Copyright (c) 1998 Jeffrey Richter
******************************************************************************/


#define STRICT
#define _WIN32_WINNT	0x0500
#include <Windows.h>
#include <WinIoCtl.h>


///////////////////////////////////////////////////////////////////////////////


class CSparseStream {
public:
	static BOOL DoesFileSystemSupportSparseStreams(LPCTSTR pszVolume);
	static BOOL DoesFileContainAnySparseStreams(LPCTSTR pszPathname);

public:
	CSparseStream(HANDLE hstream) { m_hstream = hstream; m_nReadOffset = 0; }
	~CSparseStream() { }

public:
	operator HANDLE() const { return(m_hstream); }

	public:
	BOOL IsStreamSparse() const;
	BOOL MakeSparse();
	BOOL DecommitPortionOfStream(__int64 qwFileOffsetStart, __int64 qwFileOffsetEnd);

	FILE_ALLOCATED_RANGE_BUFFER* QueryAllocatedRanges(PDWORD pdwNumEntries);
	BOOL FreeAllocatedRanges(FILE_ALLOCATED_RANGE_BUFFER* pfarb);

	BOOL  AppendQueueEntry(PVOID pvEntry, DWORD cbEntry);
	PVOID ExtractQueueEntry(PDWORD pcbEntry = NULL);
	BOOL  FreeExtractedQueueEntry(PVOID pvEntry);

private:
	HANDLE m_hstream;
	__int64 m_nReadOffset;

private:
	static BOOL AreFlagsSet(DWORD fdwFlagBits, DWORD fFlagsToCheck) {
		return((fdwFlagBits & fFlagsToCheck) == fFlagsToCheck);
	}
};


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::DoesFileSystemSupportSparseStreams(LPCTSTR pszVolume) {
	DWORD dwFileSystemFlags = 0;
	BOOL fOk = GetVolumeInformation(pszVolume, NULL, 0, NULL, NULL, 
		&dwFileSystemFlags, NULL, 0);
	fOk = fOk && AreFlagsSet(dwFileSystemFlags, FILE_SUPPORTS_SPARSE_FILES);
	return(fOk);
}


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::IsStreamSparse() const {
	BY_HANDLE_FILE_INFORMATION bhfi;
	GetFileInformationByHandle(m_hstream, &bhfi);
	return(AreFlagsSet(bhfi.dwFileAttributes, FILE_ATTRIBUTE_SPARSE_FILE));
}


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::MakeSparse() {
	DWORD dw;
	return(DeviceIoControl(m_hstream, FSCTL_SET_SPARSE, NULL, 0, NULL, 0, &dw, NULL));
}


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::AppendQueueEntry(PVOID pvEntry, DWORD cbEntry) {

	// Always write new entries to the end of the queue
	SetFilePointer(m_hstream, 0, NULL, FILE_END);

	DWORD cb;
	// Write the size of the entry
	BOOL fOk = WriteFile(m_hstream, &cbEntry, sizeof(cbEntry), &cb, NULL);

	// Write the entry itself
	fOk = fOk && WriteFile(m_hstream, pvEntry, cbEntry, &cb, NULL);
	return(fOk);
}


///////////////////////////////////////////////////////////////////////////////


PVOID CSparseStream::ExtractQueueEntry(PDWORD pcbEntry) {
	DWORD cbEntry, cb;
	PVOID pvEntry = NULL;
	LARGE_INTEGER liOffset;
	liOffset.QuadPart = m_nReadOffset;

	// Position to the next place to read from
	SetFilePointer(m_hstream, liOffset.LowPart, 
		&liOffset.HighPart, FILE_BEGIN);

	if (pcbEntry == NULL) pcbEntry = &cbEntry;

	// Read the size of the entry
	BOOL fOk = ReadFile(m_hstream, pcbEntry, sizeof(*pcbEntry), &cb, NULL);

	// Allocate memory for the queue entry
	fOk = fOk && ((pvEntry = HeapAlloc(GetProcessHeap(), 0, *pcbEntry)) != NULL);

	// Read the queue entry into the allocated memory
	fOk = fOk && ReadFile(m_hstream, pvEntry, *pcbEntry, &cb, NULL);

	if (fOk) {
		m_nReadOffset += sizeof(*pcbEntry) + *pcbEntry;
		// Decommit the storage occupied the extracted queue entries
		fOk = fOk && DecommitPortionOfStream(0, m_nReadOffset);
	}
	return(pvEntry);	// Return the queue entry's allocated memory
}


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::FreeExtractedQueueEntry(PVOID pvEntry) {
	// Free the queue entry's allocated memory
	return(HeapFree(GetProcessHeap(), 0, pvEntry));
}


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::DecommitPortionOfStream(__int64 qwFileOffsetStart, __int64 qwFileOffsetEnd) {
	DWORD dw;
	FILE_ZERO_DATA_INFORMATION fzdi;
	fzdi.FileOffset.QuadPart = qwFileOffsetStart;
	fzdi.BeyondFinalZero.QuadPart = qwFileOffsetEnd;
	return(DeviceIoControl(m_hstream, FSCTL_SET_ZERO_DATA, (LPVOID) &fzdi, 
		sizeof(fzdi), NULL, 0, &dw, NULL));
}


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::DoesFileContainAnySparseStreams(LPCTSTR pszPathname) {
	DWORD dw = GetFileAttributes(pszPathname);
	return((dw == 0xfffffff) ? FALSE : AreFlagsSet(dw, FILE_ATTRIBUTE_SPARSE_FILE));
}


///////////////////////////////////////////////////////////////////////////////


FILE_ALLOCATED_RANGE_BUFFER* CSparseStream::QueryAllocatedRanges(PDWORD pdwNumEntries) {
	FILE_ALLOCATED_RANGE_BUFFER farb;
	farb.FileOffset.QuadPart = 0;
	farb.Length.LowPart = GetFileSize(m_hstream, (PDWORD) &farb.Length.HighPart);

	// There is no way to determine the correct memory 
	// block size prior to attempting to collect this data,
	// so I just picked 1000 * sizeof(*pfarb)
	DWORD cb = 100 * sizeof(FILE_ALLOCATED_RANGE_BUFFER);
	FILE_ALLOCATED_RANGE_BUFFER* pfarb = (FILE_ALLOCATED_RANGE_BUFFER*) 
		HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, cb);

	BOOL fOk = DeviceIoControl(m_hstream, FSCTL_QUERY_ALLOCATED_RANGES,
		&farb, sizeof(farb), pfarb, cb, &cb, NULL);
	GetLastError();
	*pdwNumEntries = cb / sizeof(*pfarb);
	return(pfarb);
}


///////////////////////////////////////////////////////////////////////////////


BOOL CSparseStream::FreeAllocatedRanges(FILE_ALLOCATED_RANGE_BUFFER* pfarb) {
	// Free the queue entry's allocated memory
	return(HeapFree(GetProcessHeap(), 0, pfarb));
}


///////////////////////////////////////////////////////////////////////////////


int WINAPI WinMain(HINSTANCE, HINSTANCE, LPSTR, int){
	TCHAR szPathName[] = __TEXT("D:\\SparseFile");

	if (!CSparseStream::DoesFileSystemSupportSparseStreams("D:\\")) {
		// run "ChkNtfs /e"
		MessageBox(NULL, "File system doesn't support Sparse Files", NULL, MB_OK);
		return(0);
	}

	HANDLE hstream = CreateFile(szPathName, GENERIC_READ | GENERIC_WRITE, 
		0, NULL, CREATE_ALWAYS, 0, NULL);
	CSparseStream ss(hstream);
	BOOL f = ss.MakeSparse();
	f = ss.IsStreamSparse();


	DWORD dwNumEntries, cb;
	SetFilePointer(ss, 50 * 1024 * 1024, NULL, FILE_BEGIN);
	WriteFile(ss, "A", 1, &cb, NULL);
	cb = GetFileSize(ss, NULL);
	cb = GetCompressedFileSize(szPathName, NULL);
	FILE_ALLOCATED_RANGE_BUFFER* pfarb = ss.QueryAllocatedRanges(&dwNumEntries);
	ss.FreeAllocatedRanges(pfarb);
	ss.DecommitPortionOfStream(0, 60 * 1024 * 1024);
	pfarb = ss.QueryAllocatedRanges(&dwNumEntries);
	ss.FreeAllocatedRanges(pfarb);
	cb = GetFileSize(ss, NULL);
	cb = GetCompressedFileSize(szPathName, NULL);

	SetFilePointer(ss, 0, NULL, FILE_BEGIN);
	SetEndOfFile(ss);

	// Put a bunch of entries in the end of the queue
	BYTE bEntry[32 * 1024 - 4];	// 100KB
	for (int x = 0; x < 7; x++) ss.AppendQueueEntry(bEntry, sizeof(bEntry));
	pfarb = ss.QueryAllocatedRanges(&dwNumEntries);
	ss.FreeAllocatedRanges(pfarb);

	// Read a bunch of entries from the beginning of the queue
	for (x = 0; x < 7; x++) {
		PVOID pvEntry = ss.ExtractQueueEntry(&cb);
		ss.FreeExtractedQueueEntry(pvEntry);
		cb = GetFileSize(ss, NULL);
		cb = GetCompressedFileSize(szPathName, NULL);
		pfarb = ss.QueryAllocatedRanges(&dwNumEntries);
		ss.FreeAllocatedRanges(pfarb);
	}
	CloseHandle(hstream);
	DeleteFile(szPathName);

	return(0);
}


///////////////////////////////// End Of File /////////////////////////////////
