/* 
   Unix SMB/Netbios implementation.
   Version 2.0.
   Copyright (C) Andrew Tridgell 1994-1999

   This file is part of the port to OpenVMS
   VMS stat() routines.
   Copyright (C) Eckart Meyer 1996-1999
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*	
 * Special vms version for stat().
 * Also included lstat() (mapped to stat()) and fstat().
 *
 */

#include "incl.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stat.h>

#if 1	/* for use with samba's debugging */	/* SAMBA */
extern int DEBUGLEVEL;
#define DEBUG(level,body) ((DEBUGLEVEL>=(level))?(Debug1 body):0)
int Debug1();
#else
int DEBUGLEVEL = 8;
#define DEBUG(level,body) ((DEBUGLEVEL>=(level))?(printf body):0)
#endif

char *vms_encode_filespec(const char * __name, int dirflag);	/* SAMBA */

struct vms_ino {
	short fid1;
	short fid2;
	short fid3;
};


int vms_stat(const char *file_spec, struct stat *st_buf)
{
	char *buf;
	int status;
	time_t t;
	short ino2;
	ino_t *unix_ino;
	struct vms_ino *inofid;

	buf = vms_encode_filespec(file_spec,FALSE);
	status = stat(buf, (struct stat *)st_buf);
/*
 * If file not present try a directory which may be encoded differently
 * Confirm that this is really a directory.
 */
	if (status == -1) {
		buf = vms_encode_filespec(file_spec,TRUE);
		status = stat(buf, (struct stat *)st_buf);
		if (status == 0  &&  !(st_buf->st_mode & S_IFDIR)) status = -1;
	}
/*
 * To be compatible to UNIX, st_ino must be 1 32-bit words. VMS uses the
 * 3 16-bit words from the FID. As the goal is to uniquely identify the file by
 * st_ino, we believe that using the 1. and 3. word of the FID is sufficient,
 * since the 2. word is a sequence number only.
 */
	inofid = (struct vms_ino *)(&st_buf->st_ino);
	ino2 = inofid->fid3;
	inofid->fid3 = inofid->fid2;
	inofid->fid2 = ino2;
	unix_ino = (ino_t *)(&inofid);
/*
 * SAMBA
 *
 * A quick and (very) dirty hack to overcome the lack of real
 * access checking (who want's to program it?). We just tell
 * the client the file may be writable, so the client will at least
 * *try* to do it's work. Then the underlying VMS security will of course reject
 * the request if ACL's won't allow. The client will issue an error
 * message, but without it will not even try if standard access says
 * NO but ACL's say YES.
 */
	st_buf->st_mode |= S_IWOTH;    /* probably lie! */
/*
 * SAMBA
 * The meaning of creation time and modification time seems to be
 * reversed in LANmanager. Thus, samba changes the two times.
 */
	if (getenv("SAMBA_SWAP_FILE_TIMES")) {
		t = st_buf->st_mtime;
		st_buf->st_mtime = st_buf->st_ctime;
		st_buf->st_ctime = t;
	}

	if (status == -2) {
/*
 * SAMBA
 * samba requires that stat() returns 0 in order to display the file.
 * since we want to see them at least with size 0 we fake a stat buffer.
 */
		char *p;
		memset(st_buf,0,sizeof(struct stat));
		p = strrchr(buf,'.');
		if (p != NULL  &&  strcmp(p,".DIR") == 0) {
			st_buf->st_mode |= S_IFDIR;
		}
		else {
			st_buf->st_mode |= S_IFREG;
		}
		status = 0;
	}
	if (status == 0) st_buf->st_nlink = 1;	/* "1 hard link" */

DEBUG (3,("vms_stat: st = %d, mode = %06o\n",status,st_buf->st_mode));
DEBUG (3,("          size = %d\n",st_buf->st_size));
DEBUG (8,("          ino = %08X, %04X\n",*unix_ino,inofid->fid3));
DEBUG (8,("          fab_rfm = %d\n",st_buf->st_fab_rfm));
DEBUG (8,("          fab_rat = %d\n",st_buf->st_fab_rat));
DEBUG (8,("          fab_mrs = %d\n",st_buf->st_fab_mrs));
	return(status);
}


int vms_lstat(const char *file_spec, struct stat *st_buf)
{
	return(vms_stat(file_spec,st_buf));
}


int vms_fstat(int fd, struct stat *st_buf)
{
	int status;
	time_t t;
	short ino2;
	struct vms_ino *inofid;

	status = fstat(fd, st_buf);

	inofid = (struct vms_ino *)(&st_buf->st_ino);
	ino2 = inofid->fid3;
	inofid->fid3 = inofid->fid2;
	inofid->fid2 = ino2;

	st_buf->st_mode |= S_IWOTH;    /* probably lie! see stat() */
/*
 * SAMBA
 * The meaning of creation time and modification time seems to be
 * reversed in LANmanager. Thus, samba changes the two times.
 */
	if (getenv("SAMBA_SWAP_FILE_TIMES")) {
		t = st_buf->st_mtime;
		st_buf->st_mtime = st_buf->st_ctime;
		st_buf->st_ctime = t;
	}

	if (status == -2) {
/*
 * samba requires that stat() returns 0 in order to display the file.
 * since we want to see them at least with size 0 we fake a stat buffer.
 */
		char *p;
		memset(st_buf,0,sizeof(struct stat));
		st_buf->st_mode |= S_IFREG;
		status = 0;
	}
	return(status);
}

/*
 * vms_get_attr
 *
 * This routine is added to make the extra elements in the DECC
 * stat structure available.
 */

#include <fab.h>
struct vms_attr {
	unsigned int	fab_rfm;
	unsigned int	fab_rat;
	unsigned int	fab_fsz;
	unsigned int	fab_mrs;
};

int vms_get_attr(const char *fname, struct vms_attr *attr)
{
	int status;
	struct stat st_crtl;

	if (attr == NULL || fname == NULL) return(-1);
	status = stat(fname,&st_crtl);
	if (status == 0) {
		attr->fab_rfm = st_crtl.st_fab_rfm;
		attr->fab_rat = st_crtl.st_fab_rat;
		attr->fab_fsz = st_crtl.st_fab_fsz;
		attr->fab_mrs = st_crtl.st_fab_mrs;
	}
}

int vms_is_textfile(const char *fname)
{
	struct vms_attr st_attr;

	if (fname == NULL) return(FALSE);
	if (vms_get_attr(fname, &st_attr) < 0) return(FALSE);
	if (st_attr.fab_rfm == FAB$C_VAR  &&  (st_attr.fab_rat & FAB$M_CR))
		return(TRUE);
	return(FALSE);
}
