/* 
   Unix SMB/Netbios implementation.
   Version 2.0.
   Copyright (C) Andrew Tridgell 1994-1999

   This file is part of the port to OpenVMS
   General VMS support routines.
   Copyright (C) Eckart Meyer 1996-1999
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*	vms_sup.c
 *	V2.0			28-Feb-1999	IfN/Mey
 *
 *	Copyright E. Meyer <meyer@ifn.ing.tu-bs.de>
 *+
 * Samba VMS support.
 * Misc. routines not present on VMS.
 *
 * NOTE: includes.h *IS* included here.
 *-
 */

/* #define __FCNTL_LOADED 1 */
#include "includes.h"

#include <starlet.h>
#include <lib$routines.h>
#include <descrip.h>
#include <dvidef.h>
#include <quidef.h>
#include <jpidef.h>

/* SAMBA debugging */
extern int DEBUGLEVEL;

struct itm_list_2 {
	unsigned short len;
	unsigned short code;
	void *bufadr;
};

struct itm_list_3 {
	unsigned short len;
	unsigned short code;
	void *bufadr;
	short int *retadr;
};

struct vms_timbuf {
	short int year;		/* year since 0 */
	short int mon;		/* month of year */
	short int mday;		/* day of month */
	short int hour;
	short int min;
	short int sec;
	short int hundred;
};

/*
 * Declarations
 */


/****************************************************************************
 lp_share_modes - HACK to switch off share modes
 ***************************************************************************/
BOOL vms_lp_share_modes(i)
{
	return(FALSE);
}

/****************************************************************************
 vms_set_process_name
 ***************************************************************************/
void vms_set_process_name(char *name)
{
	int lgth;
	char buf[16];	/* max 15 characters */
	struct dsc$descriptor prcnam;

	lgth = strlen(name);
	if (lgth > 15) lgth = 15;
	strncpy(buf,name,lgth);
	prcnam.dsc$a_pointer = buf;
	prcnam.dsc$w_length = lgth;
	sys$setprn(&prcnam);
}

/****************************************************************************
 vms_process_exists
 ***************************************************************************/
int vms_process_exists(int pid)
{
	unsigned long result;
	unsigned long sts;
	long item;

	item = JPI$_PID;
	sts = lib$getjpi(&item,&pid,0,&result,0,0);
	return(sts&1);
}



#define QUI_OK	(QUI$M_QUEUE_AVAILABLE | QUI$M_QUEUE_BUSY | QUI$M_QUEUE_IDLE)
#define QUI_ERROR  (QUI$M_QUEUE_STALLED | QUI$M_QUEUE_UNAVAILABLE)
/****************************************************************************
 vms_getqui_init - Establish a $GETQUI context for a specified queue.
 ****************************************************************************/
BOOL vms_getqui_init(char *queue_name, print_status_struct *status)
{
	unsigned long sts;
	unsigned long search_flags;
	unsigned long queue_status;
	struct itm_list_3 queue_list[4];
	unsigned long iosb[2];
/*
 * Cancel any previous context.
 */
	sts = sys$getquiw(0,QUI$_CANCEL_OPERATION,0,0,0,0,0);
	if (!(sts&1)) return(False);
/*
 * Establish queue context.
 */
	queue_list[0].code = QUI$_QUEUE_STATUS;
	queue_list[0].bufadr = &queue_status;
	queue_list[0].len = sizeof(queue_status);
	queue_list[0].retadr = NULL;

	queue_list[1].code = QUI$_SEARCH_NAME;
	queue_list[1].bufadr = queue_name;
	queue_list[1].len = strlen(queue_name);
	queue_list[1].retadr = NULL;

	queue_list[2].code = QUI$_SEARCH_FLAGS;
	queue_list[2].bufadr = &search_flags;
	queue_list[2].len = sizeof(search_flags);
	queue_list[2].retadr = NULL;

	queue_list[3].code = 0;
	queue_list[3].len = 0;

	search_flags = QUI$M_SEARCH_WILDCARD;

	sts = sys$getquiw(0,QUI$_DISPLAY_QUEUE,0,queue_list,iosb,0,0);
	DEBUG(5,("getqui_init:  sts = %08X, iosb = %08X\n",sts,iosb[0]));
	if ( (!(sts&1)) ||  (!(iosb[0]&1)) ) return(False);

	DEBUG(5,("              queue_status = %08X\n",queue_status));

	if (status) {
		if (queue_status & QUI_OK) {
			StrCpy(status->message,"Queue status is OK");
			status->status=LPSTAT_OK;
		}
		else if (queue_status & QUI_ERROR) {
			StrCpy(status->message,"Queue status is ERROR");
			status->status=LPSTAT_ERROR;
		}
		else {
			StrCpy(status->message,"Queue status is STOPPED");
			status->status = LPSTAT_STOPPED;
		}
	}

	return(True);
}

/****************************************************************************
 vms_getqui_entry - Get job information for next job.
 ***************************************************************************/
BOOL vms_getqui_entry(print_queue_struct *buf, BOOL first)
{
	unsigned long sts;
	unsigned long search_flags;
	struct itm_list_3 job_list[8];
	unsigned long iosb[2];
	char job_name[40];
	char username[13];
	long job_size;
	long entry_number;
	unsigned long job_status;
	unsigned long submission_time[2];
	short int job_name_len;
	short int username_len;
	struct vms_timbuf timbuf;

	job_list[0].code = QUI$_JOB_NAME;
	job_list[0].bufadr = job_name;
	job_list[0].len = 39;
	job_list[0].retadr = &job_name_len;

	job_list[1].code = QUI$_USERNAME;
	job_list[1].bufadr = username;
	job_list[1].len = 12;
	job_list[1].retadr = &username_len;

	job_list[2].code = QUI$_JOB_SIZE;
	job_list[2].bufadr = &job_size;
	job_list[2].len = sizeof(job_size);
	job_list[2].retadr = NULL;

	job_list[3].code = QUI$_ENTRY_NUMBER;
	job_list[3].bufadr = &entry_number;
	job_list[3].len = sizeof(entry_number);
	job_list[3].retadr = NULL;

	job_list[4].code = QUI$_JOB_STATUS;
	job_list[4].bufadr = &job_status;
	job_list[4].len = sizeof(job_status);
	job_list[4].retadr = NULL;

	job_list[5].code = QUI$_SUBMISSION_TIME;
	job_list[5].bufadr = submission_time;
	job_list[5].len = sizeof(submission_time);
	job_list[5].retadr = NULL;

	job_list[6].code = QUI$_SEARCH_FLAGS;
	job_list[6].bufadr = &search_flags;
	job_list[6].len = sizeof(search_flags);
	job_list[6].retadr = NULL;

	job_list[7].code = 0;
	job_list[7].len = 0;

	search_flags = QUI$M_SEARCH_ALL_JOBS;

	sts = sys$getquiw(0,QUI$_DISPLAY_JOB,0,job_list,iosb,0,0);
	DEBUG(5,("getqui_entry: sts = %08X, iosb = %08X\n",sts,iosb[0]));
	if ( (!(sts&1)) ||  (!(iosb[0]&1)) ) return(False);

	job_name[job_name_len] = '\0';
	username[username_len] = '\0';

	DEBUG(5,("              entry_number = %d\n",entry_number));
	DEBUG(5,("              job_size     = %d\n",job_size));
	DEBUG(5,("              job_status   = %08X\n",job_status));
	DEBUG(5,("              job_name     = {%s}\n",job_name));
	DEBUG(5,("              username     = {%s}\n",username));

	buf->job = entry_number;
	buf->size = job_size * 512;
	buf->status = (job_status & QUI$M_JOB_EXECUTING)?LPQ_PRINTING:LPQ_QUEUED;
	buf->time = time(NULL);
	StrnCpy(buf->user,username,sizeof(buf->user)-1);
	StrnCpy(buf->file,job_name,sizeof(buf->file)-1);

	sts = sys$numtim(&timbuf,submission_time);
	if (sts&1) {
		struct tm tm;
		int i;
		
		i = timbuf.year / 100;
		i = timbuf.year - (i * 100);
		tm.tm_year = i;
		tm.tm_mon = timbuf.mon;
		tm.tm_mday = timbuf.mday;
		tm.tm_hour = timbuf.hour;
		tm.tm_min = timbuf.min;
		tm.tm_sec = timbuf.sec;
		buf->time = mktime(&tm);
/*		buf->time += GMT_TO_LOCAL*TimeDiff(buf->time); /* have to fix this */
	}

	return(True);
}

/****************************************************************************
 get a printer queue
 ***************************************************************************/
int get_printqueue(int snum, 
		   connection_struct *conn,print_queue_struct **queue,
		   print_status_struct *status)
{
	char *printername = PRINTERNAME(snum);
	int ret=0,count=0;
/*
 * For VMS we use the $GETQUI system service to obtain queue status.
 * This is much more efficient than spawning a subprocess (which
 * is more expensive than in UNIX), creating and reading files.
 * The actual $GETQUI code is in VMS_SUP.C.
 */
	if (status) {
		StrCpy(status->message,"");
		status->status = LPSTAT_OK;
	}

	if (!vms_getqui_init(printername,status))  /* establish $GETQUI context */
		return(0);

	while (True) {
		*queue = Realloc(*queue,sizeof(print_queue_struct)*(count+1));
		if (! *queue) {
			count = 0;
			break;
		}
 
		/* get next entry */
		if (!vms_getqui_entry(&(*queue)[count],count==0))
		break;
 
		count++;
	}	      
	return(count);
}


/****************************************************************************
 vms_printing_string_sub
 This is special version of string_sub only used in printing.c to
 encode the file name for the printing command.
 This is really a hack to avoid changing the original code.
 ***************************************************************************/
void vms_printing_string_sub(char *s,const char *pattern,const char *insert)
{
	if (strcmp(pattern,"%f") == 0) {
/*		string_sub(s,pattern,vms_encode_filespec(insert,FALSE)); */
		string_sub(s,pattern,vms_get_filespec(insert));
	}
	else {
		string_sub(s,pattern,insert);
	}
}

/****************************************************************************
 statfs - return mounted file system info
 ***************************************************************************/
int vms_statfs(const char *path, struct statfs *buf)
{
	unsigned long iosb[2];
	struct itm_list_3 dvi_itm[4];
	unsigned long sts;
	char *name;
	struct dsc$descriptor dev;

	name = vms_get_filespec(path);
	DEBUG (3,("vms_statfs: path = \"%s\", vms_file = \"%s\"\n",path,name));

	dvi_itm[0].code = DVI$_MAXBLOCK;
	dvi_itm[0].len = 4;
	dvi_itm[0].bufadr = &buf->f_blocks;	/* total blocks */
	dvi_itm[0].retadr = NULL;

	dvi_itm[1].code = DVI$_FREEBLOCKS;
	dvi_itm[1].len = 4;
	dvi_itm[1].bufadr = &buf->f_bfree;	/* free blocks */
	dvi_itm[1].retadr = NULL;

	dvi_itm[2].code = DVI$_MAXFILES;
	dvi_itm[2].len = 4;
	dvi_itm[2].bufadr = &buf->f_files;	/* total files in file system */
	dvi_itm[2].retadr = NULL;

	dvi_itm[3].code = 0;
	dvi_itm[3].len = 0;

	dev.dsc$a_pointer = name;
	dev.dsc$w_length = strlen(dev.dsc$a_pointer);
	dev.dsc$b_class = DSC$K_CLASS_S;
	dev.dsc$b_dtype = DSC$K_DTYPE_T;

	sts = sys$getdvi(0,0,&dev,&dvi_itm,iosb,0,0,0);
	if ( (!(sts&1)) || (!(iosb[0]&1)) ) {
		DEBUG (0,("vms_statfs: $GETDVI ERROR: sts = %08X, iosb = %08X\n",
			sts,iosb[0]));
		return(-1);
	}
	
	buf->f_type = 0;
	buf->f_bsize = 512;		/* unsupported */
	buf->f_bavail = buf->f_bfree;	/* avaiable blocks to non-super-user */
	buf->f_ffree = -1;		/* free files, unsupported */
	DEBUG (4,("   f_blocks = %d\n",buf->f_blocks));
	DEBUG (4,("   f_bfree = %d\n",buf->f_bfree));
	DEBUG (4,("   f_files = %d\n",buf->f_files));
	return(0);
}


#ifdef OBSOLETE
/****************************************************************************
 waitpid (T.B.S.)
 ***************************************************************************/
pid_t vms_waitpid(pid_t pid, int *stat, int options)
{
	return(0);
}
#endif

#include <stdarg.h>
#ifdef getcwd
#undef getcwd
char *getcwd(char *buffer, unsigned int size, ...);
#endif
static char *cwd = NULL;
/****************************************************************************
 getcwd - get current directory
 ***************************************************************************/
char *vms_getcwd(char *buffer, unsigned int size, ...)
{
	char *p;
	char *q;
	int n;
	p = getcwd(NULL,1024,0);
	if (!p) return(NULL);
	q = vms_decode_filespec(p);
	free(p);
	n = strlen(q);
	if (size-1 < n) n = size - 1;
	if (buffer) {
		p = buffer;
		strncpy(p,q,n);
		p[n] = '\0';
	}
	else {
		if (size >= 0) {
			p = malloc(size+1);
			strncpy(p,q,n);
			p[n] = '\0';
		}
		else {
			p = malloc(strlen(q)+1);
			StrCpy(p,q);
		}
	}
	return(p);
}


/****************************************************************************
 delete - Get the VMS filespec and delete all versions of the file.
 ***************************************************************************/
int vms_delete (const char *path)
{
	char *buf;
	int sts;
        struct dsc$descriptor nam;
	char *p;

	p = vms_get_filespec(path);
	if(!p)
	{
		errno = ENOENT;
		return(-1);
	}
	buf = malloc(strlen(p)+20);
	if(!buf) {
		DEBUG(0,("VMS Delete: can't allocate file name buffer"));
		errno = ENOENT;
		return(-1);
	}
	StrCpy(buf,p);
	pstrcat(buf,";*");
	nam.dsc$a_pointer = buf;
	nam.dsc$w_length = strlen(nam.dsc$a_pointer);
	nam.dsc$b_class = DSC$K_CLASS_S;
	nam.dsc$b_dtype = DSC$K_DTYPE_T;
	sts = lib$delete_file(&nam, 0, 0, 0, 0, 0, 0, 0, 0);
	if (!(sts&1)) DEBUG(0,("VMS-error %d deleting %s\n", sts, buf));
	free(buf);
	if (!(sts&1)) {
		errno = EACCES;
		return -1;
	}
	return(0);
}


#ifdef rename
#undef rename
int rename (const char *__old, const char *__new);
#endif
/****************************************************************************
 rename - Rename files. Take care of VMS directory files.
 ***************************************************************************/
int vms_rename(const char *oldname, const char *newname)
{
	char *buf1;
	char *buf2;
//	char *p;
	int sts;

	DEBUG(3,("vms_rename: was: %s -> %s\n",oldname,newname));
	buf2 = vms_encode_filespec(oldname,FALSE);
	buf1 = malloc(strlen(buf2)+1);
	StrCpy(buf1,buf2);
	if(!buf1) {
		DEBUG(0,("VMS Rename: can't allocate file name buffer"));
		errno = ENOENT;
		return(-1);
	}
	buf2 = vms_encode_filespec(newname,FALSE);
#if 0
	buf1 = malloc(strlen(oldname)+20);
	buf2 = malloc(strlen(newname)+20);
	if(!buf1  ||  !buf2) {
		DEBUG(0,("VMS Rename: can't allocate file name buffer"));
		errno = ENOENT;
		return(-1);
	}
	VMSMakeFilename(buf1, oldname);
	VMSMakeFilename(buf2, newname);

/* Do we need the following??? */
        p = strrchr(buf1, '.');
        if( p && !strcmp(p, ".DIR")) {
                strcat(buf2, ".DIR");
	}
/*
 * If the filename does not contain a dot, VMS automatically uses
 * the file extension from the input file.
 * Thus, if the client request a rename to an empty file extension, we
 * add a single dot to the end of the new filename which prevents VMS
 * from using the original one.
 */
	p = strrchr(buf2, ']');		/* isolate file name from path */
	if (!p) p = buf2;
	p = strchr(p, '.');		/* any dots in file name ? */
	if (!p) strcat(buf2, ".");	/* no, append one */ 
#endif

	DEBUG(3,("vms_rename: VMS: %s -> %s\n",buf1, buf2));
	sts = rename(buf1, buf2);
	free(buf1);
	return(sts);
}


/****************************************************************************
 mkdir

 Yes, even VAXC has mkdir. But we would like to really know,
 what mkdir() really does...

 This is a special version for Samba: We ignore 'mode' and just
 use the parent directory protection.
 ***************************************************************************/
int vms_mkdir( const char *name, mode_t mode, ...)
{
	char buf[256], *cp, *subdir;
	int sts;
/*	unsigned short penable = 0xFFF0;  /* copy systemprotect from parent */
	unsigned short penable = 0;	  /* copy all from parent! */
	unsigned short protect = 0;
	struct dsc$descriptor nam;
	const char prot_unix2vms[8]={0,4,10,14,1,5,11,15};

	DEBUG(0,("vms_mkdir: %s, mode: %07o\n", name, mode));
#if 0
	subdir=strrchr( name,'/');
	if (subdir) {
		subdir++;
	}
	else {
		subdir = name;
	}
	cp=strrchr( name, '.');
	errno = EACCES;
	if( cp  &&  cp != name) return( -1); /* '.' in directory specification 
                                    i.e.  K:\> mkdir  aaa.bb         */

	StrCpy(buf,"[]");
	nam.dsc$a_pointer = VMSMakeFilename(buf+2, name);
	if(!nam.dsc$a_pointer) {
		return(-1);
	}
	cp=strrchr( nam.dsc$a_pointer, '.');
	if(cp && strcmp( subdir, cp+1) ) {
		errno = EEXIST;
		if( !strcasecmp( cp, ".DIR") )return -1;
		errno = EACCES;
	}   
	cp=strchr( nam.dsc$a_pointer, ']');
	if (!cp) nam.dsc$a_pointer = buf;
	DEBUG(8,("  -> %s\n", nam.dsc$a_pointer));
	cp=strchr( nam.dsc$a_pointer, ']');
	if (!cp) return(-1);

	*cp = '.';
	strcat( cp, "]");
#else
	nam.dsc$a_pointer = vms_get_filedir(name);
#endif
	nam.dsc$w_length = strlen(nam.dsc$a_pointer);
	nam.dsc$b_class = DSC$K_CLASS_S;
	nam.dsc$b_dtype = DSC$K_DTYPE_T;

	if (getenv("SAMBA_ALTERNATE_DIRECTORY_PROTECTION")) {
		protect = prot_unix2vms[mode&7];
		mode >>= 3;
		protect <<= 4; /*world*/
		protect |= prot_unix2vms[mode&7];
		mode >>= 3;
		protect <<= 4; /*group*/
		protect |= prot_unix2vms[mode&7];
		/*mode >>= 3;*/
		protect <<= 4; /*owner*/

		protect = ~protect;

		penable = 0xFFF0;  /* copy systemprotect from parent */
	}

	sts = lib$create_dir( &nam, 0, &penable, &protect, 0, 0);

	if (!(sts&1)) {
		DEBUG(3,("VMS mkdir: sts = %08X, name = %s, mask = %04X\n", sts, nam.dsc$a_pointer, protect));
		return -1;
	}
	return 0;
}


#if 0
#ifdef delete
#undef delete
int delete(const char *__path);
#endif
/****************************************************************************
 rmdir - Get the VMS directory filespec and delete the <path>.DIR;1 file.
 ***************************************************************************/
int vms_rmdir (char *path)
{
	char buf[256];
	int sts;
	char *p;

	p = VMSMakeFilename(buf, path);
	if (p == NULL) {
		errno = ENOTDIR;
		return(-1);
	}
	strcat(buf,";1");
	DEBUG(3,("Deleting directory %s -> %s\n",path,buf));
	sts = delete(buf);
	if (sts < 0) {
		DEBUG(3,("rmdir: errno = %d, vaxc$errno = %08X\n",errno,vaxc$errno));		
		if (errno == EVMSERR) {
			if (vaxc$errno == 0x0001C032) errno = ENOTEMPTY;
		}
		return(-1);
	}
	return(0);
}
#else
/****************************************************************************
 rmdir - Get the VMS directory filespec and delete the <path>.DIR;1 file.
 ***************************************************************************/
int vms_rmdir (char *path)
{
	return(vms_delete(path));
}
#endif


#ifdef opendir
#undef opendir
DIR *opendir (const char *dirname);
#endif
/****************************************************************************
 opendir
 ***************************************************************************/
DIR *vms_opendir (const char *dirname)
{
//	return(opendir(vms_get_filedir(dirname)));
	return(opendir(vms_encode_filespec(dirname,TRUE)));
}


#ifdef readdir
#undef readdir
struct dirent *readdir (DIR * dirp);
#endif
/****************************************************************************
 readdir
 ***************************************************************************/
struct dirent *vms_readdir (DIR * dirp)
{
	struct dirent *d;
	char *p;
	d = readdir(dirp);
	if (d) {
//		p = vms_get_unix_path(d->d_name);
		p = vms_decode_filespec(d->d_name);
		pstrcpy(d->d_name, p);
	}
	return(d);
}


#ifdef chown
#undef chown
int chown  (const char *__file_spec, uid_t __owner, gid_t __group);
#endif
/****************************************************************************
 chown

 Note that filename conversion may fail since we can't distinguish
 between files and directories, but the encode routine must know.
 ***************************************************************************/
int vms_chown  (const char *file_spec, uid_t owner, gid_t group)
{
	return(chown(vms_encode_filespec(file_spec,FALSE), owner, group));
}


#ifdef chmod
#undef chmod
int chmod  (const char *__file_spec, mode_t __mode);
#endif
/****************************************************************************
 chmod

 Note that filename conversion may fail since we can't distinguish
 between files and directories, but the encode routine must know.
 ***************************************************************************/
int vms_chmod  (const char *file_spec, mode_t mode)
{
	return(chmod(vms_encode_filespec(file_spec,FALSE), mode));
}


#ifdef fopen
#undef fopen
FILE *fopen (const char *__filename, const char *__mode, ...);
#endif
/****************************************************************************
 fopen
 ***************************************************************************/
FILE *vms_fopen (const char *filename, const char *mode, ...)
{
	char *p;
	FILE *fp;
	int dbg_save = DEBUGLEVEL;
/*
 * We've to turn off debug output since the debug routine could try to
 * open the log via fopen... You've got it....
 */
	DEBUGLEVEL = 0;
	p = vms_encode_filespec(filename,FALSE);
	fp = fopen(p,mode);
	DEBUGLEVEL = dbg_save;
	return(fp);
//	return(fopen(vms_encode_filespec(filename,FALSE),mode,"ctx=stm","rfm=stmlf","rat=cr","mrs=0"));
}


/****************************************************************************
 system

 This version of the system() call tries to detect an output redirection
 in the UNIX compatible command and if found, spawns the subprocess
 with SYS$OUTPUT assigned to the specified path.

 Input redirection is not supported - SYS$INPUT is set to NLA0:

 Since _POSIX_EXIT is defined, the header files specify the __posix_ prefix
 ***************************************************************************/
//int __posix_vms_system(const char *cmdline)
int vms_system(const char *cmdline)
{
	char buf[256];
	char buf2[256];
	$DESCRIPTOR(in,"NLA0:");
	$DESCRIPTOR(out,"SYS$OUTPUT");
	$DESCRIPTOR(cmd,"");
	unsigned long sts;
	unsigned long compl;
	char *p, *q;

/*
 * Remove quotes
 */
	p = (char *)cmdline;
	if (cmdline[0] == '\"') p++;
	StrCpy(buf,p);
	if(strlen(buf) > 0)
		if (buf[strlen(buf) - 1] == '\"') buf[strlen(buf) - 1] = '\0';
/*
 * Find command and check if this is a file. If so check on .COM file
 * and prepend "@".
 * If the command is not a filename, use it as an DCL command.
 * Note that this code could be much more flexible, but as we
 * know what samba does, this might do.
 */
	StrCpy(buf2,buf);
	p = strchr(buf,' ');
	if (!p) p = buf + strlen(buf);	/* point behind command */
	*p = '\0';
	q = vms_get_filespec(buf);
	if (strcmp(q + strlen(q) - 4, ".COM") == 0) {
		StrCpy(buf,"@");
		pstrcat(buf,q);		/* append filename */
		p = p - buf + buf2;	/* point to buf2 */
		pstrcat(buf, p);	/* rest */
	}
	else {
		if (p < buf + strlen(buf)) *p = ' '; /* restore command separator */
	}
	cmd.dsc$a_pointer = buf;
/*
 * Find output redirection
 */
	p = strrchr(cmd.dsc$a_pointer,'>');
	if (p != NULL) {
		for (q = p + 1; *q == ' ' && *q != '\0'; q++);
		out.dsc$a_pointer = vms_get_filespec(q);
		if (strcmp(q,"/dev/null") == 0) {
			out.dsc$a_pointer = in.dsc$a_pointer;
		}
		if (p != cmd.dsc$a_pointer && *(p-1) == '>') p--;
		*p = '\0';
	}
/*
 * Spawn subprocess
 */
	cmd.dsc$w_length = strlen(cmd.dsc$a_pointer);
	out.dsc$w_length = strlen(out.dsc$a_pointer);
	sts = lib$spawn(&cmd,&in,&out,0,0,0,&compl,0,0,0,0,0,0);
DEBUG(5,("system: {%s}, output: {%s}\n",cmd.dsc$a_pointer,out.dsc$a_pointer));
DEBUG(5,("        sts = %08X, compl = %08X\n",sts,compl));
	if (!(sts&1)) return(127);
	return((int)compl);
}




#include <iodef.h>
#include <ttdef.h>
#include <descrip.h>

static $DESCRIPTOR(termdesc, "SYS$INPUT:");
static short termchan = -1;

/****************************************************************************
 set_noecho - just a utility routine to cancel term echo
 ***************************************************************************/
static void vms_set_noecho()
{
	int st;
	int buf[3];
	short int iosb[4];

	if (termchan == -1) {
		sys$assign(&termdesc,&termchan,0,0);
	}
	st = sys$qiow(0,termchan,IO$_SENSEMODE,iosb,0,0,buf,12,0,0,0,0);
	buf[1] |= TT$M_NOECHO;
	st = sys$qiow(0,termchan,IO$_SETMODE,iosb,0,0,buf,12,0,0,0,0);
}

/****************************************************************************
 set_echo - just a utility routine to start term echo
 ***************************************************************************/
static void vms_set_echo()
{
	int st;
	int buf[3]; 
	short int iosb[4];

	if (termchan == -1) {
		sys$assign(&termdesc,&termchan,0,0);
	}
	st = sys$qiow(0,termchan,IO$_SENSEMODE,iosb,0,0,buf,12,0,0,0,0);
	buf[1] &= ~TT$M_NOECHO;
	st = sys$qiow(0,termchan,IO$_SETMODE,iosb,0,0,buf,12,0,0,0,0);
}


static char pwd[256];
/****************************************************************************
 getpass - read with prompt and don't echo
 ***************************************************************************/
char *getpass(prompt)
char * prompt;
{
	void vms_set_echo();
	void vms_set_noecho();

	vms_set_noecho();
	printf("%s",prompt);
	gets(pwd);
	vms_set_echo();
	return(pwd);
}


#ifdef select
#undef select
int select (int __nfds, fd_set *__readfds, fd_set *__writefds, 
			fd_set *__exceptfds, struct timeval *__timeout);
#endif
/****************************************************************************
 select - not more than 32 fds
 ***************************************************************************/
int vms_select (int nfds, fd_set *readfds, fd_set *writefds, 
			fd_set *exceptfds, struct timeval *timeout)
{
	return(select ((nfds > 32 ? 32 : nfds), readfds, writefds,
			exceptfds, timeout));
}


/****************************************************************************
 do_file_lock
 This routine is a dummy since on VMS each opened file implicity locked.
 On the other hand, we do not habe the locking functions of fcntl()
 ***************************************************************************/
BOOL do_file_lock(int fd, int waitsecs, int type)
{
	return(TRUE);
}


/****************************************************************************
 fcntl_lock
 ***************************************************************************/
BOOL fcntl_lock(int fd, int op, SMB_OFF_T offset, SMB_OFF_T count, int type)
{
	return(TRUE);
}


/****************************************************************************
 lp_share_modes - share modes cannot be used.
 ***************************************************************************/
BOOL lp_share_modes(int i)
{
	return(FALSE);
}


/****************************************************************************
 lp_locking - locking cannot be used.
 ***************************************************************************/
BOOL lp_locking(int i)
{
	return(FALSE);
}


/****************************************************************************
 dummy_true - routine always returns TRUE
 ***************************************************************************/
BOOL dummy_true(void)
{
	return(TRUE);
}


/* ======= keyboard ========================================== */

static int chk_count = 0;
/****************************************************************************
check if something came in (same functions than wait_keyboard()).
called from vms_read_keyboard() timer routine at AST level!
****************************************************************************/
void vms_check_net(struct cli_state *cli)
{
	fd_set fds;
	int selrtn;
	struct timeval timeout;

	FD_ZERO(&fds);
	FD_SET(cli->fd,&fds);
	do {
		timeout.tv_sec = 0;
		timeout.tv_usec = 0;
		selrtn = sys_select(cli->fd + 1,&fds,&timeout);
	}
	while(selrtn < 0 && errno == EINTR);

	/* We deliberately use receive_smb instead of
	   client_receive_smb as we want to receive
	   session keepalives and then drop them here.
	*/
	if (FD_ISSET(cli->fd,&fds))
		receive_smb(cli->fd,cli->inbuf,0);
      
	chk_count++;
	if (chk_count > 20) {	/* 20 seconds between chkpath() calls */
		chk_count = 0;
		cli_chkpath(cli, "\\");
	}
}

/****************************************************************************
 vms_read_keyboard

 This routine has to read from keyboard and to check the socket for
 incoming data (see wait_keyboard in client.c).

 As we can't use select() for the keyboard we call the keyboard read
 routine synchroneously. Before, a cyclic timer is started which
 checks the network. Note that the checking routine is called at
 AST level. The only drawback is that there is a delay until a
 network action is noticed, but this seems no problem here.
 
 To enable command line editing we use the SMG routines to read from
 keyboard.

// I tried to strictly separate the VMS specific code from the base
// samba code. So this routine is in a common VMS support module but has
// a callback into the client.c. To avoid unresolved symbols with executables
// other than smbclient, the callback address is included as a parameter
// to vms_read_keyboard.

 ***************************************************************************/
#include <smgdef.h>
#include <smgmsg.h>
#include <ssdef.h>
#include <smg$routines.h>

static unsigned long smg_kb_id = -1;
static long time_bin[2];
char time_string[] = "0 00:00:01.01";
$DESCRIPTOR(time_dsc,time_string);
static struct cli_state *smbcli;
//static char *smbbuffer;
//static void (* client_chk_routine)();

void timer_ast(int reqid)
{
	unsigned long sts;

/*	DEBUG (3,("== timer AST ==\n")); /**/
//	(* client_chk_routine)(smbbuffer);	/* Callback to client.c */
	vms_check_net(smbcli);
	sts = sys$setimr(0,time_bin,timer_ast,reqid,0);	/* restart */
}

//int vms_read_keyboard(char *prompt, char *line, int size, char *buffer,
//		void (* chk_routine)() )
int vms_read_keyboard(char *prompt, char *line, int size, struct cli_state *cli)
{
	unsigned long sts;
	int retlen;
	struct dsc$descriptor_s dsc_line, dsc_prompt;
	struct dsc$descriptor_s *prm;
	void vms_exit_handler();

	smbcli = cli;
//	client_chk_routine = chk_routine;
	dsc_line.dsc$a_pointer = line;
	dsc_line.dsc$w_length = size - 1;
	dsc_line.dsc$b_class = DSC$K_CLASS_S;
	dsc_line.dsc$b_dtype = DSC$K_DTYPE_T;
	if (prompt == NULL) {
		prm = NULL;
	}
	else {
		dsc_prompt.dsc$a_pointer = prompt;
		dsc_prompt.dsc$w_length = strlen(prompt);
		dsc_prompt.dsc$b_class = DSC$K_CLASS_S;
		dsc_prompt.dsc$b_dtype = DSC$K_DTYPE_T;
		prm = &dsc_prompt;
	}

	if (smg_kb_id == -1) {
		sts = smg$create_virtual_keyboard(&smg_kb_id,0,0,0,0);
		if (!(sts&1)) {
			DEBUG (0,("can't create SMG keyboard - sts = %08X\n",sts));
			return(-1);
		}
		sts = sys$bintim(&time_dsc,time_bin);
		if (!(sts&1)) {
			DEBUG (0,("$BINTIM error - sts = %08X\n",sts));
			return(-1);
		}
	}
/*
 * Start cyclic timer.
 */
	sts = sys$setimr(0,time_bin,timer_ast,1,0);
	if (!(sts&1)) {
		DEBUG (0,("$SETIMR error - sts = %08X\n",sts));
		return(-1);
	}
/*
 * Read from keyboard.
 */
	retlen = 0;
	sts = smg$read_string(&smg_kb_id, &dsc_line, prm,
				0, 0, 0, 0, &retlen, 0,0,0,0,0,0);
	*(dsc_line.dsc$a_pointer + retlen) = '\0';
/*
 * We got a command line. Cancel timer and return.
 */
	sys$cantim(1,0);
	if (sts == SS$_NORMAL) return(0);
	if (sts == SMG$_EOF) return(-1);
	DEBUG (0,("keyboard read error: sts = %08X\n",sts));
	return(-1);
}

#if 1	/* for debug only */
/****************************************************************************
 dmphx - dump a buffer to the debug log in hex and ascii
 ***************************************************************************/
void _dmphx(char *msg,int len)
{
  int i, j;
  unsigned char *v, *w;

  v = w = (unsigned char *)msg;
  j = len;
  while (j>0) {
     if (j>16) i = 16; else i=j;
     for(;i>0;i--,v++) Debug1(" %02x",*v);
     if (j<16) for(i=16-j;i>0;i--) Debug1("   ");
     Debug1("        ");
     if (j>16) i = 16; else i=j;
     for(;i>0;i--,w++) Debug1("%c",(*w>0x1f && *w<0x80)? *w : '.');
     Debug1("\n");
     j -= 16;
  }
}
#endif
