/* PYVMSMODULE.C -- VMS/Python -- 19-NOV-1997 Uwe Zessin

The 'pyvms' module provides access to vms-related components:
C-RTL functions, command history, item codes (fac$_name), bitmasks (fac$M_name),
constants (fac$C_name or fac$K_name) and other information.

   29-NOV-1997 ZE. -- changes for Python V1.5B1
   02-DEC-1997 ZE. -- merge VMSDEF into PYVMS
   14-DEC-1997 ZE. -- Include LIB$ROUTINES.H + SMG$ROUTINES.H.
   29-JUL-1998 ZE. -- test-interface for quad2PYlong-integer conversion tests
   04-AUG-1998 ZE. -- test-interface for PYlong-integer2quad conversion tests
   14-NOV-1998 ZE. -- interface to pyvms_uaf_get_usernames
			- code in PYVMS__UAF_ROUTINES.C
   27-DEC-1998 ZE. -- test-interface for Uquad2PYlong-integer conversion tests
   31-DEC-1998 ZE. -- begin pyvms_crtl_open ()
   09-JAN-1999 ZE. -- test-interface for octa2PyLong conversion tests
   09-JAN-1999 ZE. -- test-interface for PyLong2octa conversion tests
   20-JAN-1999 ZE. -- test-interface for vmsobj_iosb_new()
   24-JAN-1999 ZE. -- test-interface for vmsobj_fab_new()
   25-JAN-1999 ZE. -- test-interface for vmsobj_rab_new()
   03-FEB-1999 ZE. -- test-interface for vmsobj__membuf_new()
   07-FEB-1999 ZE. -- test-interface for vmsobj_xaball_new()
   10-FEB-1999 ZE. -- test-interface for vmsobj_nam_new() + vmsobj_xabdat_new()
   13-FEB-1999 ZE. -- test-interface for vmsobj_xabfhc_new()
   18-FEB-1999 ZE. -- test-interface for vmsobj_xabitm_new()
   22-FEB-1999 ZE. -- test-interface for vmsobj_xabkey_new()
   06-MAR-1999 ZE. -- interface to pyvms_crtl_from_vms() + pyvms_crtl_to_vms()
			- code in PYVMS__CRTL_ROUTINES.C
			move pyvms_crtl_open() there, too
   08-MAR-1999 ZE. -- test-interface for vmsobj_xabpro_new() +
			vmsobj_xabrdt_new() + vmsobj_xabsum_new()
   13-MAR-1999 ZE. -- test-interface for vmsobj_xabtrm_new()
   15-MAR-1999 ZE. -- move vmsobj_* test-interface to PYVMS__VMSOBJ_ROUTINES.C
   11-MAY-1999 ZE. -- FB- pyvms_test__pyl2o, o_octaword[4] (was [2])
   29-MAY-1999 ZE. -- test-interface for vmsobj_lksb_new()
   07-JUL-1999 ZE. -- fix memory leak in pyvms_item_get()
   21-AUG-1999 ZE. -- test-interface for vmsobj__ownership_names_new()
*/

/* ------------------------------------------------------------------------- */

#include "python.h"
/* includes also: <string.h>, <errno.h> */
#include "vmsdef.h"

#include <lib$routines.h>	/* LIB$name	 */
#include <smg$routines.h>	/* SMG$name */
#include <ssdef.h>		/* SS$_NAME */

/* ------------------------------------------------------------------------- */

/* numeric value of current VMS version (PYVMS_GLOBAL.C) */
extern vmsdef_xw_vmsver vmsdef_gw_vmsver;

/* VMS version string to numeric code translation table (VMSDAT_VMSVER.MAR) */
/* @unused, yet -- extern struct vmsdef_xr_vmsvertbl VMSDEF_GR_VMSDAT_VMSVER[];*/

/* list of $xxxDEF */
extern struct vmsdef_xr_defdirtbl VMSDEF_GR_DIRECTORY[];

/* ------------------------------------------------------------------------- */

#include "pyvms_readline.h"

/* SMG$ support */
extern long				pyvms_gl_keyboard_id;	/* SMG$ */
extern long				pyvms_gl_key_table_id;	/* SMG$ */
extern struct pyvms_xr_rlbuffer*	pyvms_gar_rlbuffer;	/* listhead */
/* ------------------------------------------------------------------------- */

PyObject *pyvms_gr_error;		/* exception pyvms.error */

static char pyvms__doc__ [] =
"The 'pyvms' module provides access to vms-related components:\n\
command history, item codes (fac$_name), bitmasks (fac$M_name) and\n\
other information.";

static char pyvms_t_cant_init [] =
"can't initialize module pyvms";

static char pyvms_t_cant_deferr [] =
"can't define pyvms.error";

static char pyvms_t_cant_defvms [] =
"can't define pyvms.vms_version_number";

/* ------------------------------------------------------------------------- */

static char pyvms_crtl_from_vms__doc__[] =
"unix_filespec = pyvms.crtl_from_vms\
 (vms_filespec, action_routine, wild_flag)\n\
Convert OpenVMS filespec into UNIX style filespec.";

extern PyObject *
pyvms_crtl_from_vms (PyObject *self, PyObject *args);
/* code in PYVMS__CRTL_ROUTINES.C */

/* ------------------------------------------------------------------------- */

static char pyvms_crtl_open__doc__[] =
"pyvms.crtl_open (filename, flag [, mode=0777] [,'RMS arguments']) -> fd\n\
Open a file (for low level IO).";

extern PyObject *
pyvms_crtl_open (PyObject *self, PyObject *args);
/* code in PYVMS__CRTL_ROUTINES.C */

/* ------------------------------------------------------------------------- */

static char pyvms_crtl_to_vms__doc__[] =
"vms_filespec = pyvms.crtl_to_vms\
 (unix_filespec, action_routine, allow_wild, no_directory)\n\
Convert UNIX style filespec into OpenVMS filespec.";

extern PyObject *
pyvms_crtl_to_vms (PyObject *self, PyObject *args);
/* code in PYVMS__CRTL_ROUTINES.C */

/* ------------------------------------------------------------------------- */

/* delete 'l_delete_lines' from the beginning of the recall buffer */

/* Note: this does not change the contents of the SMG$ buffer */

long pyvms__history_delete (long l_delete_lines)
{
	struct pyvms_xr_rlbuffer	* ar_rlbuffer;
	unsigned long			  l_deleted_count;

	if (pyvms_gar_rlbuffer == NULL)
	{
	    return 0;		/* nothing in the buffer - nothing deleted */
	}

	l_deleted_count = 0;
	while (l_deleted_count < l_delete_lines)
	{
	    if (pyvms_gar_rlbuffer == NULL)
	    {
		break;				/* no more lines to delete */
	    }

	    ar_rlbuffer = pyvms_gar_rlbuffer;
	    /* remove first element from list */
	    pyvms_gar_rlbuffer = ar_rlbuffer->ar_next_rlbuff;
	    (void) free (ar_rlbuffer->at_line);		/* drop line */
	    if (ar_rlbuffer->at_prompt != NULL)
	    {
		(void) free (ar_rlbuffer->at_prompt);	/* drop prompt */
	    }
	    (void) free (ar_rlbuffer);		/* drop descriptor */
	    l_deleted_count++;			/* one more line deleted */
	}

	return l_deleted_count;			/* number of deleted lines */
} /* long pyvms__history_delete (long l_delete_lines) */

/* ------------------------------------------------------------------------- */

/* PYVMS methods */

/* d = pyvms.history_delete(n);
     n = 0 -> delete all
     n > 0 -> delete from begin of list (oldest lines first)
     n < 0 -> delete from end   of list (newest lines)
   d -> number of deleted lines
*/
static PyObject *
pyvms_history_delete (PyObject *self, PyObject *args)
{
	struct pyvms_xr_rlbuffer	* ar_rlbuffer;
	unsigned long			  l_deletecount;
	unsigned long			  l_deleted_count;
	unsigned long			  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "i", &l_deletecount))
	{
	    return NULL;
	}

	/* -------------------- */
	if (pyvms_gar_rlbuffer == NULL)
	{
	    /* nothing in the buffer - so nothing deleted */
	    return PyInt_FromLong((long)0);
	}

	/* -------------------- */
	if (l_deletecount == 0)
	{
	    /* delete current virtual keyboard (and the recall buffer) */
	    l_status = smg$delete_virtual_keyboard(&pyvms_gl_keyboard_id);
	    /* @@ ignore any errors */

	    /* re-create virtual keyboard */
	    l_status = smg$create_virtual_keyboard (&pyvms_gl_keyboard_id);
	    /* this call MUST succeed ! */
	    if (l_status != SS$_NORMAL)
	    {
		printf ("%%PYVMS_history_delete(): SMG$CREATE_VIRTUAL_KEYBOARD() failed\n");
		lib$stop (l_status);
	    }

	    /* delete whole buffer - I doubt any one uses THAT MANY lines! */
	    l_deletecount = 0x7fffffff;
	    l_deleted_count = pyvms__history_delete (l_deletecount);
	    return PyInt_FromLong(l_deleted_count);
	}

	/* -------------------- */
	if (l_deletecount > 0)
	{
	    /* delete number of given lines from BEGIN of buffer */
	    l_deleted_count = pyvms__history_delete (l_deletecount);
	    return PyInt_FromLong(l_deleted_count);
	}

	/* -------------------- */
	/* delete number of given lines from END of buffer */
	/* @@@ not implemented, yet */

	/* -------------------- */
	Py_INCREF(Py_None);
	return Py_None;
} /* pyvms_history_delete () */

/* ------------------------------------------------------------------------- */

/* get contents of history buffer as a list of tuples (prompt,command) */

/* l = pyvms.history_get_size() */
static PyObject *
pyvms_history_get (PyObject *self, PyObject *args)
{
	PyObject			* ar_list;
	PyObject			* ar_tuple;
	PyObject			* ar_prompt;
	PyObject			* ar_command;

	struct pyvms_xr_rlbuffer	* ar_rlbuffer;
	unsigned long			  l_error;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	if (pyvms_gar_rlbuffer == NULL)
	{
	    Py_INCREF(Py_None);		/* nothing in the buffer */
	    return Py_None;
	}

	/* -------------------- */
	ar_list = PyList_New(0);	/* create list object */
	if (ar_list == NULL)
	{
	    return NULL;		/* failed */
	}

	ar_rlbuffer = pyvms_gar_rlbuffer;
	while (1)
	{
	    if (ar_rlbuffer == NULL)
	    {
		break;
	    }

	    ar_prompt = PyString_FromString(ar_rlbuffer->at_prompt);
	    if (ar_prompt == NULL)
	    {
		Py_DECREF(ar_list);	/* error, drop list */
		return NULL;
	    }

	    ar_command = PyString_FromString(ar_rlbuffer->at_line);
	    if (ar_command == NULL)
	    {
		Py_DECREF(ar_prompt);	/* error, first element of tuple */
		Py_DECREF(ar_list);	/* error, drop list */
		return NULL;
	    }

	    ar_tuple = PyTuple_New(2);	/* build tuple */
	    if (ar_tuple == NULL)
	    {
		Py_DECREF(ar_prompt);	/* error, first element of tuple */
		Py_DECREF(ar_command);	/* error, second element of tuple */
		Py_DECREF(ar_list);	/* error, drop list */
		return NULL;
	    }

	    PyTuple_SetItem(ar_tuple, 0, ar_prompt);
	    PyTuple_SetItem(ar_tuple, 1, ar_command);

	    l_error = PyList_Append(ar_list, ar_tuple);
	    Py_DECREF(ar_tuple);
	    if (l_error != 0)
	    {
		Py_DECREF(ar_list);	/* error, drop list */
		return NULL;
	    }

	    ar_rlbuffer = ar_rlbuffer->ar_next_rlbuff;
	} /* while (1) */

	return ar_list;
} /* pyvms_history_get () */

/* ------------------------------------------------------------------------- */

/* pyvms.history_show(); */
static PyObject *
pyvms_history_show (PyObject *self, PyObject *args)
{
	struct pyvms_xr_rlbuffer	* ar_rlbuffer;
	unsigned long			  l_linecount;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	if (pyvms_gar_rlbuffer != NULL)
	{
	    l_linecount = 1;
	    ar_rlbuffer = pyvms_gar_rlbuffer;
	    while (1)
	    {
		/* Note: t_line includes '\n' */
/* @@ Py-outroutine */ printf ("%2d %s", l_linecount, ar_rlbuffer->at_line);

		if (ar_rlbuffer->ar_next_rlbuff == NULL)
		{
		    break;
		}

		ar_rlbuffer = ar_rlbuffer->ar_next_rlbuff;
		l_linecount++;
	    }
	}

	/* -------------------- */
	Py_INCREF(Py_None);
	return Py_None;
} /* pyvms_history_show () */

/* ------------------------------------------------------------------------- */

/* n = pyvms.history_size()*/
static PyObject *
pyvms_history_size (PyObject *self, PyObject *args)
{
	struct pyvms_xr_rlbuffer	* ar_rlbuffer;
	unsigned long			  l_linecount;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	if (pyvms_gar_rlbuffer == NULL)
	{
	    l_linecount = 0;		    /* nothing in the buffer */
	}
	else
	{
	    l_linecount = 1;
	    ar_rlbuffer = pyvms_gar_rlbuffer;
	    while (1)
	    {
		if (ar_rlbuffer->ar_next_rlbuff == NULL)
		{
		    break;
		}
		ar_rlbuffer = ar_rlbuffer->ar_next_rlbuff;
		l_linecount++;
	    }
	}

	/* -------------------- */
	return PyInt_FromLong(l_linecount);
} /* pyvms_history_size () */

/* ------------------------------------------------------------------------- */

/* val= pyvms.item_get ('$DVIDEF', 'DVI$_CYLINDERS', None/1); */
static PyObject *
pyvms_item_get (PyObject *self, PyObject *args)
{
	char			  * at_def_name;
	unsigned long		    l_def_len;
	char			  * at_item_name;
	unsigned long		    l_item_length;
	PyObject		  * ar_option;
	PyObject		  * ar_dict;
	PyObject		  * ar_intobj;
	struct vmsdef_xr_defdirtbl* ar_defdirtbl;
	struct vmsdef_xr_itmtbl	  * ar_itmtbl;

	/* -------------------- */
	ar_option = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#s#|O",
	    &at_def_name,  &l_def_len,
	    &at_item_name, &l_item_length,
	    &ar_option))
	{
	    return NULL;
	}

	/* -------------------- */
	/* look up define-name */
	ar_defdirtbl = &VMSDEF_GR_DIRECTORY[0];
	ar_itmtbl    = NULL;
	while (ar_defdirtbl->ar_itmtbl != NULL)
	{
	    if (strcmp(at_def_name,ar_defdirtbl->at_vmsdefnam) == 0)
	    {
		ar_itmtbl = ar_defdirtbl->ar_itmtbl;
		break;
	    }
	    ar_defdirtbl++;
	}

	/* -------------------- */
	if (ar_defdirtbl->ar_itmtbl == NULL)
	{
	    /* item table not found */
	    PyErr_SetString(PyExc_ValueError, "argument 1: unknown DEFine");
	    return NULL;
	}

	/* -------------------- */
	/* look up item code */
	while (ar_itmtbl->w_vmsvermin != 0)
	{
	    /* note: this lookup is independed from version number and    */
	    /*       processor architecture! That's a feature, not a bug! */
		/* @@ unsolved problem: multiple entries of the same item
		   code for several VMS versions */
	    if (strcmp(at_item_name,ar_itmtbl->at_itmnam_py) == 0)
	    {
		/* assume no bitmask is to be returned */
		ar_option = NULL;

		/* check if bitmask bits should be returned */
		/* just test if any argument other than None was given )
		/* all elements of the bitmask are returned for any version */
		if (! (ar_option == Py_None))
		{
		  if (! (ar_itmtbl->l_flags & ITMTBL_M_CVTFNC))
		  {
		    /* ar_itmtbl->ar_bitmask really points to bitmsk */
		    struct vmsdef_x_bitmsk	* ar_bitmsk;
		    PyObject			* ar_list;
		    PyObject			* ar_item;
		    unsigned long		  l_error;

		    /* address of bitmask bit structure */
		    ar_bitmsk = ar_itmtbl->itmtbl_ar_bitmsk;

		    ar_list = PyList_New(0); /* create list object */
		    if (ar_list == NULL)
		    {
		      return NULL;		/* failed */
		    }

		    if (ar_bitmsk != NULL)
		    {
		      /* fill list, if bitmask exists */
		      while (ar_bitmsk->w_vmsvermin != 0)
		      {
			ar_item = PyString_FromString(ar_bitmsk->at_bit_name);
			if (ar_item == NULL)
			{
			    Py_DECREF(ar_list);	/* error, drop list */
			    return NULL;
			}

			l_error = PyList_Append(ar_list, ar_item);
			Py_DECREF(ar_item);
			if (l_error != 0)
			{
			    Py_DECREF(ar_list); /* error, drop list */
			    return NULL;
			}
			ar_bitmsk++;	/* point to next bitmask element */
		      } /* while (ar_bitmsk->w_vmsvermin != 0) */
		      /* @@ optional: status = PyList_Sort(ar_list); */

		      /* return list */
		      ar_option = ar_list;
		    } /* if (ar_bitmsk != NULL) */
		  } /* if (! (ar_itmtbl->l_flags & ITMTBL_M_CVTFNC)) */
		} /* if (! (ar_option == Py_None)) */

		/* return data in a dictionary */
		ar_dict = PyDict_New();
		if (ar_dict == NULL)
		{
		    if (ar_option != NULL)
		    {
			/* drop list, if it has been created */
			Py_XDECREF(ar_option);
		    }
		}

		ar_intobj = PyInt_FromLong((long)ar_itmtbl->w_itmcod);
		PyDict_SetItemString(ar_dict, "item_code", ar_intobj);

		ar_intobj = PyInt_FromLong((long)ar_itmtbl->w_bufsiz);
		PyDict_SetItemString(ar_dict, "buffer_size", ar_intobj);

		ar_intobj = PyInt_FromLong((long)ar_itmtbl->w_vmsvermax);
		PyDict_SetItemString(ar_dict, "vms_version_max", ar_intobj);

		ar_intobj = PyInt_FromLong((long)ar_itmtbl->w_vmsvermin);
		PyDict_SetItemString(ar_dict, "vms_version_min", ar_intobj);

		ar_intobj = PyInt_FromLong( (long)
			(ar_itmtbl->l_flags & ITMTBL_M_ARCH_VAX)   ? 1 : 0);
		PyDict_SetItemString(ar_dict, "applies_to_vax", ar_intobj);

		ar_intobj = PyInt_FromLong( (long)
			(ar_itmtbl->l_flags & ITMTBL_M_ARCH_ALPHA) ? 1 : 0);
		PyDict_SetItemString(ar_dict, "applies_to_alpha", ar_intobj);

		/* @@ ?? w_typ_lib + w_typ_sys -W- unsigned bit! */
		{
		  long l_intobj;

		  if (
		      (ar_itmtbl->itmtbl_ar_bitmsk == NULL) ||
		      (ar_itmtbl->l_flags & ITMTBL_M_CVTFNC)
		     )
		  {
		    l_intobj = 0;
		  }
		  else
		  {
		    l_intobj = 1;
		  }
		  ar_intobj = PyInt_FromLong(l_intobj);
		}
		PyDict_SetItemString(ar_dict, "returns_bitmask", ar_intobj);
		/* @@ returns_constant ? -- cannot currently handled */
		/* VMSDEF structures have no indicator */
		/* @@ new info in struct vmsdef_xr_conmsktbl since 3-JAN-1998*/

		if (ar_option == NULL)
		{
		    ar_option = Py_None;
		    Py_INCREF (ar_option);
		}
		PyDict_SetItemString(ar_dict, "bitmask", ar_option);
		Py_DECREF (ar_option);

		return ar_dict;
	    } /* strcmp(at_item_name,ar_itmtbl->at_itmnam_py) */
	    ar_itmtbl++;		/* next item */
	} /* while (ar_itmtbl->w_vmsvermin != 0) */

	/* -------------------- */
	/* item not found */
	PyErr_SetString(PyExc_ValueError, "argument 2: unknown item");
	return NULL;
} /* pyvms_item_get () */

/* ------------------------------------------------------------------------- */

/* val= pyvms.item_list ('$DVIDEF'); */
static PyObject *
pyvms_item_list (PyObject *self, PyObject *args)
{
	char			  * at_def_name;
	unsigned long		    l_def_len;
	struct vmsdef_xr_defdirtbl* ar_defdirtbl;
	struct vmsdef_xr_itmtbl	  * ar_itmtbl;
	PyObject		  * ar_list;
	PyObject		  * ar_item;
	unsigned long		    l_error;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#", &at_def_name, &l_def_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* look up define-name */
	ar_defdirtbl = &VMSDEF_GR_DIRECTORY[0];
	ar_itmtbl    = NULL;
	while (ar_defdirtbl->ar_itmtbl != NULL)
	{
	    if (strcmp(at_def_name,ar_defdirtbl->at_vmsdefnam) == 0)
	    {
		ar_itmtbl = ar_defdirtbl->ar_itmtbl;
		break;
	    }
	    ar_defdirtbl++;
	}

	/* -------------------- */
	if (ar_defdirtbl->ar_itmtbl == NULL)
	{
	    /* item table not found */
	    PyErr_SetString(PyExc_ValueError, "argument 1: unknown DEFine");
	    return NULL;
	}

	/* -------------------- */
	/* build a list of all item code names */
	ar_list = PyList_New(0);		/* create list object */
	if (ar_list == NULL)
	{
	    return NULL;			/* failed */
	}

	while (ar_itmtbl->w_vmsvermin != 0)
	{
	    /* note: this lookup is independed from version number and    */
	    /*       processor architecture! That's a feature, not a bug! */

	    ar_item = PyString_FromString(ar_itmtbl->at_itmnam_py);
	    if (ar_item == NULL)
	    {
		Py_DECREF(ar_list);		/* error, drop list */
		return NULL;
	    }
	    l_error = PyList_Append(ar_list, ar_item);
	    Py_DECREF(ar_item);
	    if (l_error != 0)
	    {
		Py_DECREF(ar_list);		/* error, drop list */
		return NULL;
	    }

	    ar_itmtbl++;		/* next item */
	}

	/* -------------------- */
	/* @@ optional: status = PyList_Sort(ar_list); */

	/* -------------------- */
	return ar_list;
} /* pyvms_item_list () */

/* ------------------------------------------------------------------------- */

/* "py-long = test_o2pyl (q3, q2, q1, q0)"*/

extern PyObject * vms__cvt_octa2pylong (long q_octaword[4]);

static PyObject *
pyvms_test__o2pyl (PyObject *self, PyObject *args)
{
	long	o_l3;
	long	o_l2;
	long	o_l1;
	long	o_l0;
	long	q_octaword[4];

	long	l_error;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "llll", &o_l3, &o_l2, &o_l1, &o_l0))
	{
	    return NULL;
	}

	/* -------------------- */
	q_octaword[0] = o_l0;
	q_octaword[1] = o_l1;
	q_octaword[2] = o_l2;
	q_octaword[3] = o_l3;

	/* -------------------- */
	return vms__cvt_octa2pylong (&q_octaword[0]);
} /* pyvms_test__o2pyl () */

/*
$python
import pyvms
def to (q3,q2,q1,q0):
  print pyvms.test__o2pyl(q3,q2,q1,q0), hex(pyvms.test__o2pyl(q3,q2,q1,q0))
#

#
to (0,0,0,0)
to (0,0,0,1)
to (0,0,0,0xf)
to (0,0,0,0x7f)
to (0,0,0,0xff)
to (0,0,0,0x17f)
to (0,0,0,0x80)
to (0,0,0,0x4080)
to (0,0,0,0x78787878)
to (0,0,0,0x80808080)
to (0,0,0,0x75555555)
to (0,0,0,0x76545555)
to (0,0,0,0xe5555555)
to (0,0,0x1,0x75555555)
to (0,0,0x2,0x75555555)
to (0,0,0x55555,0x75555555)

to (0x12345678,0x9abcdefe,0x12345678,0x9abcdefe)
to (0xffffffff,0xffffffff,0xffffffff,0xfffffffe)
to (0xffffffff,0xffffffff,0xffffffff,0x80000000)
to (0xffffffff,0xffffffff,0xffffffff,0x00000000)
to (0xffffffff,0xffffffff,0xfffffffe,0x00000000)
to (0xffffffff,0xffffffff,0x80000000,0x00000000)
to (0xffffffff,0xffffffff,0x80000000,0x00000001)
to (0xffffffff,0xffffffff,0x80000000,0x00000000)
to (0xffffffff,0xffffffff,0x00000000,0x00000000)
to (0xffffffff,0xfffffffe,0x00000000,0x00000000)
to (0xffffffff,0x80000000,0x00000000,0x00000000)
to (0xffffffff,0x00000000,0x00000000,0x00000000)
to (0xfffffffe,0x00000000,0x00000000,0x00000000)
to (0xf0000000,0x00000000,0x00000000,0x00000000)
to (0x80000000,0x00000000,0x00000000,0x00000000)
to (0x80000000,0x00000000,0x00000000,0x00000001)

*/

/* ------------------------------------------------------------------------- */

/* "(q3, q2, q1, q0) = test_pyl2o (py-long)" */

extern long vms__cvt_pylong2octa (PyObject * r_longint, long * o_octaword);

static PyObject *
pyvms_test__pyl2o (PyObject *self, PyObject *args)
{
	long		  o_octaword[4];
	PyObject	* ar_pylong;

	long		  l_error;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O", &ar_pylong))
	{
	    return NULL;
	}

	/* -------------------- */
	if (!PyLong_Check(ar_pylong))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 1: must be long integer");
	    return NULL;
	}

	/* -------------------- */
	o_octaword[0] = 0;
	o_octaword[1] = 0;
	o_octaword[2] = 0;
	o_octaword[3] = 0;

	l_error = vms__cvt_pylong2octa (ar_pylong, &o_octaword[0]);
	/* if l_error ... -
	    PyErr_SetString(conversion failed)
	    return NULL
	*/

	/* -------------------- */
	return Py_BuildValue ("(llll)",
	    o_octaword[3], o_octaword[2], o_octaword[1], o_octaword[0]);
} /* pyvms_test__pyl2o () */

/*
$python
import pyvms
def p(l1,l2,l3,l4):
  print (hex(l1),hex(l2),hex(l3),hex(l4))
# -p

a,b,c,d = pyvms.test__pyl2o (0L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (1L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x7fffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x80000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x87654321L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0xffffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x100000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x180000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x187654321L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x1ffffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x7fffffffffffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x8000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x8765432187654321L) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (0x18000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x18765432187654321L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x800000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x876543218765432187654321L) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (0x1800000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x1876543218765432187654321L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x7fffffff800000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x76543218876543218765432187654321L) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (0x80000000000000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0x80000000000000000000000000000001L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0xffffffffffffffffffffffffffffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (0xfffffffffffffffffffffffffffffffeL) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-1L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-2L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0xfL) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x7fffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x80000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x100000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x200000000L) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x7fffffff00000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x7fffffffffffffffL) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x8000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x8000000000000001L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x8000000000000002L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x80000000ffffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x8000000100000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x8fffffffffffffffL) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x10000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x10000000000000001L) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x7fffffffffffffffffffffffL) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x800000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0xffffffffffffffffffffffffL) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x1000000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x1000000000000000000000001L) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x76543218000000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x76543218876543218765432187654321L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x7fffffffffffffffffffffffffffffffL) ; p(a,b,c,d)

a,b,c,d = pyvms.test__pyl2o (-0x12345678123456781234567812345678L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x80000000000000000000000000000000L) ; p(a,b,c,d)
a,b,c,d = pyvms.test__pyl2o (-0x80000000000000000000000000000001L) ; p(a,b,c,d)

*/

/* ------------------------------------------------------------------------- */

/* "(q-hi, q-lo) = test_pyl2q (py-long)" */

extern long vms__cvt_pylong2quad (PyObject * r_longint, long * q_quadword);

static PyObject *
pyvms_test__pyl2q (PyObject *self, PyObject *args)
{
	long		  q_quadword[2];
	PyObject	* ar_pylong;

	long		  l_error;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O", &ar_pylong))
	{
	    return NULL;
	}

	/* -------------------- */
	if (!PyLong_Check(ar_pylong))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 1: must be long integer");
	    return NULL;
	}

	/* -------------------- */
	q_quadword[1] = 0;
	q_quadword[0] = 0;

	l_error = vms__cvt_pylong2quad (ar_pylong, &q_quadword[0]);
	/* if l_error ... -
	    PyErr_SetString(conversion failed)
	    return NULL
	*/

	/* -------------------- */
	return Py_BuildValue ("(ii)", q_quadword[1], q_quadword[0]);
} /* pyvms_test__pyl2q () */

/*
>>> import pyvms
def p(l1,l2):
  print ( (l1,l2) , (hex(l1),hex(l2)) )
# -p

a,b = pyvms.test__pyl2q (0L) ; p(a,b)
a,b = pyvms.test__pyl2q (1L) ; p(a,b)
a,b = pyvms.test__pyl2q (0x7fffffffL) ; p(a,b)
a,b = pyvms.test__pyl2q (0x80000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (0xffffffffL) ; p(a,b)
a,b = pyvms.test__pyl2q (0x100000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (0x180000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (0x1ffffffffL) ; p(a,b)
a,b = pyvms.test__pyl2q (0x7fffffffffffffffL) ; p(a,b)
a,b = pyvms.test__pyl2q (0x8000000000000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (-1L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0xfL) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x7fffffffL) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x80000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x100000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x200000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x7fffffff00000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x7fffffffffffffffL) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x8000000000000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x8000000000000001L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x8000000000000002L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x80000000ffffffffL) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x8000000100000000L) ; p(a,b)
a,b = pyvms.test__pyl2q (-0x8fffffffffffffffL) ; p(a,b)
>>>
*/

/* ------------------------------------------------------------------------- */

/* "py-long = test_q2pyl (q-hi, q-lo)"*/

extern PyObject * vms__cvt_quad2pylong (long q_quadword[2]);

static PyObject *
pyvms_test__q2pyl (PyObject *self, PyObject *args)
{
	long	q_hi;
	long	q_lo;
	long	q_quadword[2];

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "ll", &q_hi, &q_lo))
	{
	    return NULL;
	}

	/* -------------------- */
	q_quadword[1] = q_hi;
	q_quadword[0] = q_lo;

	/* -------------------- */
	return vms__cvt_quad2pylong (q_quadword);
} /* pyvms_test__q2pyl () */
/*
$python
import pyvms
def tq (h,l):
  print pyvms.test__q2pyl(h,l), hex(pyvms.test__q2pyl(h,l))
#

#
tq (0,0)
tq (0,1)
tq (0,0xf)
tq (0,0x7f)
tq (0,0xff)
tq (0,0x17f)
tq (0,0x80)
tq (0,0x4080)
tq (0,0x78787878)
tq (0,0x80808080)
tq (0,0x75555555)
tq (0,0x76545555)
tq (0,0xe5555555)
tq (0x1,0x75555555)
tq (0x2,0x75555555)
tq (0x55555,0x75555555)

*/

/* ------------------------------------------------------------------------- */

/* "py-long = test_uq2pyl (q-hi, q-lo)"*/

extern PyObject * vms__cvt_uquad2pylong (unsigned long q_quadword[2]);

static PyObject *
pyvms_test__uq2pyl (PyObject *self, PyObject *args)
{
	unsigned long	q_hi;
	unsigned long	q_lo;
	unsigned long	q_quadword[2];

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "ll", &q_hi, &q_lo))
	{
	    return NULL;
	}

	/* -------------------- */
	q_quadword[1] = q_hi;
	q_quadword[0] = q_lo;

	/* -------------------- */
	return vms__cvt_uquad2pylong (q_quadword);
} /* pyvms_test__uq2pyl () */

/*
>>> import pyvms
hex( pyvms.test__uq2pyl (0,0)           )
hex( pyvms.test__uq2pyl (0,1)           )
hex( pyvms.test__uq2pyl (0,0x7fffffff)  )
hex( pyvms.test__uq2pyl (0,0x80000000)  )
hex( pyvms.test__uq2pyl (0,0xffffffff)  )
hex( pyvms.test__uq2pyl (0,0x100000000) ) # err
hex( pyvms.test__uq2pyl (1,0)           )
hex( pyvms.test__uq2pyl (0x7fffffff,0)  )
hex( pyvms.test__uq2pyl (0x80000000,0)  )
hex( pyvms.test__uq2pyl (0xffffffff,0)  )
hex( pyvms.test__uq2pyl (0xffffffff,0x80000000) )
hex( pyvms.test__uq2pyl (0xffffffff,0xffffffff) )
hex( pyvms.test__uq2pyl (0x100000000,0)  )
*/

/* ------------------------------------------------------------------------- */

static char pyvms_uaf_get_usernames__doc[] =
"username_list = pyvms.uaf_get_usernames()\n\
Return list of usernames from SYSUAF.DAT.";

extern PyObject *
pyvms_uaf_get_usernames (PyObject *self, PyObject *args);
/* code in PYVMS__UAF_ROUTINES.C */

/* ------------------------------------------------------------------------- */

/* code in PYVMS__VMSOBJ_ROUTINES.C */
extern PyObject * pyvms_vmsobj_fab    (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_iosb   (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_lksb   (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_nam    (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_rab    (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xaball (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabdat (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabfhc (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabitm (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabkey (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabpro (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabrdt (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabsum (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj_xabtrm (PyObject *self, PyObject *args);

extern PyObject * pyvms_vmsobj__membuf     (PyObject *self, PyObject *args);
extern PyObject * pyvms_vmsobj__ownshp_nam (PyObject *self, PyObject *args);
/*     ^- that's a lie, of course, but we don't really need the correct type */
/* code in PYVMS__VMSOBJ_ROUTINES.C */

/* ------------------------------------------------------------------------- */

static struct PyMethodDef pyvms_methods[] = {
	{"crtl_from_vms",     (PyCFunction)pyvms_crtl_from_vms,	METH_VARARGS,
						pyvms_crtl_from_vms__doc__},
	{"crtl_open",	      (PyCFunction)pyvms_crtl_open,	METH_VARARGS,
						pyvms_crtl_open__doc__},
	{"crtl_to_vms",	      (PyCFunction)pyvms_crtl_to_vms,	METH_VARARGS,
						pyvms_crtl_to_vms__doc__},
	{"history_delete",    (PyCFunction)pyvms_history_delete, METH_VARARGS},
	{"history_get",       (PyCFunction)pyvms_history_get,    METH_VARARGS},
	{"history_show",      (PyCFunction)pyvms_history_show,   METH_VARARGS},
	{"history_size",      (PyCFunction)pyvms_history_size,   METH_VARARGS},
	{"item_get",	      (PyCFunction)pyvms_item_get,	 METH_VARARGS},
	{"item_list",	      (PyCFunction)pyvms_item_list,      METH_VARARGS},
/*@@*/	{"test__o2pyl",	      (PyCFunction)pyvms_test__o2pyl,    METH_VARARGS},
/*@@*/	{"test__pyl2o",	      (PyCFunction)pyvms_test__pyl2o,    METH_VARARGS},
/*@@*/	{"test__pyl2q",	      (PyCFunction)pyvms_test__pyl2q,    METH_VARARGS},
/*@@*/	{"test__q2pyl",	      (PyCFunction)pyvms_test__q2pyl,    METH_VARARGS},
/*@@*/	{"test__uq2pyl",      (PyCFunction)pyvms_test__uq2pyl,   METH_VARARGS},
	{"uaf_get_usernames", (PyCFunction)pyvms_uaf_get_usernames, METH_VARARGS,
						pyvms_uaf_get_usernames__doc},
	{"vmsobj__membuf",    (PyCFunction)pyvms_vmsobj__membuf, METH_VARARGS},
	{"vmsobj__ownership_names", (PyCFunction)pyvms_vmsobj__ownshp_nam, METH_VARARGS},
	{"vmsobj_fab",	      (PyCFunction)pyvms_vmsobj_fab,	 METH_VARARGS},
	{"vmsobj_iosb",	      (PyCFunction)pyvms_vmsobj_iosb,    METH_VARARGS},
	{"vmsobj_lksb",	      (PyCFunction)pyvms_vmsobj_lksb,    METH_VARARGS},
	{"vmsobj_nam",	      (PyCFunction)pyvms_vmsobj_nam,	 METH_VARARGS},
	{"vmsobj_rab",	      (PyCFunction)pyvms_vmsobj_rab,	 METH_VARARGS},
	{"vmsobj_xaball",     (PyCFunction)pyvms_vmsobj_xaball,	 METH_VARARGS},
	{"vmsobj_xabdat",     (PyCFunction)pyvms_vmsobj_xabdat,	 METH_VARARGS},
	{"vmsobj_xabfhc",     (PyCFunction)pyvms_vmsobj_xabfhc,	 METH_VARARGS},
	{"vmsobj_xabitm",     (PyCFunction)pyvms_vmsobj_xabitm,	 METH_VARARGS},
	{"vmsobj_xabkey",     (PyCFunction)pyvms_vmsobj_xabkey,	 METH_VARARGS},
	{"vmsobj_xabpro",     (PyCFunction)pyvms_vmsobj_xabpro,	 METH_VARARGS},
	{"vmsobj_xabrdt",     (PyCFunction)pyvms_vmsobj_xabrdt,	 METH_VARARGS},
	{"vmsobj_xabsum",     (PyCFunction)pyvms_vmsobj_xabsum,	 METH_VARARGS},
	{"vmsobj_xabtrm",     (PyCFunction)pyvms_vmsobj_xabtrm,	 METH_VARARGS},
	{NULL,		NULL}		 /* Sentinel */
};

/* ---------------------------------------- */

void
initpyvms()
{
	PyObject *m, *d, *l, *s, *v;
	vmsdef_xr_defdirtbl* ar_defdirtbl;
	long		     l_error;

	m = Py_InitModule4("pyvms",
			   pyvms_methods,
			   pyvms__doc__,
			   (PyObject *)NULL,
			   PYTHON_API_VERSION);
        d = PyModule_GetDict(m);

	/* -------------------- */
        if (PyErr_Occurred())
	{
	    (void) Py_FatalError(pyvms_t_cant_init);
	}

	/* -------------------- */
	/* Initialize exception */
	pyvms_gr_error = PyErr_NewException("pyvms.error", NULL, NULL);
	if (pyvms_gr_error == NULL)
	{
	    (void) Py_FatalError(pyvms_t_cant_deferr);
	}
	if (PyDict_SetItemString(d, "error", pyvms_gr_error) != 0)
	{
	    (void) Py_FatalError(pyvms_t_cant_deferr);
	}

	/* -------------------- */
	/* set VMS version number */
	l = PyInt_FromLong((long)vmsdef_gw_vmsver);
	if (l == NULL)
	{
	    (void) Py_FatalError(pyvms_t_cant_defvms);
	}
	if (PyDict_SetItemString(d, "vms_version_number", l) != 0)
	{
	    (void) Py_FatalError(pyvms_t_cant_defvms);
	}

	/* -------------------- */
	/* Initialize vmsdef.definitions list */
	ar_defdirtbl = &VMSDEF_GR_DIRECTORY[0];
	l = PyList_New(0);
	if (l == NULL)
	{
	    return;
	}

	while (ar_defdirtbl->ar_itmtbl != NULL)
	{
	    s = PyString_FromString(ar_defdirtbl->at_vmsdefnam);
	    if (s == NULL)
	    {
		Py_DECREF(l);		/* drop list */
		l = NULL;
		break;
	    }
	    l_error = PyList_Append(l, s);
	    Py_DECREF(s);		/* drop string item */
	    if (l_error != 0)
	    {
		Py_DECREF(l);		/* drop list */
		l = NULL;
		break;
	    }
	    ar_defdirtbl++;
	}
	if (l == NULL)
	{
	    Py_FatalError("can't setup definitions[]");
	}

	s = PyString_FromString("definitions");
	if (s == NULL || PyDict_SetItem (d, s, l) )
	{
	    Py_FatalError("can't setup definitions[]");
	}

	/* -------------------- */
	if (PyErr_Occurred())
	{
	    Py_FatalError("can't initialize module pyvms");
	}

	/* -------------------- */

} /* initpyvms() */

/* ------------------------------------------------------------------------ */

/* EOF: PYVMSMODULE.C */
