/* VMS__CVT_PY2BIN.C - 09-JAN-1999 Uwe Zessin
   Generic routine to convert a Python type to binary data.
   Can be: Byte, Word, Longword, Quadword, Octaword and tuples of BWLQO.
*/

#include "vmsdef.h"

/* ------------------------------------------------------------------------- */
/* size of item types - in PYVMS_GLOBAL.C */
extern unsigned long vmsdef_gl_itmtypsiz[];
/* ------------------------------------------------------------------------- */
/* function prototypes for external routines */

/* ------------------------------ */
/* input a Python long integer - output a binary quadword */
extern long vms__cvt_pylong2quad (PyObject * r_longint, long * q_quadword);

/* ------------------------------ */
/* input a Python long integer - output a binary octaword */
extern long vms__cvt_pylong2octa (PyObject * r_longint, long * o_octaword);

/* ------------------------------------------------------------------------- */

long
vms__cvt_py2bin
	(PyObject		 * ar_py_data		/* Python data	*/
	,char			 * ab_data_addr		/* data address */
	,unsigned short int	   w_data_type		/* data type	*/
	,unsigned short int	   w_bufsiz		/* data length	*/
	)
{
	unsigned short int	w_styp;		/* data type (w/o sign bit) */

	unsigned long	  l_datatypsiz;	/* size of single datatype item	   */
	unsigned long	  l_array_size;	/* number of items		   */
	unsigned long	  l_tuple_size;	/* number of items		   */
	unsigned long  	  l_data;	/* B/W/L storage		   */
	unsigned long	  loop;	/* counter to loop over elements of 1 item */
	char	* ab_data;	/* pointer to data elements in 1 item	   */

	/* -------------------- */
	/* data type - mask off the (un)signed bit */
	w_styp = w_data_type & (~ITMTYP_M_USIGN);
	if (w_styp > ITMTYP_K_MAXTYP)
	{
	    /* out of range - prevent ACCVIO at: vmsdef_gl_itmtypsiz[w_styp] */
	    PyErr_SetString(PyExc_SystemError,
		"vms__cvt_py2bin(): unsuported datatype id number");
	    return -1;
	}

	/* -------------------- */
	l_datatypsiz = vmsdef_gl_itmtypsiz[w_styp];	/* size of 1 element */
	/* could have been given ITMTYP_K_ASCII or _ASCIIC */
	if (l_datatypsiz == 0)
	{
	    PyErr_SetString(PyExc_SystemError,
		"vms__cvt_py2bin(): /0 at 'w_bufsiz / l_datatypsiz'");
	    return -1;
	}
	l_array_size = w_bufsiz / l_datatypsiz;		/* # elements */

	/* -------------------- */
	/* if item has multiple data-elements - a tuple must be passed */
	if (l_array_size > 1)
	{
	    /* Python object must be a tuple */
	    if (!PyTuple_Check(ar_py_data))
	    {
		PyErr_SetString(PyExc_SystemError,
		   "vms__cvt_py2bin(): Python object must be a tuple");
		return -1;
	    }
	    /* tuple size and data array size must match */
	    l_tuple_size = PyTuple_Size(ar_py_data);
	    if (l_tuple_size != l_array_size)
	    {
		PyErr_SetString(PyExc_SystemError,
	       "vms__cvt_py2bin(): Python tuple/data-array size do not match");
		return -1;
	    }
	}
	else
	{
	    if (l_array_size < 1)
	    {
		PyErr_SetString(PyExc_SystemError,
		    "vms__cvt_py2bin(): datatype bigger than buffer length");
		return -1;
	    }
	    /* else - exactly 1 item to return - use code in loop, too */
	}

	/* -------------------- */
	ab_data = ab_data_addr;		/* start address */

	/* -------------------- */
	/* loop over all elements in the data-tuple */
	for (loop = 0; loop < l_array_size; loop++)
	{
	    PyObject	* ar_data;
	    long	  l_data;

	    l_data = 0;
	    if (l_array_size > 1)
	    {
		ar_data = PyTuple_GetItem(ar_py_data, loop);
		if (ar_data == NULL)
		{
		    return -1;
		}
	    }
	    else
	    {
		/* data is not an array of BWLQO - use directly */
		ar_data = ar_py_data;
	    }

	  /* BWL are represented by a 'Python integer' (32 bit) */
	  if (
	      (w_styp == ITMTYP_K_BYTE) ||
	      (w_styp == ITMTYP_K_WORD) ||
	      (w_styp == ITMTYP_K_LONG)
	     )
	  {
	    if (!PyInt_Check(ar_data)) /* item-data or item-of-tuple */
	    {
	      PyErr_SetString(PyExc_TypeError,
		"vms__cvt_py2bin(): data of item must be integer");
	      return -1;
	    }

	    /* translate Python integer into C long */
	    l_data = PyInt_AsLong(ar_data);
	    if (PyErr_Occurred())
	    {
	      return -1;
	    }

	    /* store data and advance pointer *correctly* */
	    if (w_styp == ITMTYP_K_BYTE)
	    {
	      *(char *)ab_data = l_data;
	      ab_data++;
	    }
	    else
	    {
	      if (w_styp == ITMTYP_K_WORD)
	      {
		*(short int *)ab_data = l_data;
		ab_data += 2;
	      }
	      else
	      {
	        if (w_styp == ITMTYP_K_LONG)
	        {
		  *(long *)ab_data = l_data;
		  ab_data += 4;
		} /* w_styp == ITMTYP_K_LONG */
	      } /* else- w_styp == ITMTYP_K_WORD */
	    } /* else- w_styp == ITMTYP_K_BYTE */
	  } /* w_styp == (ITMTYP_K_BYTE || WORD || LONG */
	  else
	  {
	    if (w_styp == ITMTYP_K_QUAD)
	    {
	      /* Q data is represented by a Python long integer object */
	      long   l_status;

	      if (!PyLong_Check(ar_data))
	      {
		PyErr_SetString(PyExc_TypeError,
		  "vms__cvt_py2bin(): data of Q-item must be long integer");
		return -1;
	      }

	      /* store data and advance pointer *correctly* */
	      l_status = vms__cvt_pylong2quad (ar_data, (long*)ab_data);
	      if (l_status != 1)
	      {
		PyErr_SetString(PyExc_SystemError,
		  "vms__cvt_py2bin(): vms__cvt_pylong2quad() failed");
		return -1;
	      }
	      ab_data += 8;
	    } /* w_styp == ITMTYP_K_QUAD) */
	    else
	    {
	      if (w_styp == ITMTYP_K_OCTA)
	      {
		/* O data is represented by a Python long integer object */
		long   l_status;

		if (!PyLong_Check(ar_data))
		{
		  PyErr_SetString(PyExc_TypeError,
		    "vms__cvt_py2bin(): data of O-item must be long integer");
		  return -1;
		}

		/* store data and advance pointer *correctly* */
		l_status = vms__cvt_pylong2octa (ar_data, (long*)ab_data);
		if (l_status != 1)
		{
		  PyErr_SetString(PyExc_SystemError,
		    "vms__cvt_py2bin(): vms__cvt_pylong2octa() failed");
		  return -1;
		}
		ab_data += 16;
	      }
	      else
	      {
		PyErr_SetString(PyExc_SystemError,
		  "vms__cvt_py2bin(): unimplemented data type");
		return -1;
	      } /* else- w_styp == ITMTYP_K_OCTA */
	    } /* else- w_styp == ITMTYP_K_QUAD */
	  } /* else- w_styp == (ITMTYP_K_BYTE || WORD || LONG */

	  if (l_array_size == 1)
	  {
 	    break;		/* only one element - done with for() loop */
	  }
	} /* for() loop over all tuple elements */

	/* -------------------- */
	return 1;
} /* vms__cvt_py2bin () */

/* ------------------------------------------------------------------------- */

/* EOF: VMS__CVT_PY2BIN.C */
