/* VMS__CVT_PYLONG.C - 29-JUL-1998 Uwe Zessin
   Convert OpenVMS datatype quadword to/from Python 'long int'

  27-DEC-1998 ZE. -- add vms__cvt_uquad2pylong() to translate unsigned Quadword
			to Python long integer (bitmasks, etc.)
		  -- fix sign handling in vms__cvt_pylong2quad()
  05-JAN-1999 ZE. -- add vms__cvt_octa2pylong() to translate signed Octaword
			to Python long integer
  09-JAN-1999 ZE. -- add vms__cvt_pylong2octa() to translate Python long integer
			to signed Octaword
*/

#include "Python.h"
#include "longintrepr.h"	/* need MASK, etc. */
#include "mymath.h"

#include <lib$routines.h>       /* LIB$name */

/* @@ the following macro copied from LONGOBJECT.C */
#define ABS(x) ((x) < 0 ? -(x) : (x))

/* ------------------------------------------------------------------------- */

/* @@ the following routine was copied from LONGOBJECT.C */
/* it was renamed from 'long_normalize()' to 'vms__cvt_long_normalize()' */

/* Normalize (remove leading zeros from) a long int object.
   Doesn't attempt to free the storage--in most cases, due to the nature
   of the algorithms used, this could save at most be one word anyway. */

static PyLongObject *
vms__cvt_long_normalize(v)
	register PyLongObject *v;
{
	int j = ABS(v->ob_size);
	register int i = j;
	
	while (i > 0 && v->ob_digit[i-1] == 0)
		--i;
	if (i != j)
		v->ob_size = (v->ob_size < 0) ? -(i) : i;
	return v;
} /* vms__cvt_long_normalize */
/* @@ end of copy from LONGOBJECT.C */

/* ------------------------------------------------------------------------- */

/* create a new long integer object from an OpenVMS signed quadword */
PyObject *
vms__cvt_quad2pylong (long q_quadword[2])
{
    /* a quadword fits in at most 5 'digits' */
    /* 64 bits / 15 bits(=1 digit) = 4.266  -- index 0..4 */
    PyLongObject *v = _PyLong_New(5); /* in LONGOBJECT.C */

    if (v != NULL)
    {
	unsigned long l0;
	unsigned long l1;

	int i;

	for (i = 0; i < 5; i++)
	{
	    v->ob_digit[i] = 0;
	}

	/* handle sign bit */
	if (q_quadword[1] < 0)
	{
	    v->ob_size = -(v->ob_size);
	    /* there is no '-0', turn 80000000,00000000	*/
	    /* into '-80000000,00000000'		*/
	    if ((q_quadword[1] == 0x80000000) &&
		(q_quadword[0] == 0x00000000)   )
	    {
		/* no change */
		l0 = q_quadword[0];
		l1 = q_quadword[1];
	    }
	    else
	    {
		long l_add[2];
		long l_aug[2];
		long l_array_length;

		l_add[0] = ~q_quadword[0];
		l_add[1] = ~q_quadword[1];
		l_aug[0] = 1; /* two's complement */
		l_aug[1] = 0;

		l_array_length = 2;	/* add 2 longwords = 1 quadword */

		/* ignore condition value SS$_INTOVF */
		(void) lib$addx (&l_add[0], &l_aug[0], &l_add[0],
		    &l_array_length);
		/* ignore integer overflow status */

		l0 = l_add[0];
		l1 = l_add[1];
		l1 &= 0x7fffffff;	/* mask off sign bit */
	    }
  	}
	else
	{
	    l0 = q_quadword[0];
	    l1 = q_quadword[1];
	}
/*
|<--------------- l1 -------------->|<--------------- l0 -------------->|
 6  6 5        5 4   44   4 3      3|33 2        2 1   11   1 0        0
 3210 9876543210 9876543210 98765432|10 9876543210 9876543210 9876543210
 4444 3333333333 3333322222 22222222|22 1111111111 1111100000 0000000000
*/
	v->ob_digit[0] = (digit) (l0 & MASK); /* bits 00:14 from l0 */
	l0 >>= SHIFT;		/* bits 15:31 remain in l0 */

	v->ob_digit[1] = (digit) (l0 & MASK); /* bits 15:29 from l0 */
	l0 >>= SHIFT;		/* bits 31:30 remain in l0 */

	i = l1 & 0x00001fff;	/* get bits 32:44 (=13) from l1 */
	i <<= 2;			/* keep room for lower 2 bits of l0 */
	l0 += i;			/* merge, bits 30:44 */
        v->ob_digit[2] = (digit) (l0 & MASK); /* bits 30:44 (l0,l1) */
	l1 >>= 13;		/* bits 45:63 remain in l1 */

	v->ob_digit[3] = (digit) (l1 & MASK); /* bits 45:59 (l1) */
	l1 >>= SHIFT;		/* bits 60:63 remain in l1 */

	l1 &= 0x03;	/* ignore other bits */
	v->ob_digit[4] = (digit) (l1 & MASK); /* bits 60:-63 (l1) */
	/* l1 >>= SHIFT; */

	v = vms__cvt_long_normalize(v); /* copied from LONGOBJECT.C */
    }
    return (PyObject *)v;
} /* vms__cvt_quad2pylong () */

/* ------------------------------------------------------------------------- */

/* create a new long integer object from an OpenVMS unsigned quadword */
PyObject *
vms__cvt_uquad2pylong (unsigned long q_quadword[2])
{
    /* a quadword fits in at most 5 'digits' */
    /* 64 bits / 15 bits(=1 digit) = 4.266 -- index 0..4 */
    PyLongObject *v = _PyLong_New(5); /* in LONGOBJECT.C */

    if (v != NULL)
    {
	unsigned long l0 = q_quadword[0];
	unsigned long l1 = q_quadword[1];

	int i;

	for (i = 0; i < 5; i++)
	{
	    v->ob_digit[i] = 0;
	}

#if 0
	/* handle sign bit */
    /* ... */
	{
	    l0 = q_quadword[0];
	    l1 = q_quadword[1];
  	}
#endif
/*
|<--------------- l1 -------------->|<--------------- l0 -------------->|
 6  6 5        5 4   44   4 3      3|33 2        2 1   11   1 0        0
 3210 9876543210 9876543210 98765432|10 9876543210 9876543210 9876543210
 4444 3333333333 3333322222 22222222|22 1111111111 1111100000 0000000000
*/
	v->ob_digit[0] = (digit) (l0 & MASK); /* bits 00:14 from l0 */
	l0 >>= SHIFT;		/* bits 15:31 remain in l0 */

	v->ob_digit[1] = (digit) (l0 & MASK); /* bits 15:29 from l0 */
	l0 >>= SHIFT;		/* bits 31:30 remain in l0 */

	i = l1 & 0x00001fff;	/* get bits 32:44 (=13) from l1 */
	i <<= 2;			/* keep room for lower 2 bits of l0 */
	l0 += i;			/* merge, bits 30:44 */
        v->ob_digit[2] = (digit) (l0 & MASK); /* bits 30:44 (l0,l1) */
	l1 >>= 13;		/* bits 45:63 remain in l1 */

	v->ob_digit[3] = (digit) (l1 & MASK); /* bits 45:59 (l1) */
	l1 >>= SHIFT;		/* bits 60:63 remain in l1 */

	l1 &= 0x03;	/* ignore other bits */
	v->ob_digit[4] = (digit) (l1 & MASK); /* bits 60:63 (l1) */
	/* l1 >>= SHIFT; */

	v = vms__cvt_long_normalize(v); /* copied from LONGOBJECT.C */
    }
    return (PyObject *)v;
} /* vms__cvt_uquad2pylong () */

/* ------------------------------------------------------------------------- */

/* convert a python long integer object to an OpenVMS signed quadword */
/* a big positive long int can result in a negative quadword ! */
/* returns 1, if OK */

long
vms__cvt_pylong2quad (PyObject * r_longint, unsigned long * q_quadword)
{
	PyObject *vv = r_longint;
	register PyLongObject *v;
	int l_obsize , sign;
	unsigned long t;
	unsigned long sign_bit;
	
	if (vv == NULL || !PyLong_Check(vv))
	{
	    PyErr_BadInternalCall();
	    return -1;
	}

	v = (PyLongObject *)vv;
	l_obsize = v->ob_size;

	*q_quadword      = 0;	/* pre-initialize quadword */
	*(q_quadword +1) = 0;

	if (l_obsize != 0)
	{
	    /* Python 'long integer' is not 0 */

	    sign_bit = 0;
	    sign = 1;

	    /* handle sign bit */
	    if (l_obsize < 0)
	    {
		sign = -1;
		l_obsize = -(l_obsize);
	    }

/* mask
6  6   5      5 4        4        3  3     2    2   1      1 0
32109876|54321098|76543210|98765432|10987654|32109876|54321098|76543210

1234                 12345|67890123|45                 1234567|89012345
    1234|56789012|345                 123456|78901234|5
*/
	    /* bits 14:0 */
	    *q_quadword = v->ob_digit[0];

	    /* bits 29:15 */
	    if (l_obsize > 1) /* 2... */
	    {
		t = v->ob_digit[1];
		t <<= SHIFT;
		*q_quadword |= t;
	    }

	    /* bits 31:30 in first longword of quadword */
	    /*  put remaining bits into 2nd longword at 12:0 of quadword */
	    if (l_obsize > 2) /* 3... */
	    {
		t = (v->ob_digit[2] & 3); /* mask lower 2 bits */
		t <<= 30;
		*q_quadword |= t;
		/* ... */
		q_quadword++;	/* next longword */
		t = (v->ob_digit[2] & 0xfffc); /* remove lower 2 bits */
		t >>= 2;		/* shift these bits */
		*q_quadword = t;
	    }
	    else
	    {
		/* later code assumes that pointer is at 2nd longword */
		q_quadword++;	/* next longword */
	    }

	    /* bits 27:12 in second longword of quadword */
	    if (l_obsize > 3) /* 4... */
	    {
		t = v->ob_digit[3];
		t <<= (SHIFT-2); /* 2 bits of prev digit in 1st longword */
		*q_quadword |= t;
	    }

	    /* bits 31:28 in second longword of quadword */
	    /* these are 4@@3 bits! - check sign/for overflow */
	    if (l_obsize > 4) /* 5... */
	    {
		/* @@sign_bit = v->ob_digit[4] & 8; */
		t = v->ob_digit[4] & 15; /* @@& 7; */
		t <<= (SHIFT+SHIFT-2);
		*q_quadword |= t;

		if ((v->ob_digit[4] & 0x7ff0) != 0)
		{
		    /* @@ overflow */
		    /*	PyErr_SetString(PyExc_OverflowError,
		    ||		"long integer too long to convert quadword");
		    */
		    /* @@ set overflow marker */
		    /* don't abort here */
		}
	    }

          if ((*q_quadword == 0) && (*(q_quadword-1) == 0) && (sign_bit != 0))
	  {
		/* maximum negative value - keep it */
		*q_quadword = 0x80000000; /* patch in 'sign_bit' */
	  }
	  else
	  {
	    if ((sign < 0) || (sign_bit != 0))
	    {
		long q_add1[2];
		long l_arraylength = 2;     /* 1Q = 2L -> lib$addx() */
                long l_status;

		q_add1[0] = 1; /* two's complement */
		q_add1[1] = 0;

		*q_quadword = ~(*q_quadword);	/* upper longword */
		q_quadword--;	/* pointer to 1st (=low) longword */
		*q_quadword = ~(*q_quadword);	/* lower longword */

		/* ignore condition value SS$_INTOVF */
		(void) lib$addx
			(&q_add1[0]	/* addend-array              */
			,q_quadword	/* augend-array              */
			,q_quadword	/* resultant-array           */
			,&l_arraylength /* [,array-length] longwords */
			);
	    }
	    if (l_obsize > 5) /* 6... */
	    {
		/* @@ check remaining bits for overflow */
		    /* @@ overflow */
		    /*	PyErr_SetString(PyExc_OverflowError,
		    ||		"long integer too long to convert quadword");
		    */
		    return 0;
	    }
	  }
	}
	return 1;
} /* vms__cvt_pylong2quad () */

/* ------------------------------------------------------------------------- */

/* create a new long integer object from an OpenVMS signed octaword */
PyObject *
vms__cvt_octa2pylong (long o_octaword[4])
{
    PyLongObject *v;

    /* check if the upper 2 longwords are 0 - */
    /*   then use quadword conversion routine routine	     */
    /* this results in a shorter allocation of v->ob_digit[] */
    if ((o_octaword[3] == 0) && (o_octaword[4] == 0))
    {
	return vms__cvt_quad2pylong (&o_octaword[0]);
    }

    /* an octaword fits in at most 9 'digits' */
    /* 128 bits / 15 bits(=1 digit) = 8.533 -- index 0..8 */
    v = _PyLong_New(9); /* in LONGOBJECT.C */

    if (v != NULL)
    {
	unsigned long l0;
	unsigned long l1;
	unsigned long l2;
	unsigned long l3;

	int i;

	for (i = 0; i < 9; i++)
	{
	    v->ob_digit[i] = 0;
	}

	/* handle sign bit */
	if (o_octaword[3] < 0)
	{
	    v->ob_size = -(v->ob_size);
	    /* there is no '-0', turn 80000000,00000000,0,0	*/
	    /* into '-80000000,00000000,0,0'		*/
	    if ((o_octaword[3] == 0x80000000) &&
		(o_octaword[2] == 0x00000000) &&
		(o_octaword[1] == 0x00000000) &&
		(o_octaword[0] == 0x00000000)   )
	    {
		/* no change */
		l0 = o_octaword[0];
		l1 = o_octaword[1];
		l2 = o_octaword[2];
		l3 = o_octaword[3];
	    }
	    else
	    {
		long l_add[4];
		long l_aug[4];
		long l_array_length;

		l_add[0] = ~o_octaword[0];
		l_add[1] = ~o_octaword[1];
		l_add[2] = ~o_octaword[2];
		l_add[3] = ~o_octaword[3];
		l_aug[0] = 1; /* two's complement */
		l_aug[1] = 0;
		l_aug[2] = 0;
		l_aug[3] = 0;

		l_array_length = 4;	/* add 4 longwords = 1 octaword */

		/* ignore condition value SS$_INTOVF */
		(void) lib$addx (&l_add[0], &l_aug[0], &l_add[0],
		    &l_array_length);
		/* ignore integer overflow status */

		l0 = l_add[0];
		l1 = l_add[1];
		l2 = l_add[2];
		l3 = l_add[3];
		l3 &= 0x7fffffff;	/* mask off sign bit */
	    }
  	}
	else
	{
	    l0 = o_octaword[0];
	    l1 = o_octaword[1];
	    l2 = o_octaword[2];
	    l3 = o_octaword[3];
	}
/*
       |<--------------- L1 -------------->|<--------------- L0 -------------->|
 6    6|6  6 5        5 4   44   4 3      3|33 2        2 1   11   1 0        0
 987654|3210 9876543210 9876543210 98765432|10 9876543210 9876543210 9876543210
 444444|4444 3333333333 3333322222 22222222|22 1111111111 1111100000 0000000000

  |<--------------- L3 -------------->|<--------------- L2 -------------->|
 1|1      1 1    1   1 1    1   1     |
 2|2      2 1    1   1 0    0   0 9  9|9    9 8        8 7   77   76    6|6  6
 8|76543210 9876543210 9876543210 9876|543210 9876543210 9876543210987654|3210
 8|88888888 7777777777 7777766666 6666|666666 5555555555 5555544444444444|4444
*/
	v->ob_digit[0] = (digit) (l0 & MASK); /* bits 00:14 from l0 */
	l0 >>= SHIFT;		/* bits 15:31 remain in l0 */

	v->ob_digit[1] = (digit) (l0 & MASK); /* bits 15:29 from l0 */
	l0 >>= SHIFT;		/* bits 30:31 remain in l0 */

	i = l1 & 0x00001fff;	/* get bits 32:44 (=13) from l1 */
	i <<= 2;			/* keep room for lower 2 bits of l0 */
	l0 += i;			/* merge, bits 30:44 */
        v->ob_digit[2] = (digit) (l0 & MASK); /* bits 30:44 (l0,l1) */
	l1 >>= 13;		/* bits 45:63 remain in l1 */

	v->ob_digit[3] = (digit) (l1 & MASK); /* bits 45:59 (l1) */
	l1 >>= SHIFT;		/* bits 60:63 remain in l1 */

	i = l2 & 0x000007ff;	/* get bits 64:74 (=11) from l2 */
	i <<= 4;			/* keep room for lower 4 bits of l1 */
	l1 += i;			/* merge, bits 60:74 */
	v->ob_digit[4] = (digit) (l1 & MASK); /* bits 60:74 (l1,l2) */
	l2 >>= 11;		/* bits 75:95 remain in l2 */

	v->ob_digit[5] = (digit) (l2 & MASK); /* bits 75:89 (l2) */
	l2 >>= SHIFT;		/* bits 90:95 remain in l2 */

	i = l3 & 0x0000001ff;	/* get bits 96:104 (=9) bits from l3 */
	i <<= 6;			/* keep room for lower 6bits of l2 */
        l2 += i;			/* merge, bits 90:104 */
	v->ob_digit[6] = (digit) (l2 & MASK); /* bits 90:104 (l2,l3) */
	l3 >>= 9;		/* bits 105:127 */

	v->ob_digit[7] = (digit) (l3 & MASK); /* bits 105:119 (l3) */
	l3 >>= SHIFT;		/* bits 120:127 remain in l3 */

	l3 &= 0xff;	/* ignore other bits */
	v->ob_digit[8] = (digit) (l3 & MASK); /* bits 120:127 (l3) */

	v = vms__cvt_long_normalize(v); /* copied from LONGOBJECT.C */
    }
    return (PyObject *)v;
} /* vms__cvt_octa2pylong () */

/* ------------------------------------------------------------------------- */

/* convert a python long integer object to an OpenVMS signed octaword */
/* a big positive long int can result in a negative octaword ! */
/* returns 1, if OK */

long
vms__cvt_pylong2octa (PyObject * r_longint, unsigned long * o_octaword)
{
	PyObject *vv = r_longint;
	register PyLongObject *v;
	int l_obsize , sign;
	unsigned long t;
	unsigned long sign_bit;
	
	if (vv == NULL || !PyLong_Check(vv))
	{
	    PyErr_BadInternalCall();
	    return -1;
	}

	v = (PyLongObject *)vv;
	l_obsize = v->ob_size;

	*o_octaword      = 0;	/* pre-initialize quadword */
	*(o_octaword +1) = 0;
	*(o_octaword +2) = 0;
	*(o_octaword +3) = 0;

	if (l_obsize != 0)	/* Python 'long integer' is not 0 */
	{
	    long l_tmp;

	    sign_bit = 0;
	    sign = 1;

	    /* handle sign bit */
	    if (l_obsize < 0)
	    {
		sign = -1;
		l_obsize = -(l_obsize);
	    }

/* mask
6  6   5      5 4        4        3  3     2    2   1      1 0
32109876|54321098|76543210|98765432|10987654|32109876|54321098|76543210

1234                 12345|67890123|45                 1234567|89012345
    1234|56789012|345                 123456|78901234|5
--
       |<--------------- L1 -------------->|<--------------- L0 -------------->|
 6    6|6  6 5        5 4   44   4 3      3|33 2        2 1   11   1 0        0
 987654|3210 9876543210 9876543210 98765432|10 9876543210 9876543210 9876543210
 444444|4444 3333333333 3333322222 22222222|22 1111111111 1111100000 0000000000

  |<--------------- L3 -------------->|<-------------- L2 -------------->|
 1|1      1 1    1   1 1    1   1     |
 2|2      2 1    1   1 0    0   0 9  9|9    9 8        8 7   77   76    6|6  6
 8|76543210 9876543210 9876543210 9876|543210 9876543210 9876543210987654|3210
 8|88888888 7777777777 7777766666 6666|666666 5555555555 5555544444444444|4444
*/
	    *o_octaword = v->ob_digit[0];	/* bits 0:14 */

	    if (l_obsize > 1) /* 2... */
	    {
		t = v->ob_digit[1];		/* bits 15:29 */
		t <<= SHIFT;			/* place */
		*o_octaword |= t;		/* merge: bits 0:14 + 15:29 */
	    }

	    /* bits 31:30 in first longword of octaword */
	    /*  put remaining bits into 2nd longword at 12:0 of octaword */
	    if (l_obsize > 2) /* 3... */
	    {
		t = (v->ob_digit[2] & 3);	/* bits 30:31 */
		t <<= 30;			/* place */
		*o_octaword |= t;		/* merge: bits 0:29 + 30:31 */
		/* ... */
		t = (v->ob_digit[2] & 0xfffc);	/* remove lower 2 bits */
		t >>= 2;			/* shift these bits */
		l_tmp = t;			/* bits 32:44 */
	    }
	    else
	    {
		l_tmp = 0;
	    }
	    o_octaword++;	/* next longword */
	    *o_octaword = l_tmp;

	    /* bits 45:59 in second longword of octaword */
	    if (l_obsize > 3) /* 4... */
	    {
		t = v->ob_digit[3];		/* bits 45:59 */
		t <<= 13;			/* 2 bits of prev digit */
		*o_octaword |= t;		/* merge: 32:44 + 45:59 */
	    }

	    /* bits 60:63 in second longword of octaword */
	    if (l_obsize > 4) /* 5... */
	    {
		t = v->ob_digit[4] & 0xf;	/* bits 60:63 */
		t <<= (SHIFT+SHIFT-2);
		*o_octaword |= t;		/* merge 32:59 + 60:63 */
		/* ... */
		t = (v->ob_digit[4] & 0x7ff0);	/* remove lower 4 bits */
		t >>= 4;			/* shift these bits */
		l_tmp = t;			/* bits 63:74 */
	    }
	    else
	    {
		l_tmp = 0;
	    }
	    o_octaword++;	/* next longword */
	    *o_octaword = l_tmp;

	    /* bits 75:89 in third longword of octaword */
	    if (l_obsize > 5) /* 6... */
	    {
		t = v->ob_digit[5];		/* bits 75:89 */
		t <<= 11;			/* 11 bits of prev digit */
		*o_octaword |= t;		/* merge: 64:74 + 75:89 */
	    }

	    /* bits 90:95 in third longword of octaword */
	    if (l_obsize > 6) /* 7... */
	    {
		t = v->ob_digit[6] & 0x3f;	/* bits 90:95 */
		t <<= 26;
		*o_octaword |= t;		/* merge 64:89 + 90:95 */
		/* ... */
		t = (v->ob_digit[6] & 0x7fc0);	/* remove lower 6 bits */
		t >>= 6;			/* shift these bits */
		l_tmp = t;			/* bits 96:104 */
	    }
	    else
	    {
		l_tmp = 0;
	    }
	    o_octaword++;	/* next longword */
	    *o_octaword = l_tmp;

	    /* bits 105:119 in fourth longword of octaword */
	    if (l_obsize > 7) /* 8... */
	    {
		t = v->ob_digit[7];		/* bits 105:119 */
		t <<= 9;			/* 9 bits of prev digit */
		*o_octaword |= t;		/* merge: 96:104 + 105:119 */
	    }

	    /* bits 120:127 in fourth longword of octaword */
	    if (l_obsize > 8) /* 9... */
	    {
		/* @@sign_bit = v->ob_digit[8] & 0x80; */
		t = v->ob_digit[8] & 0xff;	/* bits 120:127 */
		t <<= 24;
		*o_octaword |= t;		/* merge 96:119 + 120:127 */

		if ((v->ob_digit[8] & 0x7f00) != 0)
		{
		    /* @@ overflow */
		    /*	PyErr_SetString(PyExc_OverflowError,
		    ||		"long integer too long to convert octaword");
		    */
		    /* @@ set overflow marker */
		    /* don't abort here */
		}
	    }

          if ((*o_octaword     == 0) &&
	      (*(o_octaword-1) == 0) &&
	      (*(o_octaword-2) == 0) &&
	      (*(o_octaword-3) == 0) &&
	      (sign_bit != 0)
	     )
	  {
		/* maximum negative value - keep it */
		*o_octaword = 0x80000000; /* patch in 'sign_bit' */
	  }
	  else
	  {
	    if ((sign < 0) || (sign_bit != 0))
	    {
		long q_add1[4];
		long l_arraylength = 4;     /* -> lib$addx() */
                long l_status;

		q_add1[0] = 1;
		q_add1[1] = 0;
		q_add1[2] = 0;
		q_add1[3] = 0;

		*o_octaword = ~(*o_octaword);	/* upper longword */
		o_octaword--;	/* pointer to 1st (=low) longword */
		*o_octaword = ~(*o_octaword);	/* lower longword */
		o_octaword--;	/* pointer to 1st (=low) longword */
		*o_octaword = ~(*o_octaword);	/* lower longword */
		o_octaword--;	/* pointer to 1st (=low) longword */
		*o_octaword = ~(*o_octaword);	/* lower longword */

		/* ignore condition value SS$_INTOVF */
		(void) lib$addx
			(&q_add1[0]	/* addend-array              */
			,o_octaword	/* augend-array              */
			,o_octaword	/* resultant-array           */
			,&l_arraylength /* [,array-length] longwords */
			);
	    }
	    if (l_obsize > 9) /* 10... */
	    {
		/* @@ check remaining bits for overflow */
		    /* @@ overflow */
		    /*	PyErr_SetString(PyExc_OverflowError,
		    ||		"long integer too long to convert octaword");
		    */
		    return 0;
	    }
	  }
	}
	return 1;
} /* vms__cvt_pylong2octa () */

/* ------------------------------------------------------------------------- */

/* EOF: VMS__CVT_PYLONG.C */
