/* VMS__GETTIMEOFDAY.C -- 14-DEC-1997 Uwe Zessin

   Emulate unix's gettimeofday call; timezone is ignored.
*/

/* Date: Mon, 21 Nov 1994 17:43:00 PDT
From: Pat Rankin <rankin@EQL.CALTECH.EDU>
Subject: Re: interesting problem with delta times

[...]
>I have an application that requires to convert a VMS absolute time
>quadword into the number of seconds since 1-jan-1970 00:00 (for
>later display with the ctime() routine). Don't ask why...
[...]
>I'm looking for an elegant algorithm to implement this conversion. Checking
>the return status from lib$sub_times and trying with multiple base times seems
>a bit messy. Perhaps using lib$ediv to divide the original absolute time by
>10**9 and subtracting the known number of seconds between 17-nov-1858 00:00
>and 1-jan-1970 00:00 is the way to go? How to obtain this number of seconds?

Use lib$subx instead of lib$sub_times if the latter actually limits its result
to less than 10000 days. You'll still have to check separately for something
earlier than your arbitrary base date. After the subtraction, use lib$ediv
instead of the time conversion routine, but your divisor should be 10**7, not
10**9 (the units are hundreds of nanoseconds, not "bare" nanoseconds).

This is my rewrite of a routine included in the VMS port of `tcl'. It isn't
doing precisely what you want, but does have all the bits you need, I think,
except for the check for "too old" dates. This routine doesn't bother trying to
round to the nearest microsecond, which is not relevant to your problem anyway.

Pat Rankin, rankin@eql.caltech.edu
-----
*/
#include <descrip.h>
#include <lib$routines.h>	/* LIB$name */
#include <socket.h> /* @@+ ,defines timeval */
#include <starlet.h>		/* SYS$name */
#define VMS_UNITS_PER_SECOND 10000000L  /* hundreds of nanoseconds, 1e-7 */

int vms__gettimeofday (struct timeval *tv, void *timezone)
{
static $DESCRIPTOR(epoch_dsc, "01-JAN-1970 00:00:00.00");
static long	epoch[2] = {0,0};	/* needs one time initialization */
long		thunk = VMS_UNITS_PER_SECOND;
long		now[2], quad[2];

if (!epoch[0])
    sys$bintim(&epoch_dsc, epoch);	/* 1 Jan 0:0:0 1970 */
/* get current time, as VMS quadword time */
sys$gettim(now);
/* convert the quadword time so that it's relative to Unix epoch */
lib$subx(now, epoch, quad); /* quad = now - epoch; */
/* convert 1e-7 units into seconds and fraction of seconds */
lib$ediv(&thunk, quad, &tv->tv_sec, &tv->tv_usec);
/* convert fraction of seconds into microseconds */
tv->tv_usec /= (VMS_UNITS_PER_SECOND / 1000000);

return 0;		/* success */
}

/* EOF: VMS__GETTIMEOFDAY.C */
