/* VMS_LIB.C -- 30-OCT-1996 Uwe Zessin (ZE.)
   interface to (some) OpenVMS LIB$ routines

   06-SEP-1997 ZE. -- changes for new item-codes
   07-SEP-1997 ZE. -- "
   16-OCT-1997 ZE. -- Rename field 'w_vmsver' to 'w_vmsvermin' and add field
 		'w_vmsvermax' because some data can change from version
 		to version.
   02-NOV-1997 ZE. -- convert functions to use PyArg_ParseTuple() instead of
 		PyArg_Parse().
   07-NOV-1997 ZE. -- add GET_COMMON, PUT_COMMON
   08-NOV-1997 ZE. -- add GET_COMMAND, RUN_PROGRAM
   09-NOV-1997 ZE. -- cleanup + some optimizations
   29-NOV-1997 ZE. -- changes for Python V1.5B1
   18-DEC-1997 ZE. -- add FIND_IMAGE_SYMBOL, FREE_EF, GET_EF
   19-DEC-1997 ZE. -- add RESERVE_EF
   21-DEC-1997 ZE. -- add GET_FOREIGN
   31-DEC-1997 ZE. -- add PUT_OUTPUT, ADD_TIMES, SUB_TIMES, CREATE_DIR
   04-JAN-1998 ZE. -- add ASN_WTH_MAX
   09-JAN-1998 ZE. -- add CVT_VECTIM, FID_TO_NAME
   10-JAN-1998 ZE. -- add SET_LOGICAL w/o item-list
   11-JAN-1998 ZE. -- start SET_LOGICAL item-list support
   12-JAN-1998 ZE. -- continue SET_LOGICAL item-list support
   15-JAN-1998 ZE. -- continue SET_LOGICAL, close all memory leaks that
			have (intentionally) been left open during testing
   09-FEB-1998 ZE. -- add remaining DOC strings
   10-FEB-1998 ZE. -- convert day() and day_time() argument 1 to use 64-bit
			system time for compatibility with other routines
   11-AUG-1998 ZE. -- start conversion of 64-bit 'hex-string'
			to Python long integer
   12-AUG-1998 ZE. -- done conversion of 64-bit 'hex-string'
			to Python long integer
   06-SEP-1998 ZE. -- change 'c_typ_lib' and 'S' to w_typ_lib and ITMTYP_K_ASCII
   11-OCT-1998 ZE. -- changes to GETDVI and GETSYI, move code to separate files
   24-OCT-1998 ZE. -- add CVTF_FROM_INTERNAL_TIME - code is in
			VMS_LIB_CVTF_FRMINTTIM.C
		   -- add CVTF_TO_INTERNAL_TIME - code is in
			VMS_LIB_CVTF_TOINTTIM.C
   25-OCT-1998 ZE. -- add CVT_FROM_INTERNAL_TIME - code is in
			VMS_LIB_CVT_FRMINTTIM.C
		   -- add CVT_TO_INTERNAL_TIME - code is in
			VMS_LIB_CVT_TOINTTIM.C
		   -- add FIND_FILE + FIND_FILE_END
			- code of both routines is in VMS_LIB_FIND_FILE.C
   27-OCT-1998 ZE. -- add TRIM_FILESPEC - code is in VMS_LIB_TRIM_FILESPEC.C
   29-OCT-1998 ZE. -- changes to GETQUI, move code to VMS_LIB_GETQUI.C
   03-DEC-1998 ZE. -- changes to GETJPI, move code to VMS_LIB_GETJPI.C
   31-JAN-1999 ZE. -- add DELETE_FILE - code is in VMS_LIB_DELETE_FILE.C
   01-FEB-1999 ZE. -- add RENAME_FILE - code is in VMS_LIB_RENAME_FILE.C
   23-MAR-1999 ZE. -- replace some PyErr_SetString() with PyErr_Format()
   27-JUN-1999 ZE. -- vms_lib_methods[] is now configured via data file
		      VMS_LIB__METHODS.DAT - see 'General Manual', configuration
   15-JUL-1999 ZE. -- move DOC-strings into external files
*/

#include "python.h"
#include "vmsdef.h"

#include <libdef.h>		/* LIB$_name */
#include <lib$routines.h>	/* LIB$name */
#include <ssdef.h>		/* SS$_name */
#include <starlet.h>		/* SYS$name, ... */

#include <descrip.h>

/* ------------------------------------------------------------------------- */

PyObject *vms_lib_gr_error;		/* exception vms_lib.error */

static char vms_lib_doc__ [] =
"This is an interface to some OpenVMS LIB$ routines.\n\
See the VMS_LIB documentation for details.";

static char vms_lib_t_cant_init   [] = "can't initialize module vms_lib";
static char vms_lib_t_cant_deferr [] = "can't define vms_lib.error";

/* ------------------------------------------------------------------------- */
/* function prototypes for external routines */

/* ------------------------------ */

/* input a binary quadword - output a Python long integer */
extern PyObject * vms__cvt_quad2pylong (long q_quadword[2]);

/* ------------------------------ */

/* input a Python long integer - output a binary quadword */
extern long vms__cvt_pylong2quad (PyObject * r_longint, long * q_quadword);

/* ------------------------------------------------------------------------- */

char vms_lib_cvt_vectim__doc[] =
"resultant_time = vms_lib.cvt_vectim (input_time)\n\
Convert 7-Word Vector to Internal Time.";

PyObject *
vms_lib_cvt_vectim (PyObject *self, PyObject *args)
{
	PyObject		* ar_input_time;

	unsigned long		  l_tuple_size;
	unsigned long		  l_tuple_index;
	PyObject		* ar_tuple_element;
	unsigned long		  l_tuple_element;

	unsigned short int	  w_input_time[7];

	long	     		  q_resultant_time[2]; /* binary system time */

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O", &ar_input_time))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: input-time */
	if (!PyTuple_Check(ar_input_time))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 1: must be a tuple of 7 integers");
	    return NULL;
	}

	l_tuple_size = PyTuple_Size(ar_input_time);
	if (l_tuple_size != 7)
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 1: must be a tuple of 7 integers");
	    return NULL;
	}

	for (l_tuple_index = 0; l_tuple_index < 7; l_tuple_index++)
	{
	    ar_tuple_element = PyTuple_GetItem(ar_input_time, l_tuple_index);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		return PyErr_Format(PyExc_TypeError,
		    "argument 1: tuple-element:%d is not an integer",
		    l_tuple_index);
	    }

	    l_tuple_element = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }

	    if (l_tuple_element > 65535)
	    {
		return PyErr_Format(PyExc_TypeError,
		    "argument 1: tuple-element:%d is not a 16-bit integer",
		    l_tuple_index);
	    }

	    w_input_time[l_tuple_index] = l_tuple_element;
	}

	/* -------------------- */
	l_status = lib$cvt_vectim (&w_input_time[0], &q_resultant_time[0]);

	/* -------------------- */
	if (l_status == LIB$_NORMAL)
	{
	    /* convert binary quadword to Python long integer */
	    return vms__cvt_quad2pylong (&q_resultant_time[0]);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_cvt_vectim () */

/* ------------------------------------------------------------------------- */

char vms_lib_day__doc[] =
"number_of_days, day_time = vms_lib.day ([user-time])\n\
Day Number Returned as a Longword Integer.";

PyObject *
vms_lib_day (PyObject *self, PyObject *args)
{
	PyObject		* ar_user_time;
	long			  q_user_time[2];
	long			* aq_user_time;

	unsigned long		  l_number_of_days;
	unsigned long		  l_day_time;

	unsigned long		  l_status;

	/* -------------------- */
	ar_user_time = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|O", &ar_user_time))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1 - user-time */
	if (ar_user_time == Py_None)
	{
	    aq_user_time = 0;	/* omitted - use current date + time */
	}
        else
	{
	    if (!PyLong_Check(ar_user_time))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: must be long integer or None");
		return NULL;
	    }

	    /* convert Python long integer to binary quadword */
	    l_status = vms__cvt_pylong2quad (ar_user_time, &q_user_time[0]);
	    if (l_status == -1)
	    {
		return NULL;		/* error string has been set */
	    }
	    aq_user_time = &q_user_time[0];
	}

	/* -------------------- */
	l_status = lib$day
			(&l_number_of_days
			,aq_user_time
			,&l_day_time
			);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return Py_BuildValue ("ll", l_number_of_days, l_day_time);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_day () */

/* ------------------------------------------------------------------------- */

char vms_lib_day_of_week__doc[] =
"day_number = vms_lib.day_of_week ([user-time])\n\
Show Numeric Day of Week.";

PyObject *
vms_lib_day_of_week (PyObject *self, PyObject *args)
{
	PyObject		* ar_user_time;
	long			  q_user_time[2];
	long			* aq_user_time;

	unsigned long		  l_day_number;

	unsigned long		  l_status;

	/* -------------------- */
	ar_user_time = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|O", &ar_user_time))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: user-time */
	if (ar_user_time == Py_None)
	{
	    aq_user_time = 0;		/* omitted */
	}
        else
	{
	    if (!PyLong_Check(ar_user_time))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: must be long integer or None");
		return NULL;
	    }

	    /* convert Python long integer to binary quadword */
	    l_status = vms__cvt_pylong2quad (ar_user_time, &q_user_time[0]);
	    if (l_status == -1)
	    {
		return NULL;		/* error string has been set */
	    }
	    aq_user_time = &q_user_time[0];
	}

	/* -------------------- */
	l_status = lib$day_of_week
		(aq_user_time
		,&l_day_number
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return PyInt_FromLong(l_day_number);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_day_of_week () */

/* ------------------------------------------------------------------------- */

char vms_lib_delete_logical__doc[] =
"vms_lib.delete_logical (logical-name [,table-name])\n\
Delete Logical Name.";

PyObject *
vms_lib_delete_logical (PyObject *self, PyObject *args)
{
	char			* at_logical_name;
	struct dsc$descriptor_s   r_logical_name;
	unsigned long		  l_logical_name_len;

	char			* at_table_name;
	struct dsc$descriptor_s   r_table_name;
	struct dsc$descriptor_s * ar_table_name;
	unsigned long		  l_table_name_len;

	unsigned long		  l_status;

	/* -------------------- */
	at_table_name = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#|z#",
		&at_logical_name, &l_logical_name_len,
		&at_table_name,   &l_table_name_len   ))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: logical-name */
	if (l_logical_name_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	 "argument 1: logical-name - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up string descriptor */
	r_logical_name.dsc$w_length  = l_logical_name_len;
	r_logical_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_logical_name.dsc$b_class   = DSC$K_CLASS_S;
	r_logical_name.dsc$a_pointer = at_logical_name;

	/* -------------------- */
	/* argument 2: table-name */
	if (at_table_name == NULL)
	{
	    ar_table_name = 0;		/* omitted */
	}
	else
	{
	    /* string length in a VMS descriptor is limited to 65535 chars */
	    if (l_table_name_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	   "argument 2: table-name - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up string descriptor */
	    r_table_name.dsc$w_length  = l_table_name_len;
	    r_table_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_table_name.dsc$b_class   = DSC$K_CLASS_S;
	    r_table_name.dsc$a_pointer = at_table_name;
	    ar_table_name = &r_table_name;
	}

	/* -------------------- */
	l_status = lib$delete_logical (&r_logical_name, ar_table_name);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_delete_logical () */

/* ------------------------------------------------------------------------- */

char vms_lib_delete_symbol__doc[] =
"vms_lib.delete_symbol (symbol [,table-type-indicator])\n\
Delete CLI Symbol.";

PyObject *
vms_lib_delete_symbol (PyObject *self, PyObject *args)
{
	char			* at_symbol;
	struct dsc$descriptor_s   r_symbol;
	unsigned long		  l_symbol_len;

	PyObject		* ar_table_type_indicator;
	long			  l_table_type_indicator;
	long			* al_table_type_indicator;

	unsigned long		  l_status;

	/* -------------------- */
	ar_table_type_indicator = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#|O",
		&at_symbol, &l_symbol_len,
		&ar_table_type_indicator))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: symbol */
	if (l_symbol_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	       "argument 1: symbol - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up string descriptor */
	r_symbol.dsc$w_length  = l_symbol_len;
	r_symbol.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_symbol.dsc$b_class   = DSC$K_CLASS_S;
	r_symbol.dsc$a_pointer = at_symbol;

	/* -------------------- */
	/* argument 2: table_type_indicator */
	if (ar_table_type_indicator == Py_None)
	{
	    al_table_type_indicator = 0;	/* omitted */
	}
	else
	{
	    if (!PyInt_Check(ar_table_type_indicator))
	    {
		PyErr_SetString(PyExc_TypeError,
		   "argument 2: table-type-indicator must be integer or None");
		return NULL;
	    }
	    l_table_type_indicator = PyInt_AsLong(ar_table_type_indicator);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_table_type_indicator = &l_table_type_indicator;
	}

	/* -------------------- */
	l_status = lib$delete_symbol (&r_symbol, al_table_type_indicator);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_delete_symbol () */

/* ------------------------------------------------------------------------- */

char vms_lib_do_command__doc[] =
"vms_lib.do_command (command-string)\n\
Execute Command.";

PyObject *
vms_lib_do_command (PyObject *self, PyObject *args)
{
	char			* at_command_string;
	struct dsc$descriptor_s   r_command_string;
	unsigned long		  l_command_string_len;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#",
	     &at_command_string, &l_command_string_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: command */
	if (l_command_string_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	      "argument 1: command - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up string descriptor */
	r_command_string.dsc$w_length  = l_command_string_len;
	r_command_string.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_command_string.dsc$b_class   = DSC$K_CLASS_S;
	r_command_string.dsc$a_pointer = at_command_string;

	/* -------------------- */
	l_status = lib$do_command (&r_command_string);

	/* -------------------- */
	/* any return from LIB$DO_COMMAND indicates an error! */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_do_command () */

/* ------------------------------------------------------------------------- */

/* Find Universal Symbol in Shareable Image File */

char vms_lib_find_image_symbol__doc[] =
"symbol_value, status = \
vms_lib.find_image_symbol (filename, symbol [,image_name])\n\
Find Universal Symbol in Shareable Image File.";

/* ---------------------------------------- */
long vms_lib__fis
	(struct dsc$descriptor_s * ar_filename
	,struct dsc$descriptor_s * ar_symbol
	,long			 * al_symbol_value
	,struct dsc$descriptor_s * ar_image_name
	)
{
	unsigned long l_status;

	/* -------------------- */
	/* establish a condition handler to trap any signals and */
	/*  turn them into return values */
	(void) lib$establish (lib$sig_to_ret);
	/* no condition value is returned */

	/* -------------------- */
	l_status = lib$find_image_symbol
		(ar_filename
		,ar_symbol
		,al_symbol_value
		,ar_image_name);

	/* -------------------- */
	return (l_status);
} /* vms_lib__fis () */

/* ---------------------------------------- */

PyObject *
vms_lib_find_image_symbol (PyObject *self, PyObject *args)
{
	char			* at_filename;
	struct dsc$descriptor_s   r_filename;
	unsigned long		  l_filename_len;

	char			* at_symbol;
	struct dsc$descriptor_s   r_symbol;
	unsigned long		  l_symbol_len;

	char			* at_image_name;
	struct dsc$descriptor_s   r_image_name;
	unsigned long		  l_image_name_len;
	struct dsc$descriptor_s * ar_image_name;

	long			  l_symbol_value;
	unsigned long		  l_status;

	/* -------------------- */
	at_image_name    = NULL;
	l_image_name_len = 0;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#s#|z#",
	     &at_filename,   &l_filename_len,
	     &at_symbol,     &l_symbol_len,
	     &at_image_name, &l_image_name_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: filename */
	if (l_filename_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	     "argument 1: filename - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up string descriptor */
	r_filename.dsc$w_length  = l_filename_len;
	r_filename.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_filename.dsc$b_class   = DSC$K_CLASS_S;
	r_filename.dsc$a_pointer = at_filename;

	/* -------------------- */
	/* argument 2: symbol */
	if (l_symbol_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	       "argument 2: symbol - string size limited to 65535 characters");
	    return NULL;
	}
	r_symbol.dsc$w_length  = l_symbol_len;
	r_symbol.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_symbol.dsc$b_class   = DSC$K_CLASS_S;
	r_symbol.dsc$a_pointer = at_symbol;

	/* -------------------- */
	/* argument 3: image-name */
	if (at_image_name == NULL)
	{
	    ar_image_name = 0;		/* omitted */
	}
	else
	{
	    if (l_image_name_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	   "argument 3: image-name - string size limited to 65535 characters");
		return NULL;
	    }
	    r_image_name.dsc$w_length  = l_image_name_len;
	    r_image_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_image_name.dsc$b_class   = DSC$K_CLASS_S;
	    r_image_name.dsc$a_pointer = at_image_name;
	    ar_image_name = &r_image_name;
	}

	l_symbol_value = 0;

	/* -------------------- */
	l_status = vms_lib__fis
		(&r_filename
		,&r_symbol
		,&l_symbol_value
		,ar_image_name);

	/* -------------------- */
	/* return symbol-value and status */
	/* no error is signaled to Python - MUST check the status ! */
	return Py_BuildValue ("ll", l_symbol_value, l_status);
} /* vms_lib_find_image_symbol () */

/* ------------------------------------------------------------------------- */

char vms_lib_free_ef__doc[] =
"vms_lib.free_ef (event_flag_number)\n\
Free Event Flag.";

PyObject *
vms_lib_free_ef (PyObject *self, PyObject *args)
{
	unsigned long	l_efn;
	unsigned long	l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_efn))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = lib$free_ef (&l_efn);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_free_ef () */

/* ------------------------------------------------------------------------- */

char vms_lib_get_command__doc[] =
"data_from_command, resultant_length = \
vms_lib.get_command (prompt [,buffer_length])\n\
Get Line from SYS$COMMAND.";

PyObject *
vms_lib_get_command (PyObject *self, PyObject *args)
{
#define S_COMMAND_BUFFER 65535
	char			* at_prompt_string;
	struct dsc$descriptor_s   r_prompt_string;
	unsigned long		  l_prompt_length;
	void			* ar_prompt_string;

	char			* at_buffer;
	PyObject		* ar_buffer;

	unsigned long		  l_command_buffer;
	unsigned long 		  l_resultant_length;
	struct dsc$descriptor_s   r_resultant_string;
	unsigned short int 	  w_resultant_length;

	unsigned long		  l_status;

	/* -------------------- */
	at_prompt_string = NULL; l_prompt_length = 0;

	/* argument 2 is optional, 'integer', default=65535 */
	l_command_buffer = S_COMMAND_BUFFER;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|z#l",
	    &at_prompt_string, &l_prompt_length,
	    &l_command_buffer))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: prompt-string */
	if (at_prompt_string == NULL)
	{
	    ar_prompt_string = 0;	       /* no prompt string requested */
	}
	else
	{
	    if (l_prompt_length > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	"argument 1: prompt-string - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prompt_string.dsc$w_length  = l_prompt_length;
	    r_prompt_string.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prompt_string.dsc$b_class   = DSC$K_CLASS_S;
	    r_prompt_string.dsc$a_pointer = at_prompt_string;
	    ar_prompt_string = &r_prompt_string;
	}

	/* -------------------- */
	/* argument 2: command-buffer-size */
	if (l_command_buffer > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
  "argument 2: command-buffer-size - string size limited to 65535 characters");
	    return NULL;
	}

	/* allocate input buffer */
	at_buffer = malloc (l_command_buffer);
	if (at_buffer == NULL)
	{
	    return PyErr_NoMemory();
	}

	/* set up descriptor */
	r_resultant_string.dsc$w_length  = l_command_buffer;
	r_resultant_string.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_resultant_string.dsc$b_class   = DSC$K_CLASS_S;
	r_resultant_string.dsc$a_pointer = at_buffer;

	/* -------------------- */
	l_status = lib$get_command
	    (&r_resultant_string
	    ,ar_prompt_string		/* optional */
	    ,&w_resultant_length
	    );

	/* -------------------- */
	/* note: string truncation is not considered an error!  */
	/* Please check 2nd value returned.			*/
	if ((l_status == SS$_NORMAL) || (l_status == LIB$_INPSTRTRU))
	{
	    PyObject * ar_return;

	    /* create string object for data received from SYS$COMMAND */
	    ar_buffer = Py_BuildValue ("s#",
		at_buffer, (unsigned int)w_resultant_length);

	    /* release temporary buffer */
	    (void)free (at_buffer);

	    if (ar_buffer == NULL)
	    {
		return NULL;		/* object creation failed */
	    }

	    /* change resultant length to a negative value */
	    /*   if string truncation occured		   */
	    if (l_status == LIB$_INPSTRTRU)
	    {
		l_resultant_length = 0- (unsigned long)w_resultant_length;
	    }
	    else
	    {
		l_resultant_length = (unsigned long)w_resultant_length;
	    }

	    ar_return = Py_BuildValue ("Ol", ar_buffer, l_resultant_length);
	    Py_DECREF (ar_buffer);
	    return ar_return;
	}

	/* -------------------- */
	/* release temporary buffer */
	(void)free (at_buffer);

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_get_command () */

/* ------------------------------------------------------------------------- */

char vms_lib_get_ef__doc[] =
"event_flag_number = vms_lib.get_ef ()\n\
Get Event Flag.";

PyObject *
vms_lib_get_ef (PyObject *self, PyObject *args)
{
	unsigned long	l_efn;
	unsigned long	l_status;

	/* -------------------- */
	/* no input argument */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = lib$get_ef (&l_efn);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return PyInt_FromLong(l_efn);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_get_ef () */

/* ------------------------------------------------------------------------- */

char vms_lib_lp_lines__doc[] =
"lp_line_count = vms_lib.lp_lines ()\n\
Lines on Each Printer Page.";

PyObject *
vms_lib_lp_lines (PyObject *self, PyObject *args)
{
	unsigned long	l_lp_lines;

	/* -------------------- */
	/* no arguments taken */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	l_lp_lines = lib$lp_lines ();

	/* -------------------- */
	return PyInt_FromLong(l_lp_lines);
	/* no error documented */
} /* vms_lib_lp_lines () */

/* ------------------------------------------------------------------------- */

char vms_lib_put_common__doc[] =
"resultant_length = vms_lib.put_common (source_string)\n\
Put String to Common.";

PyObject *
vms_lib_put_common (PyObject *self, PyObject *args)
{
	char			* at_source_string;
	struct dsc$descriptor_s   r_source_string;
	unsigned long		  l_source_string_len;

	unsigned short int 	  w_resultant_length;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#",
	    &at_source_string, &l_source_string_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* maximum space in the common area is 252 bytes ... */
	if (l_source_string_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	"argument 1: source-string - string size limited to 65535 characters");
	    return NULL;
	}
	w_resultant_length = l_source_string_len;

	/* -------------------- */
	/* set up descriptor */
	r_source_string.dsc$w_length  = w_resultant_length;
	r_source_string.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_source_string.dsc$b_class   = DSC$K_CLASS_S;
	r_source_string.dsc$a_pointer = at_source_string;

	/* -------------------- */
	l_status = lib$put_common (&r_source_string, &w_resultant_length);

	/* -------------------- */
	/* Warning - string truncation is considered an error, too */
	if (l_status == SS$_NORMAL)
	{
	    return PyInt_FromLong((unsigned int)w_resultant_length);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_put_common () */

/* ------------------------------------------------------------------------- */

char vms_lib_put_output__doc[] =
"vms_lib.put_output (message_string)\n\
Put Line to SYS$OUTPUT.";

PyObject *
vms_lib_put_output (PyObject *self, PyObject *args)
{
	char*			at_message_string;
	struct dsc$descriptor_s r_message_string;
	unsigned long		l_message_string_len;

	unsigned long		l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#",
	    &at_message_string, &l_message_string_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: message-string */
	if (l_message_string_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
       "argument 1: message-string - string size limited to 65535 characters");
	    return NULL;
	}

	/* set up descriptor */
	r_message_string.dsc$w_length  = l_message_string_len;
	r_message_string.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_message_string.dsc$b_class   = DSC$K_CLASS_S;
	r_message_string.dsc$a_pointer = at_message_string;

	/* -------------------- */
	l_status = lib$put_output (&r_message_string);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_put_output () */

/* ------------------------------------------------------------------------- */

char vms_lib_reserve_ef__doc[] =
"vms_lib.reserve_ef (event_flag_number)\n\
Reserve Event Flag.";

PyObject *
vms_lib_reserve_ef (PyObject *self, PyObject *args)
{
	unsigned long	l_efn;
	unsigned long	l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_efn))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = lib$reserve_ef (&l_efn);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_reserve_ef () */

/* ------------------------------------------------------------------------- */

char vms_lib_run_program__doc[] =
"vms_lib.run_program (program_name)\n\
Run New Program.";

PyObject *
vms_lib_run_program (PyObject *self, PyObject *args)
{
	char			* at_program_name;
	struct dsc$descriptor_s	  r_program_name;
	unsigned long		  l_program_name_len;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#",
	     &at_program_name, &l_program_name_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: program-name */
	/* @@ that does not make much sense because a file specification is */
	/* @@ limited to 255 characters anyway. */
	if (l_program_name_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	 "argument 1: program-name - string size limited to 65535 characters");
	    return NULL;
	}

	/* set up descriptor */
	r_program_name.dsc$w_length  = l_program_name_len;
	r_program_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_program_name.dsc$b_class   = DSC$K_CLASS_S;
	r_program_name.dsc$a_pointer = at_program_name;

	/* -------------------- */
	l_status = lib$run_program (&r_program_name);

	/* -------------------- */
	/* any return from LIB$RUN_PROGRAM indicates an error! */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_run_program () */

/* ------------------------------------------------------------------------- */

char vms_lib_set_symbol__doc[] =
"vms_lib.set_symbol (symbol, value-string [,table-type-indicator])\n\
Set Value of CLI Symbol.";

PyObject *
vms_lib_set_symbol (PyObject *self, PyObject *args)
{
	char			* at_symbol;
	struct dsc$descriptor_s   r_symbol;
	unsigned long		  l_symbol_len;

	char			* at_value_string;
	struct dsc$descriptor_s   r_value_string;
	unsigned long		  l_value_string_len;

	PyObject		* ar_table_type_indicator;
	long			  l_table_type_indicator;
	long			* al_table_type_indicator;

	unsigned long		  l_status;

	/* -------------------- */
	ar_table_type_indicator = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#s#|O",
	    &at_symbol,       &l_symbol_len,
	    &at_value_string, &l_value_string_len,
	    &ar_table_type_indicator
	    ))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: symbol */
	if (l_symbol_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	       "argument 1: symbol - string size limited to 65535 characters");
	    return NULL;
	}

	/* set up descriptor */
	r_symbol.dsc$w_length  = l_symbol_len;
	r_symbol.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_symbol.dsc$b_class   = DSC$K_CLASS_S;
	r_symbol.dsc$a_pointer = at_symbol;

	/* -------------------- */
	/* argument 2: value-string */
	if (l_value_string_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	 "argument 1: value-string - string size limited to 65535 characters");
	    return NULL;
	}

	/* set up descriptor */
	r_value_string.dsc$w_length  = l_value_string_len;
	r_value_string.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_value_string.dsc$b_class   = DSC$K_CLASS_S;
	r_value_string.dsc$a_pointer = at_value_string;

	/* -------------------- */
	/* argument 3: table-type-indicator */
	if (ar_table_type_indicator == Py_None)
	{
	    al_table_type_indicator = 0;	/* omitted */
	}
        else
	{
	    if (!PyInt_Check(ar_table_type_indicator))
	    {
		PyErr_SetString(PyExc_TypeError,
		  "argument 3: table-type-indicator must be integer or None");
		return NULL;
	    }
	    l_table_type_indicator = PyInt_AsLong(ar_table_type_indicator);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_table_type_indicator = &l_table_type_indicator;
	}

	/* -------------------- */
	l_status = lib$set_symbol
			(&r_symbol
			,&r_value_string
			,al_table_type_indicator);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_set_symbol () */

/* ------------------------------------------------------------------------- */

char vms_lib_sub_times__doc[] =
"resultant_time = vms_lib.sub_times (time1, time2)\n\
Subtract Two Quadword Times.";

PyObject *
vms_lib_sub_times (PyObject *self, PyObject *args)
{
	PyObject	* ar_time1;
	long		  q_time1[2];

	PyObject	* ar_time2;
	long		  q_time2[2];

	long		  q_resultant_time[2];

	unsigned long	  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "OO", &ar_time1, &ar_time2))
	{
	    return NULL;
	}

	/* -------------------- */
	if (!PyLong_Check(ar_time1))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 1: must be long integer");
	    return NULL;
	}

	/* convert Python long integer to binary quadword */
	l_status = vms__cvt_pylong2quad (ar_time1, &q_time1[0]);
	if (l_status == -1)
	{
	    return NULL;		/* error string has been set */
	}
	
	/* -------------------- */
	if (!PyLong_Check(ar_time2))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 2: must be long integer");
	    return NULL;
	}

	/* convert Python long integer to binary quadword */
	l_status = vms__cvt_pylong2quad (ar_time2, &q_time2[0]);
	if (l_status == -1)
	{
	    return NULL;		/* error string has been set */
	}

	/* -------------------- */
	l_status = lib$sub_times
	    (&q_time1[0], &q_time2[0], &q_resultant_time[0]);

	/* -------------------- */
	if (l_status == LIB$_NORMAL)	/* note: _not_ SS$_NORMAL !! */
	{
	    /* convert binary quadword to Python long integer */
	    return vms__cvt_quad2pylong (&q_resultant_time[0]);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_sub_times () */

/* ------------------------------------------------------------------------- */

char vms_lib_wait__doc[] =
"vms_lib.wait (seconds)\n\
Wait a Specified Period of Time.";

PyObject *
vms_lib_wait (PyObject *self, PyObject *args)
{
	float		f_wait_time;
	unsigned long	l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "f", &f_wait_time))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = lib$wait (&f_wait_time);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status);
} /* vms_lib_wait () */

/* ------------------------------------------------------------------------- */

/* static struct PyMethodDef vms_lib_methods[] = {
--	{NULL,		NULL}
-- }; */

/* data is now in file 'VMS_LIB__METHODS.DAT' */
/* see 'General Manual', configuration guidelines */
extern struct PyMethodDef vms_lib__methods[];

void
initvms_lib()
{
	PyObject *m, *d;

	m = Py_InitModule4("vms_lib",
			   vms_lib__methods, /* @@ vms_lib_methods, */
			   vms_lib_doc__,
			   (PyObject *)NULL,
			   PYTHON_API_VERSION);
	d = PyModule_GetDict(m);

	/* -------------------- */
        if (PyErr_Occurred())
	{
	    (void) Py_FatalError(vms_lib_t_cant_init);
	}

	/* -------------------- */
	/* Initialize exception */
	vms_lib_gr_error = PyErr_NewException("vms_lib.error", NULL, NULL);
	if (vms_lib_gr_error == NULL)
	{
	    (void) Py_FatalError(vms_lib_t_cant_deferr);
	}

	if (PyDict_SetItemString(d, "error", vms_lib_gr_error) != 0)
	{
	    (void) Py_FatalError(vms_lib_t_cant_deferr);
	}
} /* initvms_lib () */

/* ------------------------------------------------------------------------- */

/* EOF: VMS_LIB.C */
