/* VMS_LIB_FIND_FILE.C -- 25-OCT-1998 Uwe Zessin
   Python interface to LIB$FIND_FILE + LIB$FIND_FILE_END

   25-OCT-1998 ZE. -- created
   02-DEC-1998 ZE. -- change to dynamic string descriptors
*/


#include "python.h"
#include "vmsdef.h"

#include <descrip.h>
#include <lib$routines.h>	/* LIB$name  */
#include <rmsdef.h>		/* RMS$_name */
#include <ssdef.h>		/* SS$_name  */

/* ------------------------------------------------------------------------- */
extern PyObject *vms_lib_gr_error;		/* exception vms_lib.error */
/* ------------------------------------------------------------------------- */

char vms_lib_find_file__doc[] =
"status, status_value, context, resultant_filespec = vms_lib.find_file \
(filespec, context, [default_filespec], [related_filespec], [flags])\n\
Find File.";

/* ------------------------------------------------------------------------- */

PyObject *
vms_lib_find_file (PyObject *self, PyObject *args)
{
	char			* at_filespec;
	struct dsc$descriptor_s   r_filespec;
	unsigned long		  l_filespec_len;

	PyObject		* ar_context;
	unsigned long		  l_context;

	char			* at_default_filespec;
	struct dsc$descriptor_s   r_default_filespec;
	struct dsc$descriptor_s * ar_default_filespec;
	unsigned long		  l_default_filespec_len;

	char			* at_related_filespec;
	struct dsc$descriptor_s   r_related_filespec;
	struct dsc$descriptor_s * ar_related_filespec;
	unsigned long		  l_related_filespec_len;

	unsigned long		  l_status_value;

	PyObject		* ar_flags;
	unsigned long		   l_flags;
	unsigned long		* al_flags;

	struct dsc$descriptor_s   r_resultant_filespec;

	PyObject		* ar_resfilspe;

	unsigned long		  l_status_lib;
	unsigned long		  l_status_free;

	PyObject		* ar_return;

	/* -------------------- */
	at_default_filespec = NULL; l_default_filespec_len = 0;
	at_related_filespec = NULL; l_related_filespec_len = 0;
	ar_flags            = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#O|z#z#O",
	    &at_filespec, &l_filespec_len,
	    &ar_context,
	    &at_default_filespec, &l_default_filespec_len,
	    &at_related_filespec, &l_related_filespec_len,
	    &ar_flags))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1 : filespec */
	if (l_filespec_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	      "argument 1: filespec - string size limited to 65535 characters");
	    return NULL;
	}
	r_filespec.dsc$w_length  = l_filespec_len;
	r_filespec.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_filespec.dsc$b_class   = DSC$K_CLASS_S;
	r_filespec.dsc$a_pointer = at_filespec;

	/* -------------------- */
	/* argument 2 : context */
	if (ar_context == Py_None)
	{
	    l_context = 0;		/* omitted */
	}
	else
	{
	    if (!PyInt_Check(ar_context))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: context - must be integer or None");
		return NULL;
	    }
	    l_context = PyInt_AsLong(ar_context);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 3 : default_filespec */
	if (at_default_filespec == NULL)
	{
	    ar_default_filespec = 0;	/* omitted */
	}
	else
	{
	    if (l_default_filespec_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
     "argument 3: default_filespec - string size limited to 65535 characters");
		return NULL;
	    }
	    r_default_filespec.dsc$w_length  = l_default_filespec_len;
	    r_default_filespec.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_default_filespec.dsc$b_class   = DSC$K_CLASS_S;
	    r_default_filespec.dsc$a_pointer = at_default_filespec;
	    ar_default_filespec = &r_default_filespec;
	}

	/* -------------------- */
	/* argument 4 - related_filespec */
	if (at_related_filespec == NULL)
	{
	    ar_related_filespec = 0;	/* omitted */
	}
	else
	{
	    if (l_related_filespec_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
     "argument 4: related_filespec - string size limited to 65535 characters");
		return NULL;
	    }
	    r_related_filespec.dsc$w_length  = l_related_filespec_len;
	    r_related_filespec.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_related_filespec.dsc$b_class   = DSC$K_CLASS_S;
	    r_related_filespec.dsc$a_pointer = at_related_filespec;
	    ar_related_filespec = &r_related_filespec;
	}

	/* -------------------- */
	/* argument 5 - flags */
	if (ar_flags == Py_None)
	{
	    al_flags = 0;		/* omitted */
	}
	else
	{
	    if (!PyInt_Check(ar_flags))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 5: flags - must be integer or None");
		return NULL;
	    }

	    l_flags  = PyInt_AsLong(ar_flags);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_flags = &l_flags;
	}

	/* -------------------- */
	/* set up string descriptor */
	/*  let LIB$FIND_FILE allocate the memory */
	r_resultant_filespec.dsc$w_length  = 0;
	r_resultant_filespec.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_resultant_filespec.dsc$b_class   = DSC$K_CLASS_D; /* not _S !! */
	r_resultant_filespec.dsc$a_pointer = 0;

	/* -------------------- */
	l_status_lib = lib$find_file
		(&r_filespec
		,&r_resultant_filespec
		,&l_context
		,ar_default_filespec	/* optional */
		,ar_related_filespec	/* optional */
		,&l_status_value
		,al_flags		/* optional */
		);

	/* -------------------- */
	if (l_status_lib == RMS$_NORMAL)
	{
	    /* this routine does not return an empty string, but 'None' */
	    /*  when the descriptor is 'empty' */
	    ar_resfilspe = Py_BuildValue
		("s#"
		,r_resultant_filespec.dsc$a_pointer
		,(unsigned int)r_resultant_filespec.dsc$w_length);
	}
	else
	{
	    ar_resfilspe = Py_None;
	    Py_INCREF(ar_resfilspe);
	}

	/* -------------------- */
	/* deallocate memory of dynamic string descriptor */
	l_status_free = lib$sfree1_dd (&r_resultant_filespec);
	if (l_status_free != SS$_NORMAL)
	{
	    PyErr_SetString(PyExc_SystemError,
		"vms_lib_find_file: LIB$SFREE1_DD() failed");
	    (void) PyErr_Print();

	    if (l_status_lib == SS$_NORMAL)
	    {
		l_status_lib = l_status_free;	/* fail anyway */
	    }
	}

	/* -------------------- */
	if (ar_resfilspe == NULL)
	{
	    return NULL;	/* Py_BuildValue() failed */
	}

	ar_return = Py_BuildValue ("lllO",
	    l_status_lib, l_status_value, l_context, ar_resfilspe);
	Py_XDECREF(ar_resfilspe);

	return ar_return;

} /* vms_lib_find_file () */

/* ------------------------------------------------------------------------- */

char vms_lib_find_file_end__doc[] =
"vms_lib.find_file_end (context)\n\
End of Find File.";

/* ------------------------------------------------------------------------- */

PyObject *
vms_lib_find_file_end (PyObject *self, PyObject *args)
{
	unsigned long		  l_context;

	unsigned long		  l_status_lib;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_context))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status_lib = lib$find_file_end (&l_context);

	/* -------------------- */
	/* the documentation of OpenVMS VAX V6.1 seems wrong - */
	/*  lib$find_file_end returns SS$_NORMAL instead RMS$_NORMAL */
	if ((l_status_lib == RMS$_NORMAL) ||
	    (l_status_lib == SS$_NORMAL)    )
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_lib_gr_error, 1, l_status_lib);
} /* vms_lib_find_file_end () */

/* ------------------------------------------------------------------------- */
/*
--  memory leak tests  --

-----

# -- check 1
import vms_lib
while (1):
  context = 0
  status = 1
  while (status & 1):
    status, status_value, context, resultant_filespec = \
      vms_lib.find_file ('SYS$SYSTEM:*.*', context, \
      None, None, 0) # 0 = allow wildcards
    # print resultant_filespec
  # -while
# -while

-----

# -- check 2
import vms_lib
while (1):
  context = 0
  status = 1
  while (status & 1):
    status, status_value, context, resultant_filespec = \
      vms_lib.find_file ('SYS$SYSTEM:*.*', context, \
      None, None, 1) # 1 = disallow wildcards -> error
    if (not (status & 1)):            # drop context - or this could
      vms_lib.find_file_end (context) #  lead to a false memory leak
  # -while
# -while

-----

*/

/* ------------------------------------------------------------------------- */

/* EOF: VMS_LIB_FIND_FILE.C */
