/* VMS_SYS.C -- 30-OCT-1996 Uwe Zessin
   interface to (some) OpenVMS SYS$ routines

   02-NOV-1997 ...
   04-NOV-1997 ZE. -- convert functions to use PyArg_ParseTuple() instead of
			PyArg_Parse().
   29-NOV-1997 ZE. -- changes for Python V1.5B1
   12-DEC-1997 ZE. -- add DELPRC, ASCUTC, BINUTC, GETUTC + NUMUTC.
			change CANWAK + FORCEX to return target pid.
   14-DEC-1997 ZE. -- add TIMCON
   17-DEC-1997 ZE. -- change RESUME to return target pid.
   18-DEC-1997 ZE. -- change WAKE to return target pid.
   19-DEC-1997 ZE. -- add SUSPND, change SETPRI to return target pid.
   21-DEC-1997 ZE. -- Still no prototype for sys$setddir in DEC C V5.6
			Provide a default one to stop compiler complaining.
   09-FEB-1998 ZE. -- add remaining DOC strings
   12-AUG-1998 ZE. -- conversion of 64-bit date+time 'hex-string'
			to Python long integer
   29-AUG-1998 ZE. -- add DELLNM
   30-AUG-1998 ZE. -- add FINISH_RDB, FIND_HELD, FIND_HOLDER, IDTOASC
   31-AUG-1998 ZE. -- add ADD_IDENT
   02-SEP-1998 ZE. -- add ADD_HOLDER, GRANTID
   03-SEP-1998 ZE. -- add MOD_HOLDER, MOD_IDENT, REM_HOLDER, REM_IDENT, REVOKID
   06-SEP-1998 ZE. -- begin GETUAI -code is in VMS_SYS_GETUAI.C
   14-SEP-1998 ZE. -- add CLREF, READEF, SETEF
   15-SEP-1998 ZE. -- add ASCEFC, DACEFC, DLCEFC, WAITFR, WFLAND, WFLOR
   16-SEP-1998 ZE. -- begin SETUAI -code is in VMS_SYS_SETUAI.C
   06-OCT-1998 ZE. -- begin DEVICE_SCAN -code is in VMS_SYS_DEVICE_SCAN.C
   12-OCT-1998 ZE. -- begin CREPRC -code is in VMS_SYS_CREPRC.C
   16-OCT-1998 ZE. -- begin CRELNT -code is in VMS_SYS_CRELNT.C
   17-OCT-1998 ZE. -- add FILESCAN -code is in VMS_SYS_FILESCAN.C
		      add GETMSG -code is in VMS_SYS_GETMSG.C
   22-NOV-1998 ZE. -- begin CRELNM -code is in VMS_SYS_CRELNM.C
   24-NOV-1998 ZE. -- begin PROCESS_SCAN - code is in VMS_SYS_PROCESS_SCAN.C
   27-NOV-1998 ZE. -- add SETPRV -code is in VMS_SYS_SETPRV.C
   28-NOV-1998 ZE. -- add TRNLNM -code is in VMS_SYS_TRNLNM.C
   29-NOV-1998 ZE. -- add DASSGN -code is in VMS_SYS_DASSGN.C
   27-DEC-1998 ZE. -- add error checks in many routines
   01-JAN-1999 ZE. -- add SETDFPROT -code is in VMS_SYS_SETDFPROT.C
   09-JAN-1999 ZE. -- conversion of 128-bit octaword 'hex-string'
			to Python long integer
   12-JAN-1999 ZE. -- add DISMOU -code is in VMS_SYS_DISMOU.C
   13-JAN-1999 ZE. -- add MOUNT -code is in VMS_SYS_MOUNT.C
   21-JAN-1999 ZE. -- add ADD_PROXY, DELETE_PROXY, DISPLAY_PROXY, VERIFY_PROXY
   22-JAN-1999 ZE. -- add INIT_VOL
   04-FEB-1999 ZE. -- updates for Alpha after long developement phase on VAX
   05-FEB-1999 ZE. -- add GETJPIW
   11-FEB-1999 ZE. -- add OPEN, CLOSE, CONNECT, DISCONNECT, GET (= RMS)
   26-FEB-1999 ZE. -- add CREATE, DELETE, DISPLAY, ENTER, ERASE, EXTEND, FIND,
			FLUSH, FREE, NXTVOL, PARSE, PUT, READ, RELEASE, REMOVE,
			RENAME, REWIND, SEARCH, SPACE, TRUNCATE, UPDATE, WAIT,
			WRITE (= RMS)
   17-MAR-1999 ZE. -- add SNDJBCW
   03-APR-1999 ZE. -- add GETQUIW
   11-MAY-1999 ZE. -- FB- fix parameters to PyArg_ParseTuple() in
			vms_sys_ascefc() - missing &
   29-MAY-1999 ZE. -- add ENQW, DEQ
   31-MAY-1999 ZE. -- add DELETE_INTRUSION, SCAN_INTRUSION
   01-JUN-1999 ZE. -- add SHOW_INTRUSION
   27-JUN-1999 ZE. -- vms_sys_methods[] is now configured via data file
		      VMS_SYS__METHODS.DAT - see 'General Manual', configuration
   15-JUL-1999 ZE. -- FB- fix assignment in vms_sys_ascefc() - not '=='
		   -- add SET_RESOURCE_DOMAIN
		   -- move DOC-strings into external files
*/


#include "python.h"
#include "vmsdef.h"

#include <descrip.h>
#include <rmsdef.h>		/* RMS$_name	 */
#include <ssdef.h>		/* SS$_name	 */
#include <starlet.h>		/* SYS$name, ... */

#ifndef __ALPHA		/* sys$setddir defined in Alpha/DEC C V5.2 */
#ifdef __DECC
#ifndef __DECC_VER	/* don't know if this is defined in all versions ... */
#define __DECC_VER 0
#endif /* __DECC_VER */
#if __DECC_VER < 50600000 /* @@ might need to change for newer version */
#ifndef __unknown_params
#define __unknown_params ...
#endif
#pragma __nostandard	  /* non-ANSI-Standard feature */
extern unsigned long sys$setddir(__unknown_params);
#pragma __standard
#endif /* __DECC_VER < 50600000 */
#endif /* __DECC */
#endif /* __ALPHA */
/* ------------------------------------------------------------------------- */

PyObject *vms_sys_gr_error;		/* exception vms_sys.error */

static char vms_sys_doc__ [] =
"This is an interface to some OpenVMS SYS$ routines.\n\
See the VMS_SYS documentation for details.";

static char vms_sys_t_cant_init   [] = "can't initialize module vms_sys";
static char vms_sys_t_cant_deferr [] = "can't define vms_sys.error";

/* ------------------------------------------------------------------------- */
/* function prototypes for external routines */

/* ------------------------------ */
/* input a binary quadword - output a Python long integer */
PyObject * vms__cvt_quad2pylong (long q_quadword[2]);

/* ------------------------------ */
/* input a Python long integer - output a binary quadword */
long vms__cvt_pylong2quad (PyObject * r_longint, long * q_quadword);

/* ------------------------------ */
/* input a binary octaword, output a Python long integer */
extern PyObject * vms__cvt_octa2pylong (long q_octaword[4]);

/* ------------------------------ */
/* input a Python long integer, output a binary octaword */
extern long vms__cvt_pylong2octa (PyObject * r_longint, long * o_octaword);

/* ------------------------------------------------------------------------- */

char vms_sys_add_holder__doc[] =
"resid = vms_sys.add_holder (id, holder [,attrib])\n\
Add Holder Record to Rights Database.";

PyObject *
vms_sys_add_holder (PyObject *self, PyObject *args)
{
	unsigned long		  l_id;

	/*unsigned*/ long	  q_holder[2];
	PyObject		* ar_holder;

	unsigned long		  l_attrib;
	PyObject		* ar_attrib;

	unsigned long		  l_status;

	/* -------------------- */
	ar_attrib = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "iO|O",
		&l_id, &ar_holder, &ar_attrib))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 2: holder */
	if (!PyTuple_Check(ar_holder))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 2: holder - must be a tuple of 2 integers");
	    return NULL;
	}
	else
	{
	    unsigned long	  l_tuple_size;
	    PyObject		* ar_tuple_element;

	    /* -------------------- */
	    l_tuple_size = PyTuple_Size(ar_holder);
	    if (l_tuple_size != 2)
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - must be a tuple of 2 integers");
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 0);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - tuple-element:0 is not an integer");
		return NULL;
	    }
	    q_holder [0] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 1);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - tuple-element:1 is not an integer");
		return NULL;
	    }
	    q_holder [1] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    /* -------------------- */
	}

	/* -------------------- */
	if (ar_attrib == Py_None)
	{
	    l_attrib = 0;	/* omitted, however passed BY VALUE */
	}
	else
	{
	    if (!PyInt_Check(ar_attrib))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: attrib - must be integer or None");
		return NULL;
	    }
	    l_attrib = PyInt_AsLong(ar_attrib);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$add_holder
		(l_id			/* [id]     -- by value !! */
		,&q_holder[0]		/* [holder] */
		,l_attrib		/* [attrib] -- by value !! */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_add_holder () */

/* ------------------------------------------------------------------------- */

char vms_sys_add_ident__doc[] =
"resid = vms_sys.add_ident (name [,id] [,attrib])\n\
Add Identifier to Rights Database.";

PyObject *
vms_sys_add_ident (PyObject *self, PyObject *args)
{
	char			* at_name;
	struct dsc$descriptor_s   r_name;
	unsigned long		  l_name_len;

	unsigned long		  l_id;
	PyObject		* ar_id;

	unsigned long		  l_attrib;
	PyObject		* ar_attrib;

	unsigned long		  l_resid;

	unsigned long		  l_status;

	/* -------------------- */
	ar_id     = Py_None;
	ar_attrib = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#|OO",
	    &at_name, &l_name_len,
	    &ar_id,
	    &ar_attrib))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: name */
	if (l_name_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
		"argument 1: name - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up descriptor */
	r_name.dsc$w_length  = l_name_len;
	r_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_name.dsc$b_class   = DSC$K_CLASS_S;
	r_name.dsc$a_pointer = at_name;

	/* -------------------- */
	/* argument 2: id */
	if (ar_id == Py_None)
	{
	    l_id = 0;		/* omitted, however passed BY VALUE */
	}
	else
	{
	    if (!PyInt_Check(ar_id))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: id - must be integer or None");
		return NULL;
	    }
	    l_id = PyInt_AsLong(ar_id);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 3: attrib */
	if (ar_attrib == Py_None)
	{
	    l_attrib = 0;	/* omitted, however passed BY VALUE */
	}
	else
	{
	    if (!PyInt_Check(ar_attrib))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: attrib - must be integer or None");
		return NULL;
	    }
	    l_attrib = PyInt_AsLong(ar_attrib);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$add_ident
		(&r_name		/* name     */
		,l_id			/* [id]     -- by value !! */
		,l_attrib		/* [attrib] -- by value !! */
		,&l_resid		/* [resid]  */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return PyInt_FromLong(l_resid);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_add_ident () */

/* ------------------------------------------------------------------------- */

char vms_sys_ascefc__doc[] =
"vms_sys.ascefc (efn, name [,prot] [,perm])\n\
Associate Common Event Flag Cluster.";

PyObject *
vms_sys_ascefc (PyObject *self, PyObject *args)
{
	unsigned long		  l_efn;

	struct dsc$descriptor_s   r_name;
	char			* at_name;
	unsigned long		  l_name_len;

	PyObject		* ar_prot;
	unsigned long		  l_prot; /* uB */

	PyObject		* ar_perm;
	unsigned long		  l_perm; /* uB */

	unsigned long		  l_status;

	/* -------------------- */
	ar_prot = Py_None;
	ar_perm = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "is#|OO",
	    &l_efn,
	    &at_name, &l_name_len,
	    &ar_prot,
	    &ar_perm))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 2: name */
	if (l_name_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
		"argument 2: name - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up descriptor */
	r_name.dsc$w_length  = l_name_len;
	r_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_name.dsc$b_class   = DSC$K_CLASS_S;
	r_name.dsc$a_pointer = at_name;

	/* -------------------- */
	/* argument 3: prot */
	if (ar_prot == Py_None)
	{
	    /* turn 'None' into '0' because argument is passed by value */
	    l_prot = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_prot))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: prot - must be integer or None");
		return NULL;
	    }
	    l_prot = PyInt_AsLong(ar_prot);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 4: perm */
	if (ar_perm == Py_None)
	{
	    /* turn 'None' into '0' because argument is passed by value */
	    l_perm = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_perm))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 4: perm - must be integer or None");
		return NULL;
	    }
	    l_perm = PyInt_AsLong(ar_perm);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$ascefc
			(l_efn		/* efn    -- by value */
			,&r_name	/* name   */
			,l_prot		/* [prot] -- by value */
			,l_perm		/* [perm] -- by value */
			);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_ascefc () */

/* ------------------------------------------------------------------------- */

char vms_sys_asctim__doc[] =
"timbuf = vms_sys.asctim ([timadr] [,cvtflg])\n\
Convert Binary Time to ASCII String.";

PyObject *
vms_sys_asctim (PyObject *self, PyObject *args)
{
	PyObject		* ar_timadr;
	long			  q_timadr[2];		/* binary system time */
	long			* aq_timadr;

#define S_TIMBUF 23
	char			t_timbuf[S_TIMBUF];	/* ASCII time */
	struct dsc$descriptor_s r_timbuf;
	unsigned short int	w_timbuf_len;

	unsigned long		l_cvtflg;	/* conversion flg, $ASCTIM */

	unsigned long		l_status;

	/* -------------------- */
	ar_timadr = Py_None;
	l_cvtflg  = 0;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|Ol", &ar_timadr, &l_cvtflg))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: timadr */
	if (ar_timadr == Py_None)
	{
	    /* no arguments given, indicate current date + time */
	    /* to comply with SYS$ASCTIM */
	    aq_timadr = 0;
	}
	else
	{
	    if (!PyLong_Check(ar_timadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: timadr - must be long integer");
		return NULL;
	    }
	    /* convert Python long integer to binary quadword */
	    l_status = vms__cvt_pylong2quad (ar_timadr, &q_timadr[0]);
	    if (l_status == -1)
	    {
		return NULL;		/* error string has been set */
	    }
	    aq_timadr = &q_timadr[0];
	}

	/* -------------------- */
	/* argument 2: flags*/
	/* automatically parsed / defaulted */

	/* -------------------- */
	/* set up descriptor */
	r_timbuf.dsc$w_length  = S_TIMBUF;
	r_timbuf.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_timbuf.dsc$b_class   = DSC$K_CLASS_S;
	r_timbuf.dsc$a_pointer = &t_timbuf[0];

	/* -------------------- */
	l_status = sys$asctim
		(&w_timbuf_len		/* [timlen] */
		,&r_timbuf		/* timbuf   */
		,aq_timadr		/* [timadr] */
		,l_cvtflg		/* [cvtflg] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return Py_BuildValue
		("s#", &t_timbuf[0], (unsigned int)w_timbuf_len);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_asctim () */

/* ------------------------------------------------------------------------- */

char vms_sys_asctoid__doc[] =
"id, attrib = vms_sys.asctoid (name)\n\
Translate Identifier Name to Identifier.";

PyObject *
vms_sys_asctoid (PyObject *self, PyObject *args)
{
	struct dsc$descriptor_s   r_name;
	char			* at_name;
	unsigned long		  l_name_len;

	unsigned long		  l_id;
	unsigned long		  l_attrib;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#", &at_name, &l_name_len))
	{
	    return NULL;
	}

	/* -------------------- */
	if (l_name_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
		"argument 1: name - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up string descriptor */
	r_name.dsc$w_length  = l_name_len;
	r_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_name.dsc$b_class   = DSC$K_CLASS_S;
	r_name.dsc$a_pointer = at_name;

	/* -------------------- */
	l_status = sys$asctoid
		(&r_name		/* name     */
		,&l_id			/* [id]     */
		,&l_attrib		/* [attrib] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return Py_BuildValue ("ll", l_id, l_attrib);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_asctoid () */

/* ------------------------------------------------------------------------- */

char vms_sys_ascutc__doc[] =
"Convert UTC to ASCII\n\
timbuf = vms_sys.ascutc (utcadr [,cvtflg]).";

PyObject *
vms_sys_ascutc (PyObject *self, PyObject *args)
{
	PyObject	* ar_utcadr;
	long		  o_utcadr[4];		/* 128-bit utc_date_time */
	long		* ao_utcadr;

#ifndef S_TIMBUF
#define S_TIMBUF 23
#endif
	char			t_timbuf[S_TIMBUF];	/* ASCII time */
	struct dsc$descriptor_s r_timbuf;
	unsigned short int	w_timbuf_len;

	unsigned long		l_cvtflg;	/* conversion flag, $ASCUTC */

	unsigned long		l_status;

	/* -------------------- */
	ar_utcadr = Py_None;
	l_cvtflg  = 0;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|Ol", &ar_utcadr, &l_cvtflg))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: utcadr */
	if (ar_utcadr == Py_None)
	{
	    ao_utcadr = 0;
	}
	else
	{
	    if (!PyLong_Check(ar_utcadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: utcadr - must be long integer");
		return NULL;
	    }
	    /* convert Python long integer to binary octaword */
	    l_status = vms__cvt_pylong2octa (ar_utcadr, &o_utcadr[0]);
	    if (l_status != SS$_NORMAL)
	    {
		PyErr_SetString(PyExc_ValueError,
		   "argument 1: utcadr - conversion from long integer failed");
		return NULL;
	    }
	    ao_utcadr = &o_utcadr[0];
	}

	/* -------------------- */
	/* argument 2: cvtflg */
	/* already parsed or defaulted */

	/* -------------------- */
	/* set up descriptor */
	r_timbuf.dsc$w_length  = S_TIMBUF;
	r_timbuf.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_timbuf.dsc$b_class   = DSC$K_CLASS_S;
	r_timbuf.dsc$a_pointer = &t_timbuf[0];

	/* -------------------- */
	l_status = sys$ascutc
		(&w_timbuf_len		/* [timlen] */
		,&r_timbuf		/* timbuf   */
		,ao_utcadr		/* [utcadr] */
		,l_cvtflg		/* [cvtflg] - by value */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* return ASCII time to Python */
	    return Py_BuildValue
		("s#", &t_timbuf[0], (unsigned int)w_timbuf_len);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_ascutc () */

/* ------------------------------------------------------------------------- */

char vms_sys_bintim__doc[] =
"timadr = vms_sys.bintim (timbuf)\n\
Convert ASCII String to Binary Time.";

PyObject *
vms_sys_bintim (PyObject *self, PyObject *args)
{
	char			* at_timbuf;	/* ASCII time */
	struct dsc$descriptor_s r_timbuf;
	unsigned long		l_timbuf_len;

	long			q_timadr[2];	/* binary system time */

	unsigned long		l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#", &at_timbuf, &l_timbuf_len))
	{
	    return NULL;
	}

	/* -------------------- */
	if (l_timbuf_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	       "argument 1: timbuf - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up descriptor */
	r_timbuf.dsc$w_length  = l_timbuf_len;
	r_timbuf.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_timbuf.dsc$b_class   = DSC$K_CLASS_S;
	r_timbuf.dsc$a_pointer = at_timbuf;

	/* -------------------- */
	l_status = sys$bintim
		(&r_timbuf		/* timbuf */
		,&q_timadr		/* timadr */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* convert binary quadword to Python long integer */
	    return vms__cvt_quad2pylong (&q_timadr[0]);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_bintim () */

/* ------------------------------------------------------------------------- */

char vms_sys_binutc__doc[] =
"utcadr = vms_sys.binutc (timbuf)\n\
Convert ASCII String to UTC Binary Time.";

PyObject *
vms_sys_binutc (PyObject *self, PyObject *args)
{
	char			* at_timbuf;	/* ASCII time */
	struct dsc$descriptor_s   r_timbuf;
	unsigned long		  l_timbuf_len;

	long			  o_utcadr[4];	/* utc_date_time */

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#", &at_timbuf, &l_timbuf_len))
	{
	    return NULL;
	}

	/* -------------------- */
	if (l_timbuf_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	       "argument 1: timbuf - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up descriptor */
	r_timbuf.dsc$w_length  = l_timbuf_len;
	r_timbuf.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_timbuf.dsc$b_class   = DSC$K_CLASS_S;
	r_timbuf.dsc$a_pointer = at_timbuf;

	/* -------------------- */
	l_status = sys$binutc
		(&r_timbuf		/* timbuf */
		,&o_utcadr		/* utcadr */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* convert binary octaword to Python long integer */
	    return vms__cvt_octa2pylong (&o_utcadr[0]);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_binutc () */

/* ------------------------------------------------------------------------- */

char vms_sys_canwak__doc[] =
"targpid = vms_sys.canwak ([pidadr] [,prcnam])\n\
Cancel Wakeup.";

PyObject *
vms_sys_canwak (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_status;

	/* -------------------- */
	ar_pidadr = Py_None;
	at_prcnam = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|Oz#",
	    &ar_pidadr,
	    &at_prcnam, &l_prcnam_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	    /* turn 'None' into '0' because argument 2 might be present */
	    /*   and this routine has to return a PID */
	    l_pidadr = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	l_status = sys$canwak (al_pidadr, ar_prcnam);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* always return PID of target process */
	    return PyInt_FromLong(l_pidadr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_canwak () */

/* ------------------------------------------------------------------------- */

char vms_sys_clref__doc[] =
"setflg = vms_sys.clref (efn)\n\
Clear Event Flag.";

PyObject *
vms_sys_clref (PyObject *self, PyObject *args)
{
	unsigned long		  l_efn;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_efn))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$clref (l_efn);	/* efn -- by value */

	/* -------------------- */
	if ( (l_status == SS$_WASCLR) || (l_status == SS$_WASSET) )
	{
	    int l_setflg;
	    l_setflg = (l_status == SS$_WASCLR) ? 0 : 1;
	    return PyInt_FromLong(l_setflg);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_clref () */

/* ------------------------------------------------------------------------- */

char vms_sys_dacefc__doc[] =
"vms_sys.dacefc (efn)\n\
Disassociate Common Event Flag Cluster.";

PyObject *
vms_sys_dacefc (PyObject *self, PyObject *args)
{
	unsigned long		  l_efn;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_efn))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$dacefc (l_efn);	/* efn -- by value */

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_dacefc () */

/* ------------------------------------------------------------------------- */

char vms_sys_dellnm__doc[] =
"vms_sys.dellnm (tabnam, [lognam], [acmode])\n\
Delete Logical Name.";

PyObject *
vms_sys_dellnm (PyObject *self, PyObject *args)
{
	char			* at_tabnam;
	struct dsc$descriptor_s   r_tabnam;
	unsigned long		  l_tabnam_len;

	char			* at_lognam;
	struct dsc$descriptor_s   r_lognam;
	struct dsc$descriptor_s * ar_lognam;
	unsigned long		  l_lognam_len;

	unsigned long		  l_acmode;
	unsigned long		* al_acmode;
	PyObject		* ar_acmode;

	unsigned long		  l_status;

	/* -------------------- */
	at_lognam = NULL;
	ar_acmode = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#|z#O",
	    &at_tabnam, &l_tabnam_len,
	    &at_lognam, &l_lognam_len,
	    &ar_acmode))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: tabnam */
	if (l_tabnam_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
	       "argument 1: tabnam - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up descriptor */
	r_tabnam.dsc$w_length  = l_tabnam_len;
	r_tabnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_tabnam.dsc$b_class   = DSC$K_CLASS_S;
	r_tabnam.dsc$a_pointer = at_tabnam;

	/* -------------------- */
	/* argument 2: lognam */
	if (at_lognam == NULL)
	{
	    ar_lognam = 0;		/* ommitted */
	}
	else
	{
	    if (l_lognam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 1: lognam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_lognam.dsc$w_length  = l_lognam_len;
	    r_lognam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_lognam.dsc$b_class   = DSC$K_CLASS_S;
	    r_lognam.dsc$a_pointer = at_lognam;
	    ar_lognam = &r_lognam;
	}

	/* -------------------- */
	/* argument 3: acmode */
	if (ar_acmode == Py_None)
	{
	    al_acmode = 0;		/* omitted */
	}
	else
	{
	    if (!PyInt_Check(ar_acmode))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: acmode - must be integer or None");
		return NULL;
	    }
	    l_acmode  = PyInt_AsLong(ar_acmode);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_acmode = &l_acmode;
	}

	/* -------------------- */
	l_status = sys$dellnm (&r_tabnam, ar_lognam, al_acmode);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_dellnm () */

/* ------------------------------------------------------------------------- */

char vms_sys_delprc__doc[] =
"targpid = vms_sys.delprc ([pidadr] [,prcnam])\n\
Delete Process.";

PyObject *
vms_sys_delprc (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s * ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_status;

	/* -------------------- */
	ar_pidadr = Py_None;
	at_prcnam = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|Oz#",
	    &ar_pidadr,
	    &at_prcnam, &l_prcnam_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	    /* turn 'None' into '0' because argument 2 might be present */
	    /*   and this routine has to return a PID */
	    l_pidadr = 0;
	    /* empty or None without argument 2 indicates deletion of */
	    /* the current process */
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	l_status = sys$delprc (al_pidadr, ar_prcnam);

	if (l_status == SS$_NORMAL)
	{
	    /* always return PID of target process */
	    return PyInt_FromLong(l_pidadr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_delprc () */

/* ------------------------------------------------------------------------- */

char vms_sys_dlcefc__doc[] =
"vms_sys.dlcefc (name)\n\
Delete Common Event Flag Cluster.";

PyObject *
vms_sys_dlcefc (PyObject *self, PyObject *args)
{
	struct dsc$descriptor_s   r_name;
	char			* at_name;
	unsigned long		  l_name_len;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "s#", &at_name, &l_name_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1 - name */
	if (l_name_len > 65535)
	{
	    PyErr_SetString(PyExc_ValueError,
		"argument 1: name - string size limited to 65535 characters");
	    return NULL;
	}
	/* set up descriptor */
	r_name.dsc$w_length  = l_name_len;
	r_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_name.dsc$b_class   = DSC$K_CLASS_S;
	r_name.dsc$a_pointer = at_name;

	/* -------------------- */
	l_status = sys$dlcefc (&r_name);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_dlcefc () */

/* ------------------------------------------------------------------------- */

char vms_sys_find_held__doc[] =
"id, attrib, fnd_ctx = vms_sys.find_held (holder [,contxt])\n\
Find Identifiers Held by User.";

PyObject *
vms_sys_find_held (PyObject *self, PyObject *args)
{
	unsigned long		  q_holder[2];
	PyObject		* ar_holder;

	unsigned long		  l_contxt;
	unsigned long		* al_contxt;
	PyObject		* ar_contxt;

	unsigned long		  l_id;

	unsigned long		  l_attrib;

	unsigned long		  l_status;

	/* -------------------- */
	ar_contxt = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O|O", &ar_holder, &ar_contxt))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: holder */
	if (!PyTuple_Check(ar_holder))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 1: holder - must be a tuple of 2 integers");
	    return NULL;
	}
	else
	{
	    unsigned long	  l_tuple_size;
	    PyObject		* ar_tuple_element;

	    /* -------------------- */
	    l_tuple_size = PyTuple_Size(ar_holder);
	    if (l_tuple_size != 2)
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: holder - must be a tuple of 2 integers");
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 0);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: holder - tuple-element:0 is not an integer");
		return NULL;
	    }
	    q_holder [0] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 1);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: holder - tuple-element:1 is not an integer");
		return NULL;
	    }
	    q_holder [1] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 2: contxt */
	if (ar_contxt == Py_None)
	{
	    l_contxt  = 0;	/* will be returned anyway ! */
	    al_contxt = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_contxt))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: contxt - must be integer or None");
		return NULL;
	    }
	    l_contxt  = PyInt_AsLong(ar_contxt);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_contxt = &l_contxt;
	}

	/* -------------------- */
	l_status = sys$find_held
		(&q_holder[0]		/* holder    */
		,&l_id			/* [id]      */
		,&l_attrib		/* [attrib]  */
		,al_contxt		/* [contxt] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return Py_BuildValue ("lll", l_id, l_attrib, l_contxt);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_find_held () */

/* ------------------------------------------------------------------------- */

char vms_sys_find_holder__doc[] =
"(holder, 0), attrib, fnd_ctx = vms_sys.find_holder (id [,contxt])\n\
Find Holder of Identifier.";

PyObject *
vms_sys_find_holder (PyObject *self, PyObject *args)
{
	unsigned long		  l_id;

	unsigned long		  l_contxt;
	unsigned long		* al_contxt;
	PyObject		* ar_contxt;

	unsigned long		  q_holder[2];

	unsigned long		  l_attrib;

	unsigned long		  l_status;

	/* -------------------- */
	ar_contxt = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l|O", &l_id, &ar_contxt))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 2: contxt */
	if (ar_contxt == Py_None)
	{
	    l_contxt  = 0;	/* will be returned anyway ! */
	    al_contxt = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_contxt))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: contxt - must be integer or None");
		return NULL;
	    }
	    l_contxt  = PyInt_AsLong(ar_contxt);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_contxt = &l_contxt;
	}

	/* -------------------- */
	l_status = sys$find_holder
		(l_id			/* id       -- by value !! */
		,&q_holder[0]		/* [holder] */
		,&l_attrib		/* [attrib] */
		,al_contxt		/* [contxt] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return Py_BuildValue ("(ll)ll",
		q_holder[0], q_holder[1], l_attrib, l_contxt);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_find_holder () */

/* ------------------------------------------------------------------------- */

char vms_sys_finish_rdb__doc[] =
"new_ctx = vms_sys.finish_rdb (contxt)\n\
Terminate Rights Database Context.";

PyObject *
vms_sys_finish_rdb (PyObject *self, PyObject *args)
{
	unsigned long		  l_contxt;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_contxt))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$finish_rdb (&l_contxt);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    return PyInt_FromLong(l_contxt);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_finish_rdb () */

/* ------------------------------------------------------------------------- */

char vms_sys_forcex__doc[] =
"targpid = vms_sys.forcex ([pidadr] [,prcnam] [,code])\n\
Force Exit.";

PyObject *
vms_sys_forcex (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_code;
	PyObject		* ar_code;

	unsigned long		  l_status;

	/* -------------------- */
	ar_pidadr = Py_None;
	at_prcnam = NULL;
	ar_code   = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|Oz#O",
	    &ar_pidadr,
	    &at_prcnam, &l_prcnam_len,
	    &ar_code))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	    /* turn 'None' into '0' because argument 2 might be present */
	    /*   and this routine has to return a PID */
	    l_pidadr = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	/* argument 3: code */
	if (ar_code == Py_None)
	{
	    l_code = 0;		/* omitted, BY VALUE */
	}
	else
	{
	    if (!PyInt_Check(ar_code))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: code - must be integer or None");
		return NULL;
	    }
	    l_code = PyInt_AsLong(ar_code);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$forcex
		(al_pidadr		/* [pidadr] */
		,ar_prcnam		/* [prcnam] */
		,l_code			/* [code]   */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* always return PID of target process */
	    return PyInt_FromLong(l_pidadr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_forcex () */

/* ------------------------------------------------------------------------- */

char vms_sys_gettim__doc[] =
"timadr = vms_sys.gettim ()\n\
Returns the current system time in a 64-bit format.";

PyObject *
vms_sys_gettim (PyObject *self, PyObject *args)
{
	long		q_timadr[2];		/* binary system time */
	unsigned long	l_status;

	/* -------------------- */
	/* no arguments taken */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$gettim (&q_timadr[0]);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* convert binary quadword to Python long integer */
	    return vms__cvt_quad2pylong (&q_timadr[0]);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_gettim () */

/* ------------------------------------------------------------------------- */

char vms_sys_getutc__doc[] =
"utcadr = vms_sys.getutc ()\n\
Get UTC Time.";

PyObject *
vms_sys_getutc (PyObject *self, PyObject *args)
{
	long		o_timadr[4];		/* utc_date_time */

	unsigned long	l_status;

	/* -------------------- */
	/* no arguments taken */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$getutc (&o_timadr[0]);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* convert binary octaword to Python long integer */
	    return vms__cvt_octa2pylong (&o_timadr[0]);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_getutc () */

/* ------------------------------------------------------------------------- */

char vms_sys_grantid__doc[] =
"setflg, targpid, id, prvatr = vms_sys.grantid ([pidadr] [,prcnam] [,id] [,name])\n\
Grant Identifier to Process.";

PyObject *
vms_sys_grantid (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  q_id[2];
	unsigned long		* aq_id;
	PyObject		* ar_id;

	char			* at_name;
	struct dsc$descriptor_s   r_name;
	struct dsc$descriptor_s	* ar_name;
	unsigned long		  l_name_len;

	unsigned long		  l_prvatr;

	unsigned long		  l_status;

	/* -------------------- */
	at_prcnam = NULL;
	ar_id     = Py_None;
	at_name   = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O|z#Oz#",
		&ar_pidadr,
		&at_prcnam, &l_prcnam_len,
		&ar_id,
		&at_name,   &l_name_len
		))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	    /* turn 'None' into '0' because argument 2 might be present */
	    /*   and this routine has to return a PID */
	    l_pidadr = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2 - prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	/* argument 3 - id - a tuple of 2 integers */
	if (ar_id == Py_None)
	{
	    q_id[0] = 0;	/* arg 3 = omitted */
	    q_id[1] = 0;
	}
	else
	{
	  if (!PyTuple_Check(ar_id))
	  {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: id - must be a tuple of 2 integers");
		return NULL;
	  }
	  else
	  {
	    unsigned long	  l_tuple_size;
	    PyObject		* ar_tuple_element;

	    /* -------------------- */
	    l_tuple_size = PyTuple_Size(ar_id);
	    if (l_tuple_size != 2)
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: id - must be a tuple of 2 integers");
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_id, 0);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: id - tuple-element:0 is not an integer");
		return NULL;
	    }
	    q_id[0] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_id, 1);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: id - tuple-element:1 is not an integer");
		return NULL;
	    }
	    q_id[1] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    /* -------------------- */
	  }
	}
	aq_id = &q_id[0];

	/* -------------------- */
	/* argument 4 - name */
	if (at_name == NULL)
	{
	    ar_name = 0;		/* omitted */
	}
	else
	{
	    if (l_name_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
		 "argument 2: name - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_name.dsc$w_length  = l_name_len;
	    r_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_name.dsc$b_class   = DSC$K_CLASS_S;
	    r_name.dsc$a_pointer = at_name;
	    ar_name = &r_name;
	}

	/* -------------------- */
	l_prvatr = 0;

	/* -------------------- */
	l_status = sys$grantid
		(al_pidadr		/* [pidadr] */
		,ar_prcnam		/* [prcnam] */
		,aq_id			/* [id]     */
		,ar_name		/* [name]   */
		,&l_prvatr		/* [prvatr] */
		);

	/* -------------------- */
	if ((l_status == SS$_WASCLR) || (l_status == SS$_WASSET) )
	{
	    unsigned long l_setflg;

	    l_setflg = (l_status == SS$_WASCLR) ? 0 : 1;
	    /* return PID of target process */
	    return Py_BuildValue ("ll(ll)l",
		l_setflg, l_pidadr, q_id[0], q_id[1], l_prvatr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_grantid () */

/* ------------------------------------------------------------------------- */

char vms_sys_hiber__doc[] =
"vms_sys.hiber ()\n\
Hibernate.";

PyObject *
vms_sys_hiber (PyObject *self, PyObject *args)
{
	unsigned long	l_status;

	/* -------------------- */
	/* no arguments taken */
	if (!PyArg_ParseTuple(args, ""))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$hiber();

	/* -------------------- */
	if (l_status == SS$_NORMAL)	/* should always return it */
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_hiber () */

/* ------------------------------------------------------------------------- */

char vms_sys_idtoasc__doc[] =
"nambuf, resid, attrib, fnd_ctx = vms_sys.idtoasc (id [,contxt])\n\
Translate Identifier to Identifier Name.";

PyObject *
vms_sys_idtoasc (PyObject *self, PyObject *args)
{
	unsigned long		  l_id;

	PyObject		* ar_contxt;
	unsigned long		  l_contxt;
	unsigned long		* al_contxt;

#define S_NAMBUF 31	/* maximum identifier name length */
	char			  t_nambuf[S_NAMBUF];
	struct dsc$descriptor_s   r_nambuf;
	unsigned short int	  w_namlen;

	unsigned long		  l_resid;
	unsigned long		  l_attrib;

	unsigned long		  l_status;

	/* -------------------- */
	ar_contxt = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l|O", &l_id, &ar_contxt))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 2 - contxt */
	if (ar_contxt == Py_None)
	{
	    l_contxt  = 0;	/* will be returned anyway ! */
	    al_contxt = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_contxt))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: contxt - must be integer or None");
		return NULL;
	    }
	    l_contxt  = PyInt_AsLong(ar_contxt);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_contxt = &l_contxt;
	}

	/* -------------------- */
	/* set up descriptor */
	r_nambuf.dsc$w_length  = S_NAMBUF;
	r_nambuf.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_nambuf.dsc$b_class   = DSC$K_CLASS_S;
	r_nambuf.dsc$a_pointer = &t_nambuf[0];

	/* -------------------- */
	l_status = sys$idtoasc
		(l_id			/* id       -- by value !! */
		,&w_namlen		/* [namlen] */
		,&r_nambuf		/* [nambuf] */
		,&l_resid		/* [resid]  */
		,&l_attrib		/* [attrib] */
		,al_contxt		/* [contxt] */
		);

	/* -------------------- */
	/* update descriptor for debugging (EXAMINE/ASCID) */
	r_nambuf.dsc$w_length = w_namlen;

	if (l_status == SS$_NORMAL)
	{
	    return Py_BuildValue ("s#lll",
		&t_nambuf[0], (unsigned long)w_namlen,
		l_resid, l_attrib, l_contxt);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_idtoasc () */

/* ------------------------------------------------------------------------- */

char vms_sys_mod_holder__doc[] =
"vms_sys.mod_holder (id, holder, [,set_attrib] [,clr_attrib])\n\
Modify Holder Record in Rights Database";

PyObject *
vms_sys_mod_holder (PyObject *self, PyObject *args)
{
	unsigned long		  l_id;

	PyObject		* ar_holder;
	unsigned long		  q_holder[2];

	PyObject		* ar_set_attrib;
	unsigned long		  l_set_attrib;

	PyObject		* ar_clr_attrib;
	unsigned long		  l_clr_attrib;

	unsigned long		  l_status;

	/* -------------------- */
	ar_set_attrib = Py_None;
	ar_clr_attrib = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "lO|OO",
		&l_id, &ar_holder, &ar_set_attrib, &ar_clr_attrib))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 2 - holder - a tuple of 2 integers */
	if (!PyTuple_Check(ar_holder))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 2: holder - must be a tuple of 2 integers");
	    return NULL;
	}
	else
	{
	    unsigned long	  l_tuple_size;
	    PyObject		* ar_tuple_element;

	    /* -------------------- */
	    l_tuple_size = PyTuple_Size(ar_holder);
	    if (l_tuple_size != 2)
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - must be a tuple of 2 integers");
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 0);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - tuple-element:0 is not an integer");
		return NULL;
	    }
	    q_holder [0] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 1);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - tuple-element:1 is not an integer");
		return NULL;
	    }
	    q_holder [1] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 3 - set-attrib */
	if (ar_set_attrib == Py_None)
	{
	    l_set_attrib = 0;	/* by value */
	}
	else
	{
	    if (!PyInt_Check(ar_set_attrib))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: set-attrib - must be integer or None");
		return NULL;
	    }
	    l_set_attrib = PyInt_AsLong(ar_set_attrib);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 4 - clr-attrib */
	if (ar_clr_attrib == Py_None)
	{
	    l_clr_attrib = 0;	/* by value */
	}
	else
	{
	    if (!PyInt_Check(ar_clr_attrib))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 4: clr-attrib - must be integer or None");
		return NULL;
	    }
	    l_clr_attrib = PyInt_AsLong(ar_clr_attrib);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$mod_holder
		(l_id			/* id           -- by value !! */
		,&q_holder[0]		/* [holder]     */
		,l_set_attrib		/* [set_attrib] -- by value !! */
		,l_clr_attrib		/* [clr_attrib] -- by value !! */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_mod_holder () */

/* ------------------------------------------------------------------------- */

char vms_sys_mod_ident__doc[] =
"vms_sys.mod_ident (id [,set_attrib] [,clr_attrib] [,new_name] [,new_value])\n\
Modify Identifier in Rights Database.";

PyObject *
vms_sys_mod_ident (PyObject *self, PyObject *args)
{
	unsigned long		  l_id;

	PyObject		* ar_set_attrib;
	unsigned long		  l_set_attrib;

	PyObject		* ar_clr_attrib;
	unsigned long		  l_clr_attrib;

	char			* at_new_name;
	struct dsc$descriptor_s   r_new_name;
	struct dsc$descriptor_s * ar_new_name;
	unsigned long		  l_new_name_len;

	PyObject		* ar_new_value;
	unsigned long		  l_new_value;

	unsigned long		  l_status;

	/* -------------------- */
	ar_set_attrib = Py_None;
	ar_clr_attrib = Py_None;
	at_new_name   = NULL;
	ar_new_value  = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l|OOz#O",
		&l_id,
		&ar_set_attrib,
		&ar_clr_attrib,
		&at_new_name, &l_new_name_len,
		&ar_new_value))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 2 - set-attrib */
	if (ar_set_attrib == Py_None)
	{
	    l_set_attrib = 0;	/* by value */
	}
	else
	{
	    if (!PyInt_Check(ar_set_attrib))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: set-attrib - must be integer or None");
		return NULL;
	    }
	    l_set_attrib = PyInt_AsLong(ar_set_attrib);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 3 - clr-attrib */
	if (ar_clr_attrib == Py_None)
	{
	    l_clr_attrib = 0;	/* by value */
	}
	else
	{
	    if (!PyInt_Check(ar_clr_attrib))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: clr-attrib - must be integer or None");
		return NULL;
	    }
	    l_clr_attrib = PyInt_AsLong(ar_clr_attrib);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	/* argument 4 - new-name */
	if (at_new_name == NULL)
	{
	    ar_new_name = 0;	/* omitted */
	}
	else
	{
	    if (l_new_name_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	     "argument 4: new-name - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_new_name.dsc$w_length  = l_new_name_len;
	    r_new_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_new_name.dsc$b_class   = DSC$K_CLASS_S;
	    r_new_name.dsc$a_pointer = at_new_name;
	    ar_new_name = &r_new_name;
	}

	/* -------------------- */
	/* argument 5 - new-value */
	if (ar_new_value == Py_None)
	{
	    l_new_value = 0;	/* by value */
	}
	else
	{
	    if (!PyInt_Check(ar_new_value))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 5: new-value - must be integer or None");
		return NULL;
	    }
	    l_new_value = PyInt_AsLong(ar_new_value);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$mod_ident
		(l_id			/* id           -- by value !! */
		,l_set_attrib		/* [set_attrib] -- by value !! */
		,l_clr_attrib		/* [clr_attrib] -- by value !! */
		,ar_new_name		/* [new_name]   */
		,l_new_value		/* [new_value]  -- by value !! */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_mod_ident () */

/* ------------------------------------------------------------------------- */

char vms_sys_numtim__doc[] =
"timbuf = vms_sys.numtim ([timadr])\n\
Convert Binary Time to Numeric Time.";

PyObject *
vms_sys_numtim (PyObject *self, PyObject *args)
{
	PyObject		* ar_timadr;	/* Python Long Integer */
	long			  q_timadr[2];	/* binary system time */
	long			* aq_timadr;

	unsigned short int	  w_timbuf[7];
#define K_YEAR_SINCE_0         0
#define K_MONTH_OF_YEAR        1
#define K_DAY_OF_MONTH         2
#define K_HOUR_OF_DAY          3
#define K_MINUTE_OF_HOUR       4
#define K_SECOND_OF_MINUTE     5
#define K_HUNDREDTHS_OF_SECOND 6

	unsigned long		  l_status;

	/* -------------------- */
	ar_timadr = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|O", &ar_timadr))
	{
	    return NULL;
	}

	/* -------------------- */
	if (ar_timadr == Py_None)
	{
	    /* no argument or argument 1 is 'None' */
	    /* - indicate current date + time	   */
	    aq_timadr = 0;
	}
	else
	{
	    if (!PyLong_Check(ar_timadr))
	    {
		PyErr_SetString(PyExc_TypeError,
			"argument 1: timadr - must be long integer");
		return NULL;
	    }
	    /* convert Python long integer to binary quadword */
	    l_status = vms__cvt_pylong2quad (ar_timadr, &q_timadr[0]);
	    if (l_status == -1)
	    {
		return NULL;		/* error string has been set */
	    }
	    aq_timadr = &q_timadr[0];
	}

	/* -------------------- */
	l_status = sys$numtim
		(&w_timbuf[0]		/* timbuf   */
		,aq_timadr		/* [timadr] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* return tuple to Python */
	    return Py_BuildValue
		("(iiiiiii)"
		,(unsigned int)w_timbuf[K_YEAR_SINCE_0]
		,(unsigned int)w_timbuf[K_MONTH_OF_YEAR]
		,(unsigned int)w_timbuf[K_DAY_OF_MONTH]
		,(unsigned int)w_timbuf[K_HOUR_OF_DAY]
		,(unsigned int)w_timbuf[K_MINUTE_OF_HOUR]
		,(unsigned int)w_timbuf[K_SECOND_OF_MINUTE]
		,(unsigned int)w_timbuf[K_HUNDREDTHS_OF_SECOND]
		);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_numtim () */

/* ------------------------------------------------------------------------- */

char vms_sys_numutc__doc[] =
"timbuf = vms_sys.numtim ([utcadr])\n\
Convert UTC Time to Numeric Components.";

PyObject *
vms_sys_numutc (PyObject *self, PyObject *args)
{
	PyObject	* ar_utcadr;
	long		  o_utcadr[4];		/* utc_date_time */
	long		* ao_utcadr;

	struct	numutc_xr_timbuf {
		unsigned short int	w_year_since_0;
		unsigned short int	w_month_of_year;
		unsigned short int	w_day_of_month;
		unsigned short int	w_hour_of_day;
		unsigned short int	w_minute_of_hour;
		unsigned short int	w_second_of_minute;
		unsigned short int	w_hundredths_of_second;
		short int		w_inacc_days;	/* [1] */
		short int		w_inacc_hours;
		short int		w_inacc_minutes;
		short int		w_inacc_seconds;
		short int		w_inacc_hundredths_of_second;
		short int		w_tdf_in_minutes;
	} numutc_r_timbuf;

	/* [1] - signed variable, */
	/* because a negative value indicates infinite inaccuracy */

	unsigned long	  l_status;

	/* -------------------- */
	ar_utcadr = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|O", &ar_utcadr))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: utcadr */
	if (ar_utcadr == Py_None)
	{
	    ao_utcadr = 0;		/* omitted */
	}
	else
	{
	    if (!PyLong_Check(ar_utcadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: utcadr - must be long integer");
		return NULL;
	    }
	    /* convert Python long integer to binary octaword */
	    l_status = vms__cvt_pylong2octa (ar_utcadr, &o_utcadr[0]);
	    if (l_status != SS$_NORMAL)
	    {
		PyErr_SetString(PyExc_ValueError,
		   "argument 1: utcadr - conversion from long integer failed");
		return NULL;
	    }
	    ao_utcadr = &o_utcadr[0];
	}

	/* -------------------- */
	l_status = sys$numutc
		(&numutc_r_timbuf	/* timbuf   */
		,ao_utcadr		/* [utcadr] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* return tuple to Python */
	    return Py_BuildValue
		("(iiiiiiiiiiiii)"
		,(unsigned int)numutc_r_timbuf.w_year_since_0
		,(unsigned int)numutc_r_timbuf.w_month_of_year
		,(unsigned int)numutc_r_timbuf.w_day_of_month
		,(unsigned int)numutc_r_timbuf.w_hour_of_day
		,(unsigned int)numutc_r_timbuf.w_minute_of_hour
		,(unsigned int)numutc_r_timbuf.w_second_of_minute
		,(unsigned int)numutc_r_timbuf.w_hundredths_of_second
/* [1] */	,(int)numutc_r_timbuf.w_inacc_days
		,(int)numutc_r_timbuf.w_inacc_hours
		,(int)numutc_r_timbuf.w_inacc_minutes
		,(int)numutc_r_timbuf.w_inacc_seconds
		,(int)numutc_r_timbuf.w_inacc_hundredths_of_second
		,(int)numutc_r_timbuf.w_tdf_in_minutes
		);
	}
/* [1] - signed variable, */
/* because a negative value indicates infinite inaccuracy */

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_numutc () */

/* ------------------------------------------------------------------------- */

char vms_sys_purgws__doc[] =
"vms_sys.purgws (start-addr, end-addr)\n\
Purge Working Set.";

PyObject *
vms_sys_purgws (PyObject *self, PyObject *args)
{
	unsigned long	l_adr_low;
	unsigned long	l_adr_high;
	unsigned long	q_inadr[2];
	unsigned long	l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "ll", &l_adr_low, &l_adr_high))
	{
	    return NULL;
	}

	/* -------------------- */
	q_inadr[0] = l_adr_low;
	q_inadr[1] = l_adr_high;

	/* -------------------- */
	l_status = sys$purgws (&q_inadr[0]);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_purgws() */

/* ------------------------------------------------------------------------- */

char vms_sys_readef__doc[] =
"setflg, state = vms_sys.readef (efn)\n\
Read Event Flags.";

PyObject *
vms_sys_readef (PyObject *self, PyObject *args)
{
	unsigned long			  l_efn;
	unsigned long			  l_state;

	unsigned long			  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_efn))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$readef
			(l_efn		/* efn   -- by value */
			,&l_state	/* state */
			);

	/* -------------------- */
	if ((l_status == SS$_WASCLR) || (l_status == SS$_WASSET) )
	{
	    unsigned long	l_setflg;

	    l_setflg = (l_status == SS$_WASCLR) ? 0 : 1;

	    return Py_BuildValue ("ll", l_setflg, l_state);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_readef () */

/* ------------------------------------------------------------------------- */

char vms_sys_rem_holder__doc[] =
"vms_sys.rem_holder (id, holder)\n\
Remove Holder Record from Rights Database";

PyObject *
vms_sys_rem_holder (PyObject *self, PyObject *args)
{
	unsigned long		  l_id;

	unsigned long		  q_holder[2];
	PyObject		* ar_holder;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "lO", &l_id, &ar_holder))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 2 - holder - a tuple of 2 integers */
	if (!PyTuple_Check(ar_holder))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 2: holder - must be a tuple of 2 integers");
	    return NULL;
	}
	else
	{
	    unsigned long	  l_tuple_size;
	    PyObject		* ar_tuple_element;

	    /* -------------------- */
	    l_tuple_size = PyTuple_Size(ar_holder);
	    if (l_tuple_size != 2)
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - must be a tuple of 2 integers");
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 0);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - tuple-element:0 is not an integer");
		return NULL;
	    }
	    q_holder [0] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    /* -------------------- */
	    ar_tuple_element = PyTuple_GetItem(ar_holder, 1);
	    if (ar_tuple_element == NULL)
	    {
		return NULL;
	    }
	    if (!PyInt_Check(ar_tuple_element))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 2: holder - tuple-element:1 is not an integer");
		return NULL;
	    }
	    q_holder [1] = PyInt_AsLong(ar_tuple_element);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$rem_holder
		(l_id			/* id       -- by value !! */
		,&q_holder[0]		/* [holder] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_rem_holder () */

/* ------------------------------------------------------------------------- */

char vms_sys_rem_ident__doc[] =
"vms_sys.rem_ident (id)\n\
Remove Identifier from Rights Database.";

PyObject *
vms_sys_rem_ident (PyObject *self, PyObject *args)
{
	unsigned long		  l_id;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_id))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$rem_ident
		(l_id);			/* id -- by value !! */


	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_rem_ident () */

/* ------------------------------------------------------------------------- */

char vms_sys_resume__doc[] =
"targpid = vms_sys.resume ([pidadr] [,prcnam])\n\
Resume Process.";

PyObject *
vms_sys_resume (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_status;

	/* -------------------- */
	ar_pidadr = Py_None;
	at_prcnam = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|Oz#",
	    &ar_pidadr,
	    &at_prcnam, &l_prcnam_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	      /* turn 'None' into '0' because argument 2 might be present */
	      /*   and this routine has to return a PID */
	      l_pidadr = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		  return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	l_status = sys$resume (al_pidadr, ar_prcnam);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* always return PID of target process */
	    return PyInt_FromLong(l_pidadr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_resume () */

/* ------------------------------------------------------------------------- */

char vms_sys_revokid__doc[] =
"setflg, targpid, id, prvatr = vms_sys.revokid ([pidadr] [,prcnam] [,id] [,name])\n\
Revoke Identifier from Process.";

PyObject *
vms_sys_revokid (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  q_id[2];
	unsigned long		* aq_id;
	PyObject		* ar_id;

	char			* at_name;
	struct dsc$descriptor_s   r_name;
	struct dsc$descriptor_s	* ar_name;
	unsigned long		  l_name_len;

	unsigned long		  l_prvatr;

	unsigned long		  l_status;

	/* -------------------- */
	at_prcnam = NULL;
	ar_id     = Py_None;
	at_name   = NULL;

	/* -------------------- */
	/* allow int or None --------v---v */ 
	if (!PyArg_ParseTuple(args, "O|z#Oz#",
		&ar_pidadr,
		&at_prcnam, &l_prcnam_len,
		&ar_id,
		&at_name,   &l_name_len
		))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1 - pidadr */
	if (ar_pidadr == Py_None)
	{
	    /* turn 'None' into '0' because argument 2 might be present */
	    /*   and this routine has to return a PID */
	    l_pidadr = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2 - prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;

	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	/* argument 3 - id - a tuple of 2 integers */
	if (ar_id == Py_None)
	{
	    q_id[0] = 0;	/* arg 3 = omitted */
	    q_id[1] = 0;
	}
	else
	{
	    if (!PyTuple_Check(ar_id))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: id - must be a tuple of 2 integers");
		return NULL;
	    }
	    else
	    {
		unsigned long	  l_tuple_size;
		PyObject	* ar_tuple_element;

		/* -------------------- */
		l_tuple_size = PyTuple_Size(ar_id);
		if (l_tuple_size != 2)
		{
		    PyErr_SetString(PyExc_TypeError,
			"argument 2: id - must be a tuple of 2 integers");
		    return NULL;
		}
		/* -------------------- */
		ar_tuple_element = PyTuple_GetItem(ar_id, 0);
		if (ar_tuple_element == NULL)
		{
		    return NULL;
		}
		if (!PyInt_Check(ar_tuple_element))
		{
		    PyErr_SetString(PyExc_TypeError,
			"argument 2: id - tuple-element:0 is not an integer");
		    return NULL;
		}
		q_id[0] = PyInt_AsLong(ar_tuple_element);
		if (PyErr_Occurred())
		{
		    return NULL;
		}
		/* -------------------- */
		ar_tuple_element = PyTuple_GetItem(ar_id, 1);
		if (ar_tuple_element == NULL)
		{
		    return NULL;
		}
		if (!PyInt_Check(ar_tuple_element))
		{
		    PyErr_SetString(PyExc_TypeError,
			"argument 2: id - tuple-element:1 is not an integer");
		    return NULL;
		}
		q_id[1] = PyInt_AsLong(ar_tuple_element);
		if (PyErr_Occurred())
		{
		    return NULL;
		}
	    }
	}
	aq_id = &q_id[0];

	/* -------------------- */
	/* argument 4 - name */
	if (at_name == NULL)
	{
	    ar_name = 0;		/* omitted */
	}
	else
	{
	    if (l_name_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
		 "argument 3: name - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_name.dsc$w_length  = l_name_len;
	    r_name.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_name.dsc$b_class   = DSC$K_CLASS_S;
	    r_name.dsc$a_pointer = at_name;
	    ar_name = &r_name;
	}

	/* -------------------- */
	l_prvatr = 0;

	/* -------------------- */
	l_status = sys$revokid
		(al_pidadr		/* [pidadr] */
		,ar_prcnam		/* [prcnam] */
		,aq_id			/* [id]     */
		,ar_name		/* [name]   */
		,&l_prvatr		/* [prvatr] */
		);

	/* -------------------- */
	if ((l_status == SS$_WASCLR) || (l_status == SS$_WASSET) )
	{
	    unsigned long  l_setflg;
	    l_setflg = (l_status == SS$_WASCLR) ? 0 : 1;
	    /* always return PID of target process */
	    return Py_BuildValue ("ll(ll)l",
		l_setflg, l_pidadr, q_id[0], q_id[1], l_prvatr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_revokid () */

/* ------------------------------------------------------------------------- */

char vms_sys_schdwk__doc[] =
"targpid = vms_sys.schdwk ([pidadr], [prcnam], daytim [,reptim])\n\
Schedule Wakeup.";

PyObject *
vms_sys_schdwk (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	PyObject		* ar_daytim;
	long			  q_daytim[2];

	PyObject		* ar_reptim;
	long			  q_reptim[2];
	long			* aq_reptim;

	unsigned long		  l_status;

	/* -------------------- */
	at_prcnam    = NULL;
	ar_daytim    = Py_None;
	ar_reptim    = Py_None;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "Oz#O|O",
		&ar_pidadr,
		&at_prcnam, &l_prcnam_len,
		&ar_daytim,
		&ar_reptim))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	    l_pidadr = 0;		/* omitted */
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;	/* always return PID */

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;	/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	/* argument 3: daytim */
	if (!PyLong_Check(ar_daytim))
	{
	    PyErr_SetString(PyExc_TypeError,
		"argument 3: daytim - must be long integer");
	    return NULL;
	}

	/* convert Python long integer to binary quadword */
	l_status = vms__cvt_pylong2quad (ar_daytim, &q_daytim[0]);
	if (l_status == -1)
	{
	    return NULL;		/* error string has been set */
	}

	/* -------------------- */
	/* argument 4: reptim */
	if (ar_reptim == Py_None)
	{
	    aq_reptim = 0;		/* omitted */
	}
	else
	{
	    if (!PyLong_Check(ar_reptim))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 4: reptim - must be long integer");
		return NULL;
	    }

	    l_status = vms__cvt_pylong2quad (ar_reptim, &q_reptim[0]);
	    if (l_status == -1)
	    {
		return NULL;		/* error string has been set */
	    }
	    aq_reptim = &q_reptim[0];
	}

	/* -------------------- */
	l_status = sys$schdwk
		(al_pidadr		/* [pidadr] */
		,ar_prcnam		/* [prcnam] */
		,&q_daytim[0]		/* daytim   */
		,aq_reptim		/* [reptim] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* always return PID of target process */
	    return PyInt_FromLong(l_pidadr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_schdwk () */

/* ------------------------------------------------------------------------- */
/*
   Allows you to read and change the default directory string for the process.
   Note that the device name is stored in the logical name SYS$DISK !
*/

char vms_sys_setddir__doc[] =
"cur-dir = vms_sys.setddir ([new-dir])\n\
Set Default Directory.";

PyObject *
vms_sys_setddir (PyObject *self, PyObject *args)
{
	char			* at_newdir;
	struct dsc$descriptor_s   r_newdir;
	struct dsc$descriptor_s	* ar_newdir;
	unsigned long		  l_newdir_len;

	char			  t_curdir[256];
	struct dsc$descriptor_s   r_curdir;
	unsigned short int	  w_curdir_len;

	unsigned long		  l_status;

	/* -------------------- */
	at_newdir = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|z#", &at_newdir, &l_newdir_len))
	{
	    return NULL;
	}

	/* -------------------- */
	if (at_newdir == NULL)
	{
	    ar_newdir = 0;	/* omitted - don't set new directory */
	}
	else
	{
	    if (l_newdir_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 1: newdir - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_newdir.dsc$w_length  = l_newdir_len;
	    r_newdir.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_newdir.dsc$b_class   = DSC$K_CLASS_S;
	    r_newdir.dsc$a_pointer = at_newdir;
	    ar_newdir = &r_newdir;
	}

	/* -------------------- */
	/* set up descriptor */
	r_curdir.dsc$w_length  = sizeof(t_curdir);
	r_curdir.dsc$b_dtype   = DSC$K_DTYPE_T;
	r_curdir.dsc$b_class   = DSC$K_CLASS_S;
	r_curdir.dsc$a_pointer = &t_curdir[0];
	w_curdir_len = 0;

	/* -------------------- */
	l_status = sys$setddir
		(ar_newdir		/* [new-dir-addr] */
		,&w_curdir_len		/* [length-addr]  */
		,&r_curdir		/* [cur-dir-addr] */
		);

	/* -------------------- */
	if (l_status == RMS$_NORMAL)
	{
	    /* always always return the current directory */
	    return Py_BuildValue ("s#",
		&t_curdir[0], (unsigned int)w_curdir_len);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_setddir () */

/* ------------------------------------------------------------------------- */

char vms_sys_setef__doc[] =
"setflg = vms_sys.setef (efn)\n\
Set Event Flag.";

PyObject *
vms_sys_setef (PyObject *self, PyObject *args)
{
	unsigned long		  l_efn;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_efn))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$setef (l_efn);	/* efn -- by value */

	/* -------------------- */
	if ((l_status == SS$_WASCLR) || (l_status == SS$_WASSET) )
	{
	    unsigned long  l_setflg;
	    l_setflg = (l_status == SS$_WASCLR) ? 0 : 1;
	    return PyInt_FromLong(l_setflg);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_setef () */

/* ------------------------------------------------------------------------- */

char vms_sys_setpri__doc[] =
"targpid, previous_priority, previous_policy = vms_sys.setpri (\
[pid], [prcnam], priority [, policy])\n\
Set Priority.";

PyObject *
vms_sys_setpri (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_pri;
	unsigned long		  l_prvpri;

	unsigned long		  l_policy;
	unsigned long		* al_policy;
	PyObject		* ar_policy;

	unsigned long		  l_prvpol;
	unsigned long		* al_prvpol;

	unsigned long		  l_status;

	/* -------------------- */
	ar_policy = Py_None;

	/* -------------------- */
	/* argument 3 (policy) is optional and only implemented on Alpha */
	if (!PyArg_ParseTuple(args, "Oz#i|O",
		&ar_pidadr,
		&at_prcnam, &l_prcnam_len,
		&l_pri,
		&ar_policy))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1 pidadr */
	if (ar_pidadr == Py_None)
	{
	    l_pidadr = 0;		/* omitted - but return target PID */
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;	/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	/* argument 3: policy */
#ifndef __ALPHA
	/* these arguments are not supported on VAX */
	al_policy = 0;
	al_prvpol = 0;
	l_prvpol  = 0;			/* fill in some value for return */
#else
	if (ar_policy == Py_None)
	{
	    al_policy = 0;		/* omitted */
	}
	else
	{
	    if (!PyInt_Check(ar_policy))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 3: policy - must be integer or None");
		return NULL;
	    }
	    l_policy  = PyInt_AsLong(ar_policy);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	    al_policy = &l_policy;
	}

	al_prvpol = &l_prvpol;	/* always returned */
#endif

	/* -------------------- */
	l_status = sys$setpri
		(al_pidadr		/* [pidadr]  */
		,ar_prcnam		/* [prcnam]  */
		,l_pri			/* pri       */
		,&l_prvpri		/* [prvpri]  */
		,al_policy		/* VAX:[nullarg], Alpha:[policy] */
		,al_prvpol		/* VAX:[nullarg], Alpha:[prvpol] */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* always return 'prvpol',even on VAX where it is not implemented */
	    return Py_BuildValue ("lll", l_pidadr, l_prvpri, l_prvpol);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_setpri () */

/* ------------------------------------------------------------------------- */

char vms_sys_setprn__doc[] =
"vms_sys.setprn ([prcnam])\n\
Set Process Name.";

PyObject *
vms_sys_setprn (PyObject *self, PyObject *args)
{
	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_status;

	/* -------------------- */
	at_prcnam = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|z#", &at_prcnam, &l_prcnam_len))
	{
	    return NULL;
	}

	/* -------------------- */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;	/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 1: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	l_status = sys$setprn (ar_prcnam);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_setprn () */

/* ------------------------------------------------------------------------- */

char vms_sys_setswm__doc[] =
"oldflg = vms_sys.setswm ([swpflg])\n\
Set Process Swap Mode.";

PyObject *
vms_sys_setswm (PyObject *self, PyObject *args)
{
	unsigned long	  l_swpflg;
	PyObject	* ar_swpflg;

	unsigned long	  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O", &ar_swpflg))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: swpflg */
	if (ar_swpflg == Py_None)
	{
	    l_swpflg = 1;		/* disable swapping */
	}
	else
	{
	    if (!PyInt_Check(ar_swpflg))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: swpflg - must be integer or None");
		return NULL;
	    }
	    l_swpflg = PyInt_AsLong(ar_swpflg);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$setswm(l_swpflg);

	/* -------------------- */
	if ((l_status == SS$_WASCLR) || (l_status == SS$_WASSET))
	{
	    l_swpflg = (l_status == SS$_WASCLR) ? 0 : 1;
	    return PyInt_FromLong(l_swpflg);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_setswm () */

/* ------------------------------------------------------------------------- */

char vms_sys_subsystem__doc[] =
"oldflg = vms_sys.subsystem ([enbflg])\n\
Subsystem.";

PyObject *
vms_sys_subsystem (PyObject *self, PyObject *args)
{
	unsigned long	  l_enbflg;
	PyObject	* ar_enbflg;

	unsigned long	  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O", &ar_enbflg))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: enbflg */
	if (ar_enbflg == Py_None)
	{
	    l_enbflg = 0;		/* save (disable) subsystem */
	}
	else
	{
	    if (!PyInt_Check(ar_enbflg))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: enbflg - must be integer or None");
		return NULL;
	    }
	    l_enbflg = PyInt_AsLong(ar_enbflg);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}

	/* -------------------- */
	l_status = sys$subsystem (l_enbflg);

	/* -------------------- */
	if ((l_status == SS$_WASCLR) || (l_status == SS$_WASSET))
	{
	    if (l_status == SS$_WASCLR)
	    {
		l_enbflg = 0;
	    }
	    else
	    {
		l_enbflg = 1;
	    }
	    return PyInt_FromLong(l_enbflg);
	}

	/* -------------------- */
	/* error (according to the documentation this can't happen) */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_subsystem () */

/* ------------------------------------------------------------------------- */

char vms_sys_suspnd__doc[] =
"targpid = vms_sys.suspnd ([pidadr] [,prcnam] [,flags])\n\
Suspend Process.";

PyObject *
vms_sys_suspnd (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s	* ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_flags;

	unsigned long		  l_status;

	/* -------------------- */
	ar_pidadr = Py_None;
	at_prcnam = NULL;
	l_flags   = 0;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "|Oz#i",
	    &ar_pidadr,
	    &at_prcnam, &l_prcnam_len,
	    &l_flags))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	    /* turn 'None' into '0' because argument 2 might be present */
	    /*   and this routine has to return a PID */
	    l_pidadr = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	l_status = sys$suspnd
		(al_pidadr		/* [pidadr] */
		,ar_prcnam		/* [prcnam] */
		,l_flags		/* flags    */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* always return PID of target process */
	    return PyInt_FromLong(l_pidadr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_suspnd () */

/* ------------------------------------------------------------------------- */

char vms_sys_timcon__doc[] =
"utcadr = vms_sys.timcon ([timadr] , cvtflg=1)\n\
timadr = vms_sys.timcon ([utcadr] , cvtflg=0)\n\
Time Converter.";

PyObject *
vms_sys_timcon (PyObject *self, PyObject *args)
{
	long			  q_timadr[2];	/* binary system time */
	long			  o_utcadr[4];	/* utc_date_time      */
	long			* ax_argtim;

	/* ----- */
	PyObject		* ar_argtim;	/* argument 1	   */

	unsigned long		  l_cvtflg;	/* conversion flag $TIMCON */

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "Ol", &ar_argtim, &l_cvtflg))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: argtim */
	if (ar_argtim == Py_None)
	{
	      ax_argtim = 0;		/* omitted */
	}
	else
	{
	  /* conversion direction doesn't matter - */
	  /*  must always be long integer */
	  if (!PyLong_Check(ar_argtim))
	  {
	      PyErr_SetString(PyExc_TypeError,
		"argument 1: argtim - must be long integer");
	      return NULL;
	  }
	  /* ---------- */
	  /* l_cvtflg = 0 -- convert 128bit utc to 64bit bintim */
	  if (l_cvtflg == 0)
	  {
	      /* convert Python long integer to binary octaword */
	      l_status = vms__cvt_pylong2octa (ar_argtim, &o_utcadr[0]);
	      if (l_status != SS$_NORMAL)
	      {
		  PyErr_SetString(PyExc_ValueError,
		   "argument 1: argtim - conversion from long integer failed");
		  return NULL;
	      }
	      ax_argtim = &o_utcadr[0];
	  } /* (l_cvtflg == 0) */
	  else
	  /* l_cvtflg = 1 -- convert 64bit bintim to 128bit utc */
	  /*	ar_argtim must be Python Long-Integer ! */
	  {
	    /* convert Python long integer to binary quadword */
	    l_status = vms__cvt_pylong2quad (ar_argtim, &q_timadr[0]);
	    if (l_status == -1)
	    {
	      return NULL;		/* error string has been set */
	    }
	    ax_argtim = &q_timadr[0];
	  } /* if (l_cvtflg == 0) */
	} /* if (ar_argtim == Py_None) */

	/* -------------------- */
	/* l_cvtflg = 0 -- convert 128bit utc to 64bit bintim */
	if (l_cvtflg == 0)
	{
	    l_status = sys$timcon
		(&q_timadr[0]		/* [timadr] */
		,ax_argtim		/* [utcadr] */
		,l_cvtflg		/* cvtflg   */
		);
	    if (l_status == SS$_NORMAL)
	    {
		/* convert binary quadword to Python long integer */
		return vms__cvt_quad2pylong (&q_timadr[0]);
	    }
	}
	else
	{
	    l_status = sys$timcon
		(ax_argtim		/* [timadr] */
		,&o_utcadr[0]		/* [utcadr] */
		,l_cvtflg		/* cvtflg   */
		);
	    if (l_status == SS$_NORMAL)
	    {
		/* convert binary octaword to Python long integer */
		return vms__cvt_octa2pylong (&o_utcadr[0]);
	    }
	} /* if (l_cvtflg == 0) */

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_timcon () */

/* ------------------------------------------------------------------------- */

char vms_sys_waitfr__doc[] =
"vms_sys.waitfr (efn)\n\
Wait for Single Event Flag.";

PyObject *
vms_sys_waitfr (PyObject *self, PyObject *args)
{
	unsigned long		  l_efn;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "l", &l_efn))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$waitfr (l_efn);	/* efn -- by value */

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_waitfr () */

/* ------------------------------------------------------------------------- */

char vms_sys_wake__doc[] =
"targpid = vms_sys.wake ([pidadr] [,prcnam])\n\
Wake Process from Hibernation.";

PyObject *
vms_sys_wake (PyObject *self, PyObject *args)
{
	unsigned long		  l_pidadr;
	unsigned long		* al_pidadr;
	PyObject		* ar_pidadr;

	char			* at_prcnam;
	struct dsc$descriptor_s   r_prcnam;
	struct dsc$descriptor_s * ar_prcnam;
	unsigned long		  l_prcnam_len;

	unsigned long		  l_status;

	/* -------------------- */
	ar_pidadr = Py_None;
	at_prcnam = NULL;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "O|z#",
	    &ar_pidadr,
	    &at_prcnam, &l_prcnam_len))
	{
	    return NULL;
	}

	/* -------------------- */
	/* argument 1: pidadr */
	if (ar_pidadr == Py_None)
	{
	    /* turn 'None' into '0' because argument 2 might be present */
	    /*   and this routine has to return a PID */
	    l_pidadr = 0;
	}
	else
	{
	    if (!PyInt_Check(ar_pidadr))
	    {
		PyErr_SetString(PyExc_TypeError,
		    "argument 1: pidadr - must be integer or None");
		return NULL;
	    }
	    l_pidadr = PyInt_AsLong(ar_pidadr);
	    if (PyErr_Occurred())
	    {
		return NULL;
	    }
	}
	al_pidadr = &l_pidadr;

	/* -------------------- */
	/* argument 2: prcnam */
	if (at_prcnam == NULL)
	{
	    ar_prcnam = 0;		/* omitted */
	}
	else
	{
	    if (l_prcnam_len > 65535)
	    {
		PyErr_SetString(PyExc_ValueError,
	       "argument 2: prcnam - string size limited to 65535 characters");
		return NULL;
	    }
	    /* set up descriptor */
	    r_prcnam.dsc$w_length  = l_prcnam_len;
	    r_prcnam.dsc$b_dtype   = DSC$K_DTYPE_T;
	    r_prcnam.dsc$b_class   = DSC$K_CLASS_S;
	    r_prcnam.dsc$a_pointer = at_prcnam;
	    ar_prcnam = &r_prcnam;
	}

	/* -------------------- */
	l_status = sys$wake (al_pidadr, ar_prcnam);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    /* always return PID of target process */
	    return PyInt_FromLong(l_pidadr);
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_wake () */

/* ------------------------------------------------------------------------- */

char vms_sys_wfland__doc[] =
"vms_sys.wfland (efn, mask)\n\
Wait for Logical AND of Event Flags.";

PyObject *
vms_sys_wfland (PyObject *self, PyObject *args)
{
	unsigned long		  l_efn;
	unsigned long		  l_mask;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "ll", &l_efn, &l_mask))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$wfland
		(l_efn		/* efn  -- by value */
		,l_mask		/* mask -- by value */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_wfland () */

/* ------------------------------------------------------------------------- */

char vms_sys_wflor__doc[] =
"vms_sys.wflor (efn, mask)\n\
Wait for Logical OR of Event Flags.";

PyObject *
vms_sys_wflor (PyObject *self, PyObject *args)
{
	unsigned long		  l_efn;
	unsigned long		  l_mask;

	unsigned long		  l_status;

	/* -------------------- */
	if (!PyArg_ParseTuple(args, "ll", &l_efn, &l_mask))
	{
	    return NULL;
	}

	/* -------------------- */
	l_status = sys$wflor
		(l_efn		/* efn  -- by value */
		,l_mask		/* mask -- by value */
		);

	/* -------------------- */
	if (l_status == SS$_NORMAL)
	{
	    Py_INCREF(Py_None);
	    return Py_None;
	}

	/* -------------------- */
	/* error */
	return PyVMS_ErrSetVal(vms_sys_gr_error, 1, l_status);
} /* vms_sys_wflor () */

/* ------------------------------------------------------------------------- */

/* static struct PyMethodDef vms_sys_methods[] = {
--	{NULL,		NULL}
-- }; */

/* data is now in file 'VMS_SYS__METHODS.DAT' */
/* see 'General Manual', configuration guidelines */
extern struct PyMethodDef vms_sys__methods[];

void
initvms_sys()
{
	PyObject *m, *d;

	m = Py_InitModule4("vms_sys",
			   vms_sys__methods,
			   vms_sys_doc__,
			   (PyObject *)NULL,
			   PYTHON_API_VERSION);
	d = PyModule_GetDict(m);

	/* -------------------- */
        if (PyErr_Occurred())
	{
	    (void) Py_FatalError(vms_sys_t_cant_init);
	}

	/* -------------------- */
	/* Initialize exception */
	vms_sys_gr_error = PyErr_NewException("vms_sys.error", NULL, NULL);
	if (vms_sys_gr_error == NULL)
	{
	    (void) Py_FatalError(vms_sys_t_cant_deferr);
	}

	if (PyDict_SetItemString(d, "error", vms_sys_gr_error) != 0)
	{
	    (void) Py_FatalError(vms_sys_t_cant_deferr);
	}
} /* initvms_sys () */

/* ------------------------------------------------------------------------- */

/* EOF: VMS_SYS.C */
