/*
# X-BASED SKEWB
#
#  Skewb2d.c
#
###
#
#  Copyright (c) 1994		David Albert Bagley, bagleyd@source.asset.com
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Skewb2d */

#include <stdio.h>
#include <math.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include "SkewbP.h"
#include "Skewb2dP.h"

static void InitializeSkewb2D();
static void ResizeSkewb2D();
static void ExposeSkewb2D();
static Boolean SetValuesSkewb2D();
static void move_skewb2d_tl();
static void move_skewb2d_tr();
static void move_skewb2d_cw();
static void move_skewb2d_bl();
static void move_skewb2d_br();
static void move_skewb2d_ccw();
static void practice_skewb2d();
static void maybe_practice_skewb2d();
static void randomize_skewb2d();
static void maybe_randomize_skewb2d();
static void orientize_skewb2d();
static void move_skewb2d();
static void resize_polyhedrons();
static int position_polyhedrons();
static void no_move_polyhedrons();
static void practice_polyhedrons();
static void randomize_polyhedrons();
static void move_polyhedrons();
static void write_diagonal();
static void draw_frame();
static void draw_all_polyhedrons();
static void draw_diamond();
static void draw_triangle();
static void draw_orient_line();

static char defaultTranslationsSkewb2D[] =
  "<KeyPress>q: quit()\n\
   Ctrl<KeyPress>C: quit()\n\
   <KeyPress>Home: move_tl()\n\
   <KeyPress>KP_7: move_tl()\n\
   <KeyPress>R7: move_tl()\n\
   <KeyPress>Prior: move_tr()\n\
   <KeyPress>KP_9: move_tr()\n\
   <KeyPress>R9: move_tr()\n\
   <KeyPress>Begin: move_cw()\n\
   <KeyPress>KP_5: move_cw()\n\
   <KeyPress>R11: move_cw()\n\
   <KeyPress>End: move_bl()\n\
   <KeyPress>KP_1: move_bl()\n\
   <KeyPress>R1: move_bl()\n\
   <KeyPress>Next: move_br()\n\
   <KeyPress>KP_3: move_br()\n\
   <KeyPress>R3: move_br()\n\
   <Btn1Down>: move_ccw()\n\
   <Btn1Motion>: move_ccw()\n\
   <KeyPress>p: practice()\n\
   <Btn2Down>(2+): practice()\n\
   <Btn2Down>: maybe_practice()\n\
   <KeyPress>r: randomize()\n\
   <Btn3Down>(2+): randomize()\n\
   <Btn3Down>: maybe_randomize()\n\
   <KeyPress>o: orientize()";

static XtActionsRec actionsListSkewb2D[] =
{
  {"quit", (XtActionProc) quit_skewb},
  {"move_tl", (XtActionProc) move_skewb2d_tl},
  {"move_tr", (XtActionProc) move_skewb2d_tr},
  {"move_cw", (XtActionProc) move_skewb2d_cw},
  {"move_bl", (XtActionProc) move_skewb2d_bl},
  {"move_br", (XtActionProc) move_skewb2d_br},
  {"move_ccw", (XtActionProc) move_skewb2d_ccw},
  {"practice", (XtActionProc) practice_skewb2d},
  {"maybe_practice", (XtActionProc) maybe_practice_skewb2d},
  {"randomize", (XtActionProc) randomize_skewb2d},
  {"maybe_randomize", (XtActionProc) maybe_randomize_skewb2d},
  {"orientize", (XtActionProc) orientize_skewb2d}
};

static XtResource resourcesSkewb2D[] =
{
  {XtNfaceColor0, XtCLabel, XtRString, sizeof(String),
   XtOffset(SkewbWidget, skewb.face_name[0]), XtRString, "Red"},
  {XtNfaceColor1, XtCLabel, XtRString, sizeof(String),
   XtOffset(SkewbWidget, skewb.face_name[1]), XtRString, "Blue"},
  {XtNfaceColor2, XtCLabel, XtRString, sizeof(String),
   XtOffset(SkewbWidget, skewb.face_name[2]), XtRString, "White"},
  {XtNfaceColor3, XtCLabel, XtRString, sizeof(String),
   XtOffset(SkewbWidget, skewb.face_name[3]), XtRString, "Green"},
  {XtNfaceColor4, XtCLabel, XtRString, sizeof(String),
   XtOffset(SkewbWidget, skewb.face_name[4]), XtRString, "Pink"},
  {XtNfaceColor5, XtCLabel, XtRString, sizeof(String),
   XtOffset(SkewbWidget, skewb.face_name[5]), XtRString, "Yellow"},
  {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
   XtOffset(SkewbWidget, skewb.foreground), XtRString, XtDefaultForeground},
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
   XtOffset(SkewbWidget, core.width), XtRString, "300"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
   XtOffset(SkewbWidget, core.height), XtRString, "400"},
  {XtNorient, XtCOrient, XtRBoolean, sizeof(Boolean),
   XtOffset(SkewbWidget, skewb.orient), XtRString, "FALSE"}, /* DEFAULTORIENT */
  {XtNmono, XtCMono, XtRBoolean, sizeof(Boolean),
   XtOffset(SkewbWidget, skewb.mono), XtRString, "FALSE"},
  {XtNface, XtCFace, XtRInt, sizeof(int),
   XtOffset(SkewbWidget, skewb.face), XtRString, "-1"},
  {XtNcube, XtCCube, XtRInt, sizeof(int),
   XtOffset(SkewbWidget, skewb.cube), XtRString, "-1"},
  {XtNdirection, XtCDirection, XtRInt, sizeof(int),
   XtOffset(SkewbWidget, skewb.direction), XtRString, "-1"},
  {XtNpractice, XtCBoolean, XtRBoolean, sizeof(Boolean),
   XtOffset(SkewbWidget, skewb.practice), XtRString, "FALSE"},
  {XtNstart, XtCBoolean, XtRBoolean, sizeof(Boolean),
   XtOffset(SkewbWidget, skewb.started), XtRString, "FALSE"},
  {XtNselectCallback, XtCCallback, XtRCallback, sizeof(caddr_t),
   XtOffset(SkewbWidget, skewb.select), XtRCallback, NULL}
};

Skewb2DClassRec skewb2dClassRec =
{
  {
    (WidgetClass) &skewbClassRec,	/* superclass */
    "Skewb2D",				/* class name */
    sizeof(Skewb2DRec),			/* widget size */
    NULL,				/* class initialize */
    NULL,				/* class part initialize */
    FALSE,				/* class inited */
    InitializeSkewb2D,			/* initialize */
    NULL,				/* initialize hook */
    XtInheritRealize,			/* realize */
    actionsListSkewb2D,			/* actions */
    XtNumber(actionsListSkewb2D),	/* num actions */
    resourcesSkewb2D,			/* resources */
    XtNumber(resourcesSkewb2D),		/* num resources */
    NULLQUARK,				/* xrm class */
    TRUE,				/* compress motion */
    TRUE,				/* compress exposure */
    TRUE,				/* compress enterleave */
    TRUE,				/* visible interest */
    NULL,				/* destroy */
    ResizeSkewb2D,			/* resize */
    ExposeSkewb2D,			/* expose */
    SetValuesSkewb2D,			/* set values */
    NULL,				/* set values hook */
    XtInheritSetValuesAlmost,		/* set values almost */
    NULL,				/* get values hook */
    XtInheritAcceptFocus,		/* accept focus */
    XtVersion,				/* version */
    NULL,				/* callback private */
    defaultTranslationsSkewb2D,		/* tm table */
    NULL,				/* query geometry */
    NULL,				/* display accelerator */
    NULL				/* extension */
  },
  {
    0					/* ignore */
  },
  {
    0					/* ignore */
  }
};

WidgetClass skewb2dWidgetClass = (WidgetClass) &skewb2dClassRec;
static SkewbLoc slide_next_row[MAXFACES][MAXORIENT][MAXORIENT / 2] =
{
  {
    {{2,   CW}, {1, HALF}},
    {{5,  CCW}, {1, STRT}},
    {{3, STRT}, {5,   CW}},
    {{3, HALF}, {2,  CCW}}
  },
  {
    {{4, STRT}, {5,   CW}},
    {{0, STRT}, {5,  CCW}},
    {{2,  CCW}, {0, HALF}},
    {{2,   CW}, {4, HALF}}
  },
  {
    {{4,   CW}, {1,  CCW}},
    {{0,  CCW}, {1,   CW}},
    {{3,  CCW}, {0,   CW}},
    {{3,   CW}, {4,  CCW}}
  },
  {
    {{4, HALF}, {2,  CCW}},
    {{0, HALF}, {2,   CW}},
    {{5,   CW}, {0, STRT}},
    {{5,  CCW}, {4, STRT}}
  },
  {
    {{5,   CW}, {1, STRT}},
    {{2,  CCW}, {1, HALF}},
    {{3, HALF}, {2,   CW}},
    {{3, STRT}, {5,  CCW}}
  },
  {
    {{0,   CW}, {1,   CW}},
    {{4,  CCW}, {1,  CCW}},
    {{3,   CW}, {4,   CW}},
    {{3,  CCW}, {0,  CCW}}
  }
};
static SkewbLoc min_to_maj[MAXFACES][MAXORIENT] =
{ /* other equivalent mappings possible */
  {{3,   CW}, {2, STRT}, {1,  CCW}, {5, STRT}},
  {{2, STRT}, {4,  CCW}, {5, HALF}, {0,   CW}},
  {{3, STRT}, {4, STRT}, {1, STRT}, {0, STRT}},
  {{5, HALF}, {4,   CW}, {2, STRT}, {0,  CCW}},
  {{3,  CCW}, {5, STRT}, {1,   CW}, {2, STRT}},
  {{3, HALF}, {0, STRT}, {1, HALF}, {4, STRT}}
};
static int plane_to_cube[MAXRECT] = {6, 0, 6, 1, 2, 3, 6, 4, 6, 6, 5, 6};
static int cube_to_plane[MAXFACES] = {1, 3, 4, 5, 7, 10};
XPoint diamond_unit[MAXORIENT] = {{0, 1}, {1, -1}, {1, 1}, {-1, 1}},
  triangle_unit[MAXORIENT][3] = {
                                  {{2, 0},  {1, 0},  {0, 1}},
                                  {{3, 2},  {0, 1}, {-1, 0}},
                                  {{1, 3}, {-1, 0}, {0, -1}},
                                  {{0, 1}, {0, -1},  {1, 0}}
                                },
  letter_unit[MAXCUBES] = {{2, 0}, {2, 2}, {0, 2}, {0, 0}, {1, 1}};
XPoint diamond_list[MAXORIENT], triangle_list[MAXORIENT][3],
  letter_list[MAXCUBES], offset_list[MAXCUBES];

static void InitializeSkewb2D(request, new)
Skewb2DWidget request, new;
{
  ResizeSkewb2D(new);
}

static void ResizeSkewb2D(w)
Skewb2DWidget w;
{
  w->skewb.delta = 3;
  w->skewb.vertical = (w->core.height >= w->core.width);
  if (w->skewb.vertical)
    w->skewb2d.face_length = (int) MIN(w->core.height / MAXY,
      w->core.width / MAXX);
  else
    w->skewb2d.face_length = (int) MIN(w->core.height / MAXX,
      w->core.width / MAXY);
  w->skewb2d.face_length = MAX(w->skewb2d.face_length - w->skewb.delta - 1, 0);
  w->skewb2d.diamond_length = w->skewb2d.face_length - w->skewb.delta;
  w->skewb2d.triangle_length = w->skewb2d.diamond_length / 2 - w->skewb.delta;
  w->skewb2d.view_length = w->skewb2d.face_length + w->skewb.delta;
  if (w->skewb.vertical) {
    w->skewb.puzzle_width = MAXX * (w->skewb2d.view_length - 1) +
      w->skewb.delta;
    w->skewb.puzzle_height = MAXY * (w->skewb2d.view_length - 1) +
      w->skewb.delta;
  } else {
    w->skewb.puzzle_width = MAXY * (w->skewb2d.view_length - 1) +
      w->skewb.delta;
    w->skewb.puzzle_height = MAXX * (w->skewb2d.view_length - 1) +
      w->skewb.delta;
  }
  w->skewb.puzzle_offset.x = ((int) w->core.width - w->skewb.puzzle_width)
    / 2;
  w->skewb.puzzle_offset.y = ((int) w->core.height - w->skewb.puzzle_height)
    / 2;
  resize_polyhedrons(w);
}

static void ExposeSkewb2D(w, event, region)
Skewb2DWidget w;
XEvent *event;
Region region; /* Not used */
{
  if (w->core.visible) {
    draw_frame(w, w->skewb.puzzle_GC);
    draw_all_polyhedrons(w);
  }
}

static Boolean SetValuesSkewb2D(current, request, new)
Skewb2DWidget current, request, new;
{
  Boolean redraw = FALSE;

  if (new->skewb2d.diamond_length != current->skewb2d.diamond_length) {
    ResizeSkewb2D(new);
    redraw = TRUE;
  }
  if (new->skewb.face != SKEWB_IGNORE) {
    move_polyhedrons(new, new->skewb.face, new->skewb.cube,
      new->skewb.direction);
    new->skewb.randomized = !check_solved((SkewbWidget) new);
    new->skewb.face = SKEWB_IGNORE;
  }
  return (redraw);
}

static void practice_skewb2d(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  practice_polyhedrons(w);
}

static void maybe_practice_skewb2d(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  if (!w->skewb.started)
    practice_polyhedrons(w);
}

static void randomize_skewb2d(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  randomize_polyhedrons(w);
}

static void maybe_randomize_skewb2d(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  if (!w->skewb.started)
    randomize_polyhedrons(w);
}

static void orientize_skewb2d(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  skewbCallbackStruct cb;

  cb.reason = SKEWB_ORIENT;
  XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

static void move_skewb2d_ccw(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  move_skewb2d(w, event->xbutton.x, event->xbutton.y, CCW,
    (int) (event->xbutton.state & ControlMask));
}

static void move_skewb2d_tl(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  move_skewb2d(w, event->xbutton.x, event->xbutton.y, TL,
    (int) (event->xkey.state & ControlMask));
}

static void move_skewb2d_tr(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  move_skewb2d(w, event->xbutton.x, event->xbutton.y, TR,
    (int) (event->xkey.state & ControlMask));
}

static void move_skewb2d_cw(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  move_skewb2d(w, event->xbutton.x, event->xbutton.y, CW,
    (int) (event->xkey.state & ControlMask));
}

static void move_skewb2d_bl(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  move_skewb2d(w, event->xbutton.x, event->xbutton.y, BL,
    (int) (event->xkey.state & ControlMask));
}

static void move_skewb2d_br(w, event, args, n_args)
Skewb2DWidget w;
XEvent *event;
char *args[];
int n_args;
{
  move_skewb2d(w, event->xbutton.x, event->xbutton.y, BR,
    (int) (event->xkey.state & ControlMask));
}

static void move_skewb2d(w, x, y, direction, control)
Skewb2DWidget w;
int x, y, direction, control;
{
  skewbCallbackStruct cb;
  int face, corner, new_face, rotate;

  if (control) {
    if (!position_polyhedrons(w, x, y, &face, &corner, &direction))
      return;
    move_polyhedrons(w, face, corner, direction);
    cb.reason = SKEWB_CONTROL;
    cb.face = face;
    cb.corner = corner;
    cb.direction = direction;
    XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
    new_face = min_to_maj[face][corner].face;
    rotate = min_to_maj[face][corner].rotation % MAXORIENT;
    direction = (rotate + direction) % MAXORIENT;
    corner = (corner + rotate + 2) % MAXORIENT;
    move_polyhedrons(w, new_face, corner, direction);
    cb.face = new_face;
    cb.corner = corner;
    cb.direction = direction;
    XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
  } else if (!w->skewb.randomized && !w->skewb.practice)
    no_move_polyhedrons(w);
  else {
    if (!position_polyhedrons(w, x, y, &face, &corner, &direction))
      return;
    move_polyhedrons(w, face, corner, direction);
    w->skewb.randomized = !check_solved((SkewbWidget) w);
    cb.reason = SKEWB_MOVED;
    cb.face = face;
    cb.corner = corner;
    cb.direction = direction;
    XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
    if (!w->skewb.randomized) {
      cb.reason = SKEWB_SOLVED;
      XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
    }
  }
}

static void resize_polyhedrons(new)
Skewb2DWidget new;
{
  int i, j;

  new->skewb.orient_line_length = new->skewb2d.diamond_length / 8;
  new->skewb.letter_offset.x = -2;
  new->skewb.letter_offset.y = 4;
  for (i = 0; i < MAXORIENT; i++)
  {
    diamond_list[i].x = diamond_unit[i].x * (new->skewb2d.diamond_length / 2 -
      new->skewb.delta);
    diamond_list[i].y = diamond_unit[i].y * (new->skewb2d.diamond_length / 2 -
      new->skewb.delta);
    for (j = 0; j < 3; j++)
    {
      triangle_list[i][j].x = triangle_unit[i][j].x *
        (new->skewb2d.diamond_length / 2 - 3 * new->skewb.delta);
      triangle_list[i][j].y = triangle_unit[i][j].y *
        (new->skewb2d.diamond_length / 2 - 3 * new->skewb.delta);
    }
    if (letter_unit[i].x == 0)
      letter_list[i].x = new->skewb2d.diamond_length / 8 +
        new->skewb.letter_offset.x;
    else if (letter_unit[i].x == 2)
      letter_list[i].x = 7 * new->skewb2d.diamond_length / 8 - 2 +
        new->skewb.letter_offset.x;
    if (letter_unit[i].y == 0)
      letter_list[i].y = new->skewb2d.diamond_length / 8 + 2 +
        new->skewb.letter_offset.y;
    else if (letter_unit[i].y == 2)
      letter_list[i].y = 7 * new->skewb2d.diamond_length / 8 - 3 +
        new->skewb.letter_offset.y;

    if (triangle_unit[i][0].x == 0)
      offset_list[i].x = new->skewb.delta - 1;
    else if (triangle_unit[i][0].x == 1)
      offset_list[i].x = new->skewb2d.diamond_length / 2 -
        2 * new->skewb.delta - 1;
    else if (triangle_unit[i][0].x == 2)
      offset_list[i].x = new->skewb2d.diamond_length / 2 +
        2 * new->skewb.delta;
    else if (triangle_unit[i][0].x == 3)
      offset_list[i].x = new->skewb2d.diamond_length - new->skewb.delta - 1;
    if (triangle_unit[i][0].y == 0)
      offset_list[i].y = new->skewb.delta - 1;
    else if (triangle_unit[i][0].y == 1)
      offset_list[i].y = new->skewb2d.diamond_length / 2 -
        2 * new->skewb.delta - 1;
    else if (triangle_unit[i][0].y == 2)
      offset_list[i].y = new->skewb2d.diamond_length / 2 +
        2 * new->skewb.delta - 1;
    else if (triangle_unit[i][0].y == 3)
      offset_list[i].y = new->skewb2d.diamond_length - new->skewb.delta - 2;
   }
  if (diamond_unit[0].x == 0)
    offset_list[MAXORIENT].x = new->skewb.delta - 2;
  else if (diamond_unit[0].x == 1)
    offset_list[MAXORIENT].x = new->skewb2d.diamond_length / 2 - 1;
  if (diamond_unit[0].y == 0)
    offset_list[MAXORIENT].y = new->skewb.delta - 2;
  else if (diamond_unit[0].y == 1)
    offset_list[MAXORIENT].y = new->skewb2d.diamond_length / 2 - 2;
  if (letter_unit[MAXORIENT].x == 1)
    letter_list[MAXORIENT].x = new->skewb2d.diamond_length / 2 - 2 +
      new->skewb.letter_offset.x;
  if (letter_unit[MAXORIENT].y == 1)
    letter_list[MAXORIENT].y = new->skewb2d.diamond_length / 2 - 2 +
      new->skewb.letter_offset.y;
}

static int position_polyhedrons(w, x, y, face, corner, direction)
Skewb2DWidget w;
int x, y;
int *face, *corner, *direction;
{
  int face_x, face_y;
  int i, j;

  x -= w->skewb.puzzle_offset.x;
  y -= w->skewb.puzzle_offset.y;
  face_x = x / w->skewb2d.view_length;
  face_y = y / w->skewb2d.view_length;
  i = x - face_x * w->skewb2d.view_length;
  j = y - face_y * w->skewb2d.view_length;
  if (i - j > w->skewb2d.view_length / 2 - 3)
    *corner = TR;
  else if (i + j > 3 * w->skewb2d.view_length / 2)
    *corner = BR;
  else if (j - i > w->skewb2d.view_length / 2 - 2)
    *corner = BL;
  else if (i + j < w->skewb2d.view_length / 2 + 7)
    *corner = TL;
  else /* *corner == MAXORIENT : this is ambiguous */
     return FALSE;
  if (*direction == CW)
    *direction = (*corner + 1) % MAXORIENT;
  else if (*direction == CCW)
    *direction = (*corner + 3) % MAXORIENT;
  if ((face_x != 1 && face_y != 1) ||
      (face_x >= 3 && w->skewb.vertical) ||
      (face_y >= 3 && !w->skewb.vertical) ||
      ((*corner + *direction) % 2 == 0))
    return FALSE;
  *face = plane_to_cube[face_x + face_y * MAXX];
  if (face_x == 3) {
    *face = MAXFACES - 1;
    *corner = (*corner + HALF) % MAXORIENT;
    if (*direction < MAXORIENT)
      *direction = (*direction + HALF) % MAXORIENT;
  }
  return TRUE;
}

static void no_move_polyhedrons(w)
Skewb2DWidget w;
{
  skewbCallbackStruct cb;

  cb.reason = SKEWB_IGNORE;
  XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

static void practice_polyhedrons(w)
Skewb2DWidget w;
{
  skewbCallbackStruct cb;

  cb.reason = SKEWB_PRACTICE;
  XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

static void randomize_polyhedrons(w)
Skewb2DWidget w;
{
  skewbCallbackStruct cb;
  int face, corner, direction;
  int big = MAXCUBES * 3 + NRAND(2);

  if (w->skewb.practice)
    practice_polyhedrons(w);
  w->skewb.randomized = TRUE;
  cb.reason = SKEWB_RESET;
  XtCallCallbacks((Widget) w, XtNselectCallback, &cb);

#ifdef DEBUG
  big = 3;
#endif

  while (big--) {
    face = NRAND(MAXFACES);
    corner = NRAND(MAXORIENT);
    direction = ((NRAND(2)) ? corner + 1 : corner + 3) % MAXORIENT;
    move_polyhedrons(w, face, corner, direction);

    cb.reason = SKEWB_MOVED;
    cb.face = face;
    cb.corner = corner;
    cb.direction = direction;
    XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
  }
  w->skewb.randomized = TRUE;
  cb.reason = SKEWB_RANDOMIZE;
  XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
  w->skewb.randomized = !check_solved((SkewbWidget) w);
  if (!w->skewb.randomized) {
    cb.reason = SKEWB_SOLVED;
    XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
  }
}

static void move_polyhedrons(w, face, corner, direction)
Skewb2DWidget w;
int face, corner, direction;
{
  int new_face, new_direction, new_corner, k, size, rotate;

  /* corner as MAXORIENT is ambiguous */
  for (size = MINOR; size <= MAJOR; size++) {
    read_diagonal((SkewbWidget) w, face, corner, 0, size);
    for (k = 1; k <= MAXROTATE; k++) {
      new_face = slide_next_row[face][corner][direction / 2].face;
      rotate = slide_next_row[face][corner][direction / 2].rotation %
        MAXORIENT;
      new_direction = (rotate + direction) %  MAXORIENT;
      new_corner = (rotate + corner) %  MAXORIENT;
      if (k != MAXROTATE)
        read_diagonal((SkewbWidget) w, new_face, new_corner, k, size);
      rotate_diagonal((SkewbWidget) w, rotate, k - 1, size);
      write_diagonal(w, new_face, new_corner, k - 1, size);
      face = new_face;
      corner = new_corner;
      direction = new_direction;
    }
    if (size == MINOR) {
      new_face = min_to_maj[face][corner].face;
      rotate = min_to_maj[face][corner].rotation % MAXORIENT;
      direction = (rotate + direction) % MAXORIENT;
      corner = (corner + rotate + 2) % MAXORIENT;
      face = new_face;
    }
  }
}

static void write_diagonal(w, face, corner, orient, size)
Skewb2DWidget w;
int face, corner, orient, size;
{
  int g, h;

  if (size == MINOR) {
    w->skewb.cube_loc[face][corner] = w->skewb.minor_loc[orient];
    draw_triangle(w, face, corner);
  } else /* size == MAJOR */ {
    w->skewb.cube_loc[face][MAXORIENT] =
      w->skewb.major_loc[orient][MAXORIENT - 1];
    draw_diamond(w, face);
    for (g = 1; g < MAXORIENT; g++) {
      h = (corner + g) % MAXORIENT;
      w->skewb.cube_loc[face][h] = w->skewb.major_loc[orient][g - 1];
      draw_triangle(w, face, h);
    }
  }
}

static void draw_frame(w, gc)
Skewb2DWidget w;
GC gc;
{
  int i;
  XPoint pos[MAX(MAXX, MAXY) + 1], letters;
  
  for (i = 0; i <= MAX(MAXX, MAXY); i++) {
    pos[i].x = i * w->skewb2d.view_length + w->skewb.puzzle_offset.x;
    pos[i].y = i * w->skewb2d.view_length + w->skewb.puzzle_offset.y;
  }
  XDrawLine(XtDisplay(w), XtWindow(w), gc,
    pos[1].x, pos[0].y, pos[2].x, pos[0].y);
  XDrawLine(XtDisplay(w), XtWindow(w), gc,
    pos[3].x, pos[1].y, pos[3].x, pos[2].y);
  XDrawLine(XtDisplay(w), XtWindow(w), gc,
    pos[1].x, pos[3].y, pos[2].x, pos[3].y);
  XDrawLine(XtDisplay(w), XtWindow(w), gc,
    pos[0].x, pos[1].y, pos[0].x, pos[2].y);
  letters.x = pos[0].x + w->skewb2d.view_length / 2 - w->skewb.delta;
  letters.y = pos[0].y + w->skewb2d.view_length / 2;
  XDrawString(XtDisplay(w), XtWindow(w), gc,
    letters.x + 5 * w->skewb.letter_offset.x,
    letters.y + w->skewb.letter_offset.y, "Front", 5);
  letters.x = pos[2].x + w->skewb2d.view_length / 2 - w->skewb.delta;
  letters.y = pos[2].y + w->skewb2d.view_length / 2;
  XDrawString(XtDisplay(w), XtWindow(w), gc,
    letters.x + 4 * w->skewb.letter_offset.x,
    letters.y + w->skewb.letter_offset.y, "Back", 4);
  if (w->skewb.vertical) {
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[1].x, pos[0].y, pos[1].x, pos[4].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[2].x, pos[0].y, pos[2].x, pos[4].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[0].x, pos[1].y, pos[3].x, pos[1].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[0].x, pos[2].y, pos[3].x, pos[2].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[1].x, pos[4].y, pos[2].x, pos[4].y);
  } else {
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[0].x, pos[1].y, pos[4].x, pos[1].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[0].x, pos[2].y, pos[4].x, pos[2].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[1].x, pos[0].y, pos[1].x, pos[3].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[2].x, pos[0].y, pos[2].x, pos[3].y);
    XDrawLine(XtDisplay(w), XtWindow(w), gc,
      pos[4].x, pos[1].y, pos[4].x, pos[2].y);
  }
}   

static void draw_all_polyhedrons(w)
Skewb2DWidget w;
{
  int face, corner;

  for (face = 0; face < MAXFACES; face++) {
    draw_diamond(w, face);
    for (corner = 0; corner < MAXORIENT; corner++)
      draw_triangle(w, face, corner);
  }
}

static void draw_diamond(w, face)
Skewb2DWidget w;
{
  int dx, dy, orient;

  orient = w->skewb.cube_loc[face][MAXORIENT].rotation;
  if (w->skewb.vertical || face != MAXFACES - 1) {
    dx = (cube_to_plane[face] % MAXX) * w->skewb2d.view_length +
      w->skewb.delta;
    dy = (cube_to_plane[face] / MAXX) * w->skewb2d.view_length +
      w->skewb.delta;
  } else {
    dx = (cube_to_plane[face] / MAXX) * w->skewb2d.view_length +
      w->skewb.delta;
    dy = (cube_to_plane[face] % MAXX) * w->skewb2d.view_length +
      w->skewb.delta;
    orient = (orient + HALF) % STRT;
  }
  dx += w->skewb.puzzle_offset.x + w->skewb.delta;
  dy += w->skewb.puzzle_offset.y + w->skewb.delta;
  diamond_list[0].x = offset_list[MAXORIENT].x + dx;
  diamond_list[0].y = offset_list[MAXORIENT].y + dy;
  XFillPolygon(XtDisplay(w), XtWindow(w),
     w->skewb.face_GC[w->skewb.cube_loc[face][MAXORIENT].face],
     diamond_list, 4, Convex, CoordModePrevious);
  if (w->skewb.depth == 1 || w->skewb.mono) {
    int letter_x, letter_y;
    char buf[2];

    sprintf(buf, "%c",
      w->skewb.face_name[w->skewb.cube_loc[face][MAXORIENT].face][0]);
    letter_x = dx + letter_list[MAXORIENT].x;
    letter_y = dy + letter_list[MAXORIENT].y;
    XDrawString(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
      letter_x, letter_y, buf, 1);
  }
  if (w->skewb.orient)
    draw_orient_line(w, MAXORIENT, orient, dx, dy);
}

static void draw_triangle(w, face, corner)
Skewb2DWidget w;
int face, corner;
{
  int dx, dy, letter_x, letter_y, orient, new_corner;

  orient = w->skewb.cube_loc[face][corner].rotation;
  if (w->skewb.vertical || face != MAXFACES - 1) {
    dx = (cube_to_plane[face] % MAXX) * w->skewb2d.view_length +
      w->skewb.delta - 1;
    dy = (cube_to_plane[face] / MAXX) * w->skewb2d.view_length +
      w->skewb.delta - 1;
    new_corner = corner;
  } else {
    dx = (cube_to_plane[face] / MAXX) * w->skewb2d.view_length +
      w->skewb.delta - 1;
    dy = (cube_to_plane[face] % MAXX) * w->skewb2d.view_length +
      w->skewb.delta - 1;
    new_corner = (corner + HALF) % STRT;
    orient = (orient + HALF) % STRT;
  }
  dx += w->skewb.puzzle_offset.x + w->skewb.delta;
  dy += w->skewb.puzzle_offset.y + w->skewb.delta;
  letter_x = dx + letter_list[new_corner].x;
  letter_y = dy + letter_list[new_corner].y;
  triangle_list[new_corner][0].x = offset_list[new_corner].x + dx;
  triangle_list[new_corner][0].y = offset_list[new_corner].y + dy;
  XFillPolygon(XtDisplay(w), XtWindow(w),
    w->skewb.face_GC[w->skewb.cube_loc[face][corner].face],
    triangle_list[new_corner], 3, Convex, CoordModePrevious);
  if (w->skewb.depth == 1 || w->skewb.mono) {
    char buf[2];

    sprintf(buf, "%c",
      w->skewb.face_name[w->skewb.cube_loc[face][corner].face][0]);
    XDrawString(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
      letter_x, letter_y, buf, 1);
  }
  if (w->skewb.orient)
    draw_orient_line(w, new_corner, orient,
      letter_x - w->skewb.letter_offset.x, letter_y - w->skewb.letter_offset.y);
}

static void draw_orient_line(w, cube, orient, dx, dy)
Skewb2DWidget w;
int cube, orient, dx, dy;
{
  if (cube == MAXORIENT)
    switch (orient) {
      case TR:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx + w->skewb2d.diamond_length / 2 - 2,
                  dy + w->skewb.delta - 2,
                  dx + w->skewb2d.diamond_length / 2 - 2,
                  dy + w->skewb.delta + w->skewb.orient_line_length);
        return;
      case BR:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx + w->skewb2d.diamond_length - 2,
                  dy + w->skewb2d.diamond_length / 2 - 2,
                  dx + w->skewb2d.diamond_length -
                    w->skewb.orient_line_length - 7,
                  dy + w->skewb2d.diamond_length / 2 - 2);
        return;
      case BL:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx + w->skewb2d.diamond_length / 2 - 2,
                  dy + w->skewb2d.diamond_length - 2,
                  dx + w->skewb2d.diamond_length / 2 - 2,
                  dy + w->skewb2d.diamond_length -
                    w->skewb.orient_line_length - 7);
        return;
      case TL:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx + w->skewb.delta - 2,
                  dy + w->skewb2d.diamond_length / 2 - 2,
                  dx + w->skewb.delta + w->skewb.orient_line_length,
                  dy + w->skewb2d.diamond_length / 2 - 2);
        return;
      default:
        printf("OOPS\n");
    }
  else /* cube != MAXORIENT */
    switch (orient) {
      case TR:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx, 
                  dy - w->skewb.delta,
                  dx,
                  dy - w->skewb.delta - w->skewb.orient_line_length / 2);
        return;
      case BR:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx + w->skewb.delta,
                  dy,
                  dx + w->skewb.delta + w->skewb.orient_line_length / 2,
                  dy);
        return;
      case BL:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx,
                  dy + w->skewb.delta,
                  dx,
                  dy + w->skewb.delta + w->skewb.orient_line_length / 2);
        return;
      case TL:
        XDrawLine(XtDisplay(w), XtWindow(w), w->skewb.inverse_GC,
                  dx - w->skewb.delta,
                  dy,
                  dx - w->skewb.delta - w->skewb.orient_line_length / 2,
                  dy);
        return;
      default:
        printf("OOPS\n");
    }
}
