/* image.c : image handling and display routines
**
** Written and Copyright (C) 1994 by Michael J. Gourlay
**
** NO WARRANTEES, EXPRESS OR IMPLIED.
*/

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "diw_map.h"
#include "tga.h"

#include "image.h"

#define MAX(x,y) ((x)>(y) ? (x) : (y))

/* --------------------------------------------------------------- */

/* Global "original" images */
rgba_image_t orig_image[NUM_ORIG_IMAGES];

/* --------------------------------------------------------------- */

/* free_image: free memory of the img
*/
void
free_image(rgba_image_t *imgP)
{
  if(imgP->ri != NULL) my_free(imgP->ri);
  if(imgP->gi != NULL) my_free(imgP->gi);
  if(imgP->bi != NULL) my_free(imgP->bi);
  if(imgP->ai != NULL) my_free(imgP->ai);
}

/* --------------------------------------------------------------- */

/* alloc_image: allocate memory for the img
** return -1 if failed, 0 otherwize
*/
int
alloc_image(rgba_image_t *imgP, char *proc)
{
  if((imgP->ri=(UCHAR*)my_calloc(imgP->ncols, imgP->nrows,"alloc_image"))==NULL)
  {
    fprintf(stderr, "%s: Bad Alloc\n", proc);
    return(-1);
  }
  if((imgP->gi=(UCHAR*)my_calloc(imgP->ncols, imgP->nrows,"alloc_image"))==NULL)
  {
    fprintf(stderr, "%s: Bad Alloc\n", proc);
    return(-1);
  }
  if((imgP->bi=(UCHAR*)my_calloc(imgP->ncols, imgP->nrows,"alloc_image"))==NULL)
  {
    fprintf(stderr, "%s: Bad Alloc\n", proc);
    return(-1);
  }
  if((imgP->ai=(UCHAR*)my_calloc(imgP->ncols, imgP->nrows,"alloc_image"))==NULL)
  {
    fprintf(stderr, "%s: Bad Alloc\n", proc);
    return(-1);
  }

  return(0);
}

/* --------------------------------------------------------------- */

/* reset_image: copy "original" image into image spaces of diw_maps
** type determines whether images are the "src" (1) or "dst" (2)
*/
void
reset_images(int type)
{
  int indx;
  rgba_image_t *imgP, *dmiP;

  /* Copy the "original" image into the diw_maps */
  for(indx=0; indx<NUM_DIW_MAPS; indx++) {
    if(global_diw_map[indx].width != 0) {
      switch(type) {
        case 1:
          imgP = &orig_image[0];
          dmiP = &global_diw_map[indx].src_img;
          break;

        case 2:
          imgP = &orig_image[1];
          dmiP = &global_diw_map[indx].dst_img;
          break;

        default:
          fprintf(stderr, "reset_images: Bad Value: type %i\n", type);
          return;
          break;
      }

      free_image(dmiP);

      /* Copy the image info from the original image */
      *dmiP = *imgP;

      /* Allocate memory for the image copies */
      if(alloc_image(dmiP, "reset_images")) return;

      /* Copy the original image into the diw_map image space */
      memcpy(dmiP->ri, imgP->ri, dmiP->ncols * dmiP->nrows);
      memcpy(dmiP->gi, imgP->gi, dmiP->ncols * dmiP->nrows);
      memcpy(dmiP->bi, imgP->bi, dmiP->ncols * dmiP->nrows);
      memcpy(dmiP->ai, imgP->ai, dmiP->ncols * dmiP->nrows);
    }
  }
}

/* --------------------------------------------------------------- */

/* load_img: load image into memory.  Allocate new image space.
*/
int
load_img(char *fn, rgba_image_t *imgP)
{
  int           tga_return;
  tga_hdr_t     tga_hdr;
  FILE         *infP=NULL;

  /* Open the input file for binary reading */
  if(fn!=NULL && (infP=fopen(fn, "rb"))==NULL) {
    fprintf(stderr, "load_img: could not open '%s' for input\n", fn);
    return(-1);
  }

  /* Load the image header */
    /* Targa */
    if(tga_return = load_tga_header(&tga_hdr, imgP, infP)) {
      fprintf(stderr, "load_tga_header returned %i\n", tga_return);
      return(tga_return);
    }

  /* Free the memory for the previous image planes */
  free_image(imgP);

  /* Allocate new memory for the new image planes */
  if(alloc_image(imgP, "load_img")) return(-1);

  /* Load the new image */
    /* Targa */
    load_tga_image(&tga_hdr, imgP, infP);

  /* Close the input file */
  fclose(infP);

  return(0);
}

/* --------------------------------------------------------------- */

/* save_img: dissolve 2 images and save dissolved image to file
** dimensions of the output image are the same as the srcP.
*/
int
save_img(char *fn, rgba_image_t *siP, rgba_image_t *diP, double t)
{
  register int   nx, ny;
  register int   xi, yi;
  register int   rsi, gsi, bsi, asi;
  register int   rdi, gdi, bdi, adi;
  int            tga_return;
  rgba_image_t   img;
  tga_hdr_t      tga_hdr;
  FILE          *outfP=NULL;

  /* Initialize the info for the saved image */
  ny = img.nrows = siP->nrows;
  nx = img.ncols = siP->ncols;

  img.compressed = img.color_mapped = 0;
  img.pixel_size = 32;

  if(siP->compressed || diP->compressed) img.compressed = 1;
  img.pixel_size = MAX(siP->pixel_size, diP->pixel_size);
  if(siP->color_mapped && diP->color_mapped) img.color_mapped = 1;

  /* Allocate space for image data */
  if(alloc_image(&img, "save_img")) return(-1);

  /* Dissolve the two images according to the dissolve parameter */
  for(yi=0; yi<ny; yi++) {
    for(xi=0; xi<nx; xi++) {
      rsi = (1.0-t) * siP->ri[yi * nx + xi];
      gsi = (1.0-t) * siP->gi[yi * nx + xi];
      bsi = (1.0-t) * siP->bi[yi * nx + xi];
      asi = (1.0-t) * siP->ai[yi * nx + xi];
      if((diP!=NULL) && (xi<diP->ncols) && (yi < diP->nrows)) {
        rdi = t * diP->ri[yi * diP->ncols + xi];
        gdi = t * diP->gi[yi * diP->ncols + xi];
        bdi = t * diP->bi[yi * diP->ncols + xi];
        adi = t * diP->ai[yi * diP->ncols + xi];
      } else {
        rdi = 0;
        gdi = 0;
        bdi = 0;
        adi = 0;
      }
      img.ri[yi*nx+xi] = (int)(rsi + rdi + 0.5);
      img.gi[yi*nx+xi] = (int)(gsi + gdi + 0.5);
      img.bi[yi*nx+xi] = (int)(bsi + bdi + 0.5);
      img.ai[yi*nx+xi] = (int)(asi + adi + 0.5);
    }
  }

  /* Open the output image file for binary writing */
  if(fn!=NULL && (outfP=fopen(fn, "wb"))==NULL) {
    fprintf(stderr, "save_img: could not open '%s' for output\n", fn);
    return(-1);
  }

  /* Set the image header */
    /* Targa */
    tga_hdr.id_len = 0;

    /* cmap_type depends on the img_type */
    tga_hdr.cmap_type = 0;

    /* img_type comes from the user */
    tga_hdr.img_type = TGA_RGB;

    if(img.compressed) tga_hdr.img_type += TGA_RLE;

    tga_hdr.cmap_index = 0;

    /* cmap_len depends on the img_type and pixel_size */
    tga_hdr.cmap_len = 0;

    /* cmap_size depends on the img_type and pixel_size */
    tga_hdr.cmap_size = 0;

    tga_hdr.x_off = 0;
    tga_hdr.y_off = 0;

    /* pixel_size depends on the img_type */
    tga_hdr.pixel_size = img.pixel_size;

    tga_hdr.att_bits = 0;
    tga_hdr.reserved = 0;
    tga_hdr.origin_bit = 0;
    tga_hdr.interleave = TGA_IL_None;

  /* Save the image header */
    /* Targa */
    if(tga_return = save_tga_header(&tga_hdr, &img, outfP)) {
      fprintf(stderr, "save_tga_header returned %i\n", tga_return);
      return(tga_return);
    }

  /* Save the dissolved image */
    /* Targa */
    save_tga_image(&tga_hdr, &img, outfP);

  /* Free the dissolved image */
  free_image(&img);

  /* Close the output image file */
  fclose(outfP);

  return(0);
}

/* --------------------------------------------------------------- */

/* dither_image: dissolve 2 images and store into ximage
** 't' determines the percent of dissolve between images.
** brightness factor 'brite' is applied to dissolved image.
*/
void
dither_image(Visual *visual, rgba_image_t *srcP, rgba_image_t *dstP, double t, double brite, XImage *ximage)
{
  int           xi, yi;
  unsigned char rb, gb, bb;
  int           ri, gi, bi;
  int           rsi, gsi, bsi;
  int           rdi, gdi, bdi;
  int           re, ge, be;
  Pixel         pixel; /* changed from int to Pixel: WA (MJG 13sep95) */
  /* 24bit variables added WA (MJG 13sep95) */
  int           rShift, gShift, bShift, rRange, gRange, bRange;

  re = ge = be = 0;

  /* 24bit color code added WA (MJG 13sep95) (unverified by MJG) */
  /* Based on xpaint sources: palette.c */
  if (visual->class == TrueColor) {
    int v;
    rShift = gShift = bShift = 0;
    for (v = visual->red_mask; (v & 1) == 0; v >>= 1) rShift++;
    for (rRange=0; v; v>>=1) rRange++;
    for (v = visual->green_mask; (v & 1) == 0; v >>= 1) gShift++;
    for (gRange=0; v; v>>=1) gRange++;
    for (v = visual->blue_mask; (v & 1) == 0; v >>= 1) bShift++;
    for (bRange=0; v; v>>=1) bRange++;
  }

  for(yi=0; yi<srcP->nrows; yi++) {
    for(xi=0; xi<srcP->ncols; xi++) {
      rsi = (1.0-t) * srcP->ri[yi * srcP->ncols + xi];
      gsi = (1.0-t) * srcP->gi[yi * srcP->ncols + xi];
      bsi = (1.0-t) * srcP->bi[yi * srcP->ncols + xi];
      if((dstP!=NULL) && (xi<dstP->ncols) && (yi < dstP->nrows)) {
        rdi = t * dstP->ri[yi * dstP->ncols + xi];
        gdi = t * dstP->gi[yi * dstP->ncols + xi];
        bdi = t * dstP->bi[yi * dstP->ncols + xi];
      } else {
        rdi = 0;
        gdi = 0;
        bdi = 0;
      }

      ri = (int)(brite * (rsi + rdi) + 0.5);
      gi = (int)(brite * (gsi + gdi) + 0.5);
      bi = (int)(brite * (bsi + bdi) + 0.5);

      /* TrueColor code due to WA (MJG 13sep95) */
      if (visual->class == TrueColor) {
        pixel = (((ri<<rRange)>>8) << rShift)
               |(((gi<<gRange)>>8) << gShift)
               |(((bi<<bRange)>>8) << bShift);
        /* Store TrueColor pixel in image */
        XPutPixel(ximage, xi, yi, pixel);
      } else {
        ri += re; if(ri>255) ri=255; else if(ri<0) ri=0;
        gi += ge; if(gi>255) gi=255; else if(gi<0) gi=0;
        bi += be; if(bi>255) bi=255; else if(bi<0) bi=0;
        /* Approximate true color */
        rb = ri & (3<<6);
        gb = gi & (3<<6);
        bb = bi & (3<<6);
        pixel = (rb>>2)|(gb>>4)|(bb>>6);
        /* Remember truncation error */
        re = ri - (diw_xcolors[pixel].red   >> 8);
        ge = gi - (diw_xcolors[pixel].green >> 8);
        be = bi - (diw_xcolors[pixel].blue  >> 8);

        /* Store pixel index in colormapped image */
        /* MJG 19jul94: fixed indexing */
        ximage->data[yi * ximage->width + xi] = diw_xcolors[pixel].pixel;
      }
    } /* for xi */
  } /* for yi */
}

/* --------------------------------------------------------------- */

/* make_test_image: generate a test image
** Uses the incoming values of ncols and nrows to determine image size.
** If ncols or nrows are zero, default values are used instead.
** type determines which test image to use.
** Memory for the images is allocated and imgP is set.
*/
int
make_test_image(rgba_image_t *imgP, int type)
{
  register int   xi, yi;
  register UCHAR p;

  if(imgP->ncols <= 0) imgP->ncols = 320;
  if(imgP->nrows <= 0) imgP->nrows = 240;
  imgP->compressed = 1;
  imgP->color_mapped = 0;
  imgP->pixel_size = 24;
  imgP->type = TARGA_MAGIC;

  if(alloc_image(imgP, "make_test_image")) return(1);

  for(yi=0; yi<imgP->nrows; yi++) {
    for(xi=0; xi<imgP->ncols; xi++) {
      p = 15 + 240*((float)xi/imgP->ncols)*((float)yi/imgP->nrows);
      if((xi%40>20 && yi%40<20) || (xi%40<20 && yi%40>20))
        p=0;
      if(type & 1) {
        imgP->ri[yi*(imgP->ncols) + xi] = p;
      } else {
        imgP->ri[yi*(imgP->ncols) + xi] = 255-p;
      }
      if(type & 2) {
        imgP->gi[yi*(imgP->ncols) + xi] = p;
      } else {
        imgP->gi[yi*(imgP->ncols) + xi] = 255-p;
      }
      if(type & 4) {
        imgP->bi[yi*(imgP->ncols) + xi] = p;
      } else {
        imgP->bi[yi*(imgP->ncols) + xi] = 255-p;
      }
      imgP->ai[yi*(imgP->ncols) + xi] = OPAQUE;
    }
  }
  return(0);
}
