[ident ('V3.1-2; Copyright  1993 Anglia Polytechnic University'),
inherit ('sys$library:starlet')]
program tscon_main;

const
	max_text_size = 1500;
	eth_addr_size = 6;
	serv_pw_size = 8;
	default_disc_char = %x10; { Ctrl/p }

type
	%include 'appl_vmsdef:datatypes'

	service_pw_type = packed array [1..serv_pw_size] of char;
	ethernet_address_type = packed array [1..eth_addr_size] of _ubyte;

	text_type = varying [max_text_size] of char;
	fixed_text_type = packed array [1..max_text_size] of char;
	string_desc = record
			    length : integer;
			    addr : ^fixed_text_type;
		      end;
	sigargs_type = array [0..15] of unsigned;
	mechargs_type = array [0..4] of unsigned;

var
	diagnostics, batch_mode, auto_login, auto_login_priv,
	force_mode, character_mode : [volatile] boolean := false;
	server_found, connected, currently_connected : boolean := false;
	input_spec, command_line, log_file, node_name, ethernet_device : text_type;
	service_pw : service_pw_type;
	disconnect_char : char;
	local_address, remote_address : ethernet_address_type;
	access_user, access_pw, privileged_pw : text_type;

	tscon_tables : [external, unsafe] unsigned;

	cli$_absent,
	tscon_normal,
	tscon_servconfail,
	tscon_servconlost,
	tscon_servfind,
	tscon_noethernet : [external, value] unsigned;

        sts, exit_status : [volatile] unsigned;

%include 'appl_vmsdef:rtldef'
%include 'appl_vmsdef:clidef'

[unbound, external (lib$get_input)] procedure prompt_routine; external;
	    { Used for cli$dcl_parse }
function tscon_display_init (input_file : varying [isize] of char;
			     poll_interval : integer;
			     idle_timeout_interval : integer;
			     command : varying [csize] of char;
			     force : boolean;
			     chars : boolean;
			     disc_char : char;
			     logging : boolean
			    ): boolean; external;
procedure tscon_mop_diagnostics; external;
[asynchronous] procedure tscon_put_blank; external;
[asynchronous] procedure tscon_purge_typeahead; external;

{******************************************************************************}
procedure do_heading;
{ Produce heading announcing successful connection }
  var
	disc_char : char;
	tscon_connected, tscon_disconnect: [value, external] unsigned;
  begin
	if batch_mode then
	    lib$signal (tscon_connected, 2,
			node_name.length,
			iaddress (node_name.body))
	else
	  begin
	    disc_char := chr (ord('@') + ord(disconnect_char));
	    lib$signal (tscon_connected, 2,
			node_name.length,
			iaddress (node_name.body),
			tscon_disconnect, 2,
			1, iaddress (disc_char));
	  end;
  end; { do_heading }

{******************************************************************************}
[global] function tscon_note_connected: boolean;
{ Record the start of a connection. }
{ Return boolean for whether the connection has just started }
  begin
	if not connected then
	  begin
		do_heading;
		currently_connected := true;
		connected := true;
		tscon_note_connected := true;
	  end
	else tscon_note_connected := false;
  end; { tscon_note_connected }

{******************************************************************************}
[asynchronous, unbound, global] function tscon_msg_handler
				    (var sigargs : sigargs_type;
				     var mechargs : mechargs_type
				    ): unsigned;
{ Condition handler which deals with messaging }
  var
	msg_shown : boolean;
	own_signal : sigargs_type;
	idx : integer;
	tscon_end, tscon_servpwfail, tscon_connected, tscon_nologin,
	tscon_disconnecting : [value, external] unsigned;

  [asynchronous, unbound] procedure tscon_put_message; external;

  begin
	msg_shown := false;

	if (sigargs[1] <> ss$_unwind)
	and (sigargs[1] <> ss$_debug)
	and (sigargs[1] <> tscon_disconnecting) then 
	  begin
		if not batch_mode then
		  begin
		    tscon_purge_typeahead;
		    tscon_put_blank;
		  end;

		exit_status := sigargs[1]; { Status upon exit }

		{ Strip off the PC & PSL }
		own_signal[0] := sigargs[0]-2;
		for idx := 1 to own_signal[0]::integer do
		    own_signal[idx] := sigargs[idx];

		$putmsg (msgvec := own_signal,
			 actrtn := tscon_put_message);
		msg_shown := true;
	  end;

	if (sigargs[1] = tscon_connected)
	or (sigargs[1] = tscon_nologin)
	or (sigargs[1] = tscon_end) then tscon_msg_handler := ss$_continue
	else
	  begin
	    if (sigargs[1] = tscon_servpwfail)
	    or (sigargs[1] = tscon_disconnecting) then
	      begin
		tscon_msg_handler := ss$_continue;
		mechargs[3] := tscon_normal; { Set return status (r0) }
		$unwind (); { To caller of declarer of this handler }
			    { which is tscon_initiate }
	      end
	    else tscon_msg_handler := ss$_resignal;
	  end;

	if msg_shown then sigargs[1] := uor (sigargs[1], sts$m_inhib_msg);
  end; { tscon_msg_handler }

{******************************************************************************}
procedure signal_error (sts1, sts2 : unsigned);
			
  begin
	lib$signal (sts1, 0, sts2, 0);
  end; { signal_error }

{******************************************************************************}
procedure console_dialogue;
{ Initiate dialogue with terminal server's remote console port }
  var
	flags, sts : unsigned;
	node : text_type;
	tscon_end : [external, value] unsigned;

  function tscon_initiate
		       (%descr dev_name : varying [dsize] of char;
		        %ref loc_addr : ethernet_address_type;
		        %ref rem_addr : ethernet_address_type;
			%descr password : service_pw_type;
		        flags : unsigned;
			%descr logspec : varying [lsize] of char;
			%descr acc_user : varying [usize] of char;
			%descr acc_pw : varying [asize] of char;
			%descr priv_pw : varying [psize] of char
		       ): unsigned; external;
  begin
	flags := 0;
	if batch_mode then flags := uor (flags, 1);
	if diagnostics then flags := uor (flags, 2);
	if auto_login then flags := uor (flags, 4);
	if auto_login_priv then flags := uor (flags, 8);

	connected := false;
	currently_connected := false;
	sts := tscon_initiate (ethernet_device,
			       local_address,
			       remote_address,
			       service_pw,
			       flags,
			       log_file,
			       access_user,
			       access_pw,
			       privileged_pw);
	currently_connected := false;

	if not odd (sts) then
	  begin
		if connected then signal_error (tscon_servconlost, sts)
			     else signal_error (tscon_servconfail, sts);
	  end;

	if (not batch_mode) and (connected) then
	  begin
		lib$sys_trnlog ('SYS$NODE',,node);
		if node = 'SYS$NODE' then node := 'local node';
		lib$signal (tscon_end, 2, node.length, iaddress (node.body));
	  end;

	if diagnostics then tscon_mop_diagnostics;
  end; { console_dialogue }

{******************************************************************************}
procedure get_remote_ethernet_info (test_name : varying [tsize] of char);
{ Get info about the remote (terminal server) Ethernet node }
  type
	serv_pw_desc_type = record
				asize : unsigned;
				aaddr : [unsafe] ^service_pw_type;
			    end;
  var
	serv_pw_desc : serv_pw_desc_type;
	sts : unsigned;

  function tscon_lookup_server (name : varying [nsize] of char;
				%ref ether_addr : ethernet_address_type;
				%ref service_pw : serv_pw_desc_type;
				var acc_user : varying [usize] of char;
				var acc_pw : varying [asize] of char;
				var priv_pw : varying [psize] of char
			       ): unsigned; external;
  begin
	with serv_pw_desc do
	  begin
		asize := serv_pw_size;
		aaddr := iaddress (service_pw);
	  end;

	sts := tscon_lookup_server (test_name,
				    remote_address,
				    serv_pw_desc,
				    access_user,
				    access_pw,
				    privileged_pw);
	if not odd (sts) then
	  begin
		if sts <> tscon_servfind then
		     signal_error (tscon_servfind, sts)
		else lib$signal (sts);
	  end
	else
	  begin
		server_found := true;
		node_name := test_name;
	  end;
  end; { get_remote_ethernet_info }

{******************************************************************************}
procedure get_local_ethernet_info;
{ Get info about the local Ethernet node from the DECnet node & circuit databases }
  var
	sts : unsigned;
	local_hardware_address : ethernet_address_type;

  function ccat_net_ethernet (var hardware_addr : ethernet_address_type;
			      var physical_addr : ethernet_address_type;
			      var device_name : varying [dsize] of char
			     ): unsigned; external;

  begin
	sts := ccat_net_ethernet (local_hardware_address,
				  local_address,
				  ethernet_device);
	if not odd (sts) then signal_error (tscon_noethernet, sts);
  end; { get_local_ethernet_info }

{******************************************************************************}
function strip_quotes (instr : varying [isize] of char
		      ): text_type;
{ Strip quotation marks }
  var
	result : text_type;
  begin
	result := instr;
	if result.length > 1 then
	  if result.body[1] = '"' then
	    result := substr (result, 2, result.length-1);
	if result.length > 0 then
	  if result.body[result.length] = '"' then
	    result.length := result.length - 1;

	strip_quotes := result;
  end; { strip_quotes }

{******************************************************************************}
procedure database_files;
{ Handle tscon database files (explicit or implicit) }
  var
	sts : unsigned;
	name : text_type;

  function tscon_process_db_file (name : varying [nsize] of char
			         ): unsigned; external;
  function tscon_default_db_file: unsigned; external;

  begin
	sts := cli$present ('DATABASE');
	if sts = cli$_absent then
	  begin	{ Call for default db file }
		tscon_default_db_file;
	  end
	else if odd (sts) then
	  begin
		sts := cli$get_value ('DATABASE', name);
		while odd (sts) do
		  begin
			tscon_process_db_file (name);
			sts := cli$get_value ('DATABASE', name);
		  end;
	  end;
  end; { database_files }

{******************************************************************************}
[global] function tscon_parse: unsigned; { Main body }
{ Parse the command line }
  var
	command, value : text_type;
	disc_char, idle_timeout_s, poll_cs : integer;
	tscon_invdiscch : [value, external] unsigned;

  begin
	exit_status := tscon_normal;
	get_local_ethernet_info;

	cli$get_value ('NODE', node_name);
	input_spec.length := 0;
	if odd (cli$present ('INPUT')) then
	  begin
		cli$get_value ('INPUT', input_spec);
		input_spec := strip_quotes (input_spec);
		    { There may be quotation marks to specify a null }
		    { parameter before a non-null one }
	  end;

	diagnostics := odd (cli$present ('DIAGNOSTICS'));

	if odd (cli$present ('OUTPUT')) then
	    cli$get_value ('OUTPUT', log_file);

	poll_cs := 0;
	if odd (cli$present ('POLL_INTERVAL')) then
	  begin
	    cli$get_value ('POLL_INTERVAL', value);
	    readv (value, poll_cs);
	  end;

	idle_timeout_s := 0;
	if odd (cli$present ('IDLE_TIMEOUT')) then
	  begin
	    cli$get_value ('IDLE_TIMEOUT', value);
	    readv (value, idle_timeout_s);
	  end;

	if odd (cli$present ('COMMAND')) then
	  begin
		cli$get_value ('COMMAND', command);
		command := strip_quotes (command);
	  end
	else command.length := 0;

	auto_login := odd (cli$present ('LOGIN.USER'));
	auto_login_priv := odd (cli$present ('LOGIN.PRIVILEGED'));

	disconnect_char := chr (default_disc_char);
	if odd (cli$present ('DISCONNECT')) then
	  begin
		cli$get_value ('DISCONNECT', value);
		readv (value, disc_char);
		if (disc_char = 0)  { NUL }
		or (disc_char = 13) { CR  }
		or (disc_char > 31) then
		     lib$signal (tscon_invdiscch)
		else disconnect_char := chr (disc_char);
	  end;

	force_mode := false;

	if odd (cli$present ('MODE.CHARACTER')) then
	  begin
		force_mode := true;
		character_mode := true;
	  end;

	if odd (cli$present ('MODE.LINE')) then
	  begin
		force_mode := true;
		character_mode := false;
	  end;

	database_files;

	batch_mode := tscon_display_init (input_spec, poll_cs,
					  idle_timeout_s, command,
					  force_mode, character_mode,
					  disconnect_char,
					  (log_file.length > 0));

	get_remote_ethernet_info (node_name);

	if server_found then console_dialogue;

	tscon_parse := exit_status;
  end; { tscon_parse }

{******************************************************************************}
{ Main program }
begin
	lib$get_foreign (command_line);
	sts := cli$dcl_parse ('TSCON '+command_line, tscon_tables, prompt_routine);
	if odd (sts) then sts := cli$dispatch;
	$exit (uor (sts, sts$m_inhib_msg));
end.
